package no.statnett.ecp;

import no.statnett.ecp.bck.Backup;
import no.statnett.ecp.brs.BrokerShell;
import no.statnett.ecp.cc.ConnectivityCheck;
import no.statnett.ecp.cds.CDShell;
import no.statnett.ecp.div.BrannslukkerAFRRCAPBudTilNVE;
import no.statnett.ecp.div.ComponentListing;
import no.statnett.ecp.div.MessagePathListing;
import no.statnett.ecp.div.SC;
import no.statnett.ecp.edx.EDXMonitor;
import no.statnett.ecp.eft.EDXFailTrash;
import no.statnett.ecp.emd.ECPMsgDelay;
import no.statnett.ecp.eml.ECPMsgLogger;
import no.statnett.ecp.epa.ECPPerfAnalyzer;
import no.statnett.ecp.gm.GapMonitor;
import no.statnett.ecp.pf.Performance;
import no.statnett.ecp.qc.QueueCleaner;
import no.statnett.ecp.qm.QueueMonitor;
import no.statnett.ecp.qp.QueueProtector;
import no.statnett.ecp.utils.Div;
import no.statnett.ecp.utils.LogOut;

import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.LogManager;

public class EKit {

    public static final String VERSION = "v1.19.3"; //

    public static void main(String[] args) throws Exception {
        LogManager.getLogManager().getLogger("").setLevel(Level.OFF); // Mutes unwanted log-messages from the libraries/dependencies
        if (args.length < 1) {
            usage(false);
        } else {
            try {
                String tool = args[0];
                // remove first element of args
                args = Arrays.copyOfRange(args, 1, args.length);

                if (tool.equals("QC")) {
                    QueueCleaner.main(args);
                } else if (tool.equals("QP")) {
                    QueueProtector.main(args);
                } else if (tool.equals("QM")) {
                    QueueMonitor.main(args);
                } else if (tool.equals("BRS")) {
                    BrokerShell.main(args);
                } else if (tool.equals("CDS")) {
                    CDShell.main(args);
                } else if (tool.equals("EMD")) {
                    ECPMsgDelay.main(args);
                } else if (tool.equals("EPA")) {
                    ECPPerfAnalyzer.main(args);
                } else if (tool.equals("EDX")) {
                    EDXMonitor.main(args);
                } else if (tool.equals("EFT")) {
                  EDXFailTrash.main(args);
                } else if (tool.equals("EML")) {
                    ECPMsgLogger.main(args);
                } else if (tool.equals("GM")) {
                    GapMonitor.main(args);
                } else if (tool.equals("CC")) {
                    ConnectivityCheck.main(args);
                } else if (tool.equals("BCK")) {
                    Backup.main(args);
                } else if (tool.equals("CL")) {
                    ComponentListing.main(args);
                } else if (tool.equals("SC")) {
                    SC.main(args); // non-exposed tools, only useful for dealing with EDX Service Catalogue - old syntax - little interest
                } else if (tool.equals("ACNVE")) {
                    BrannslukkerAFRRCAPBudTilNVE.main(args); // non-exposed, very specific tool for Statnett (retrieves some files from EDX-DMS if they can be found, filters XML and send to NVE)
                } else if (tool.equals("MPL")) {
                    MessagePathListing.main(args); // non-exposed, very specific tool for TSOs (checks if BSPs have specified the appropriate message paths)
                } else if (tool.equals("PF")) {
                    Performance.main(args); // non-exposed, a performance-test-tool to test Artemis throughput
                } else if (tool.equals("HELP")) {
                    usage(true);
                } else {
                    usage(false);
                }
            } catch (Exception e) {
                System.err.println(LogOut.e() + " " + e.getClass() + ": " + e.getMessage() + ", Stacktrace: " + Div.onelinerStacktrace(e.getStackTrace(), "no.statnett.ecp"));
            }
        }
    }

    private static void usage(boolean specialTools) {
        System.out.println("\nekit " + VERSION + " is short for ECCoSP Kit, built by Statnett (Morten Simonsen). Tested up till 4.17");
        System.out.println("The kit offers the following. Main interface/API/source used for data is specified in brackets []\n");
        System.out.println("QC  (QueueCleaner)      " + String.format("%-7s", QueueCleaner.VERSION) + " [AMQP] - clean queues based on filters (4.12)");
        System.out.println("QP  (QueueProtector)    " + String.format("%-7s", QueueProtector.VERSION) + " [AMQP] - protects download-queue from bursts");
        System.out.println("QM  (QueueMonitor)      " + String.format("%-7s", QueueMonitor.VERSION) + " [HTTP] - monitor queues in Artemis-broker (4.14+)");
        System.out.println("BRS (BrokerShell)       " + String.format("%-7s", BrokerShell.VERSION) + " [HTTP] - manage/monitor Artemis-brokers (4.14+)");
        System.out.println("CDS (CDShell)           " + String.format("%-7s", CDShell.VERSION) + " [HTTP] - browse/manage ECP components (4.16+)");
        System.out.println("CC  (ConnectivityCheck) " + String.format("%-7s", ConnectivityCheck.VERSION) + " [HTTP] - check connectivity to ECP-endpoints (4.12+)");
        System.out.println("BCK (Backup)            " + String.format("%-7s", Backup.VERSION) + " [HTTP] - export/import database from/to ECP/EDX during runtime (4.15+)");
        System.out.println("EMD (ECPMsgDelay)       " + String.format("%-7s", ECPMsgDelay.VERSION) + " [LOGS] - calculate delays for messages transmitted over the ECP-network (4.12+)");
        System.out.println("EPA (ECPPerfAnalyzer)   " + String.format("%-7s", ECPPerfAnalyzer.VERSION) + " [LOGS] - calculate throughput of the ECP/EDX-endpoint (4.14+)");
        System.out.println("EDX (EDXMonitor)        " + String.format("%-7s", EDXMonitor.VERSION) + " [LOGS] - monitor the flow of messages between ECP and EDX (4.14+)");
        System.out.println("EML (ECPMsgLogger)      " + String.format("%-7s", ECPMsgLogger.VERSION) + " [LOGS] - messages to/from ECP pretty-printed to log (4.12+)");
        System.out.println("EFT (EDXFailTrash)      " + String.format("%-7s", EDXFailTrash.VERSION) + " [FILE] - monitor failed/non-routed msg (4.12+)");
        if (specialTools) {
            System.out.println("\nSpecial tools - only useful for Statnett or TSOs. Less documented: ");
            System.out.println("GM  (GapMonitor)        " + String.format("%-7s", GapMonitor.VERSION) + " [EML]  - monitor gaps in the ECP-message traffic - very specific usecase");
            System.out.println("CL  (ComponentListing)  " + String.format("%-7s", ComponentListing.VERSION) + " [HTTP] - list components info");
            System.out.println("SC  (SC)                " + String.format("%-7s", SC.VERSION) + " [HTTP] - tool to make SC-xml");
            System.out.println("MPL (MsgPathListing)    " + String.format("%-7s", MessagePathListing.VERSION) + " [HTTP] - list message path info");
            System.out.println("PF  (Performance)       " + String.format("%-7s", Performance.VERSION) + " [AMQP] - test Artemis throughput");
        }

        System.out.println("\nTo get more help on usage, run 'java -jar ekit.jar <QC|QP|QM|BRS|CDS|CC|BCK|EMD|EPA|EDX|EML|EFT|HELP>'");

        System.out.println("\nTo maintain a stable ECP/EDX-endpoint there are two fundamental aspects which every endpoint should handle:");
        System.out.println("  1) Queues must not grow for a long time or too large");
        System.out.println("     a) For ECP 4.12 and below, use QC for queue cleaning and some monitoring");
        System.out.println("     b) For ECP 4.14 and above, use BRS for queue cleaning, and QM for queue monitoring");
        System.out.println("  2) Connectivity to the remote endpoints you communicate with, must always be monitored. Use CC for this.");
        System.out.println("The other tools are useful in their own right, but not strictly necessary.");

        System.out.println("\n\nDisclaimer/License: ekit.jar can be used by anyone for any purpose without any restrictions. The tools are ");
        System.out.println("used in production in Statnett, but the author does not guarantee that the tools are bug-free or that they work ");
        System.out.println("as expected - no liability is taken. The source code is attached to the jar-file and can be viewed/scanned. The ");
        System.out.println("only major library used outside of core Java-classes is QPid JMS which is necessary to connect to AMQP-broker. ");
        System.out.println("The Qpid library in turn depends upon Netty library. Apart from this all other code is written by the author. ");
        System.out.println("Another goal for this code is that it should have a small footprint when it comes to CPU/Memory.\n");
    }
}
