package no.statnett.ecp.brs;

import no.statnett.ecp.brs.actions.*;
import no.statnett.ecp.brs.state.*;
import no.statnett.ecp.utils.Options;
import no.statnett.ecp.utils.SimpleParser;
import no.statnett.ecp.utils.URLParser;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;

public class BrokerShell {

  public static final String VERSION = "v2.2.0";

  // Options from input-args
  private static Map<String, Login> extraLogins = new HashMap<>();
  private static List<String> commands = null;

  // Useful global variables, set based on commands
  private static boolean echo = true;
  private static final QueueFilter queueFilter = new QueueFilter();
  private static final MessageFilter messageFilter = new MessageFilter();
  private static final ConsumerFilter consumerFilter = new ConsumerFilter();
  private static final List<String> filterHistory = new ArrayList<>();
  private static boolean filterChangedViewNotUpdated = true;

  /* INITIALIZE */

  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String commandsFromFile = Options.parseString(initArgsList, "-f");
    String commandsFromArg = Options.parseString(initArgsList, "-c");
    if (commandsFromFile != null) {
      File commandFile = Options.checkFile(commandsFromFile, false);
      commands = Files.readAllLines(commandFile.toPath());
    } else if (commandsFromArg != null) {
      commands = Arrays.asList(commandsFromArg.split(";"));
    }
    if (commands != null) { // Some commands have been specified, we'll filter out empty lines and comments
      commands = commands.stream().filter(c -> !c.trim().isEmpty()).filter(c -> !c.trim().startsWith("#")).collect(Collectors.toList());
      if (commands.isEmpty()) {
        System.out.println("Error: No commands found in file (-f) or in argument, exiting");
        System.exit(1);
      }
    }
    String loginFileStr = Options.parseString(initArgsList, "-h");
    if (loginFileStr != null) {
      File loginFile = Options.checkFile(loginFileStr, false);
      extraLogins = parseLoginFile(loginFile);
    }
    return initArgsList;
  }

  private static Map<String, Login> parseLoginFile(File loginFile) throws IOException {
    // The loginFile has the following format, the 3 last fields are optional:
    // hostname [user password protocol]
    Map<String, Login> loginMap = new HashMap<>();
    for (String line : Files.readAllLines(loginFile.toPath())) {
      if (line.startsWith("#") || line.trim().isEmpty())
        continue;
      String[] parts = line.split(" ");
      if (parts.length == 4) {
        loginMap.put(parts[0], new Login(parts[0], parts[1], parts[2], parts[3]));
      } else if (parts.length == 1) {
        loginMap.put(parts[0], new Login(parts[0], null, null, null));
      }
    }
    return loginMap;
  }


  /* THE MAIN METHODS */

  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    // READ OPTIONS
    List<String> mandatoryArgs = parseOptions(args);
    if (mandatoryArgs.size() != 1) {
      if (!mandatoryArgs.isEmpty()) {
        System.out.println("\nError: Expected exactly 1 (one) mandatory argument, either URL/PATH to broker, but received " + mandatoryArgs.size() + "\n");
      }
      if (echo)
        usage();
      System.exit(1);
    }

    // LOAD CONFIG - PREPARE INITIAL STATE
    Map<String, String> brokerConfig = URLParser.parse(mandatoryArgs.get(0), false);
    if (!extraLogins.isEmpty()) {
      extraLogins.remove(brokerConfig.get("host")); // Remove the main host from the extraHosts list - to avoid double listing
    }
    Config config = new Config(brokerConfig, extraLogins);
    State state = new State();

    // RUN
    if (commands != null) {
      script(state, config, commands);
    } else {
      interactive(state, config);
    }
  }

  private static void script(State state, Config config, List<String> commmands) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    // There is at least ONE command - this has been checked in parseOptions()
    boolean echoOn = !commmands.get(0).startsWith("e") || !commmands.get(0).endsWith("off");
    for (String command : commmands) {
      if (echoOn)
        System.out.println(">" + command);
      executeCommand(state, config, command);
      echoOn = echo; // The executeCommand will update the echo-flag if the command is "e off" or "e on"
      if (command.equalsIgnoreCase("x") || command.equalsIgnoreCase("exit") || command.equalsIgnoreCase("quit"))
        break;
    }
    if (echo)
      System.out.println("Exiting BrokerShell");
  }

  private static void interactive(State state, Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    System.out.println("BrokerShell for Artemis " + VERSION + " - Interactive mode. Enter h for help");
    Scanner scanner = new Scanner(System.in);
    String input;
    do {
      System.out.print(">");
      input = scanner.nextLine();
      if (!input.trim().isEmpty()) {
        for (String command : input.split(";"))
          executeCommand(state, config, command);
      }
    } while (!input.equalsIgnoreCase("x") && !input.equalsIgnoreCase("exit") && !input.equalsIgnoreCase("quit"));
    scanner.close();
    System.out.println("Exiting BrokerShell");
  }

  private static void executeCommand(State state, Config config, String input) throws NoSuchAlgorithmException, IOException, KeyManagementException {

    // Misc/generic commands + view-command (v)

    if (input.startsWith("e"))
      echo = !input.substring(1).trim().equalsIgnoreCase("off");
    else if (echo && input.equalsIgnoreCase("h"))
      help();
    else if (echo && input.startsWith("fl"))
      showSettings(input);
    else if (echo && input.equalsIgnoreCase("fh"))
      filterHistory();
    else if (input.equalsIgnoreCase("v") || input.equalsIgnoreCase("q")) {
      state.setQiList(refreshQueueList(config));
    }


    // Message filter commands:

    else if (input.startsWith("msrt")) {
      messageFilter.setMsrt(FilterResult.parseSortCmd(input, 9, echo));
    } else if (input.startsWith("age")) {
      noLongerSupportedFilter("mage");
    } else if (input.startsWith("mage")) {
      messageFilter.setMage(FilterResult.parseFilter(input, 4, true, -1, null, echo, messageFilter.getMage()));
    } else if (input.startsWith("mtyp")) {
      messageFilter.setMtyp(FilterResult.parseFilter(input, 4, false, null, null, echo, messageFilter.getMtyp()));
    } else if (input.startsWith("mbid")) {
      messageFilter.setMbid(FilterResult.parseFilter(input, 4, false, null, null, echo, messageFilter.getMbid()));
    } else if (input.startsWith("msnd")) {
      messageFilter.setMsnd(FilterResult.parseFilter(input, 4, false, null, null, echo, messageFilter.getMsnd()));
    } else if (input.startsWith("mrcv")) {
      messageFilter.setMrcv(FilterResult.parseFilter(input, 4, false, null, null, echo, messageFilter.getMrcv()));
    } else if (input.startsWith("mlim")) {
      messageFilter.setMlim(FilterResult.parseFilter(input, 1, false, 0, null, echo, null));
    }

    // Queue filter commands:
    else if (input.startsWith("qsrt")) {
      queueFilter.setQsrt(FilterResult.parseSortCmd(input, 6, echo));
    } else if (input.matches("^c\\s*\\d+$")) {
      noLongerSupportedFilter("qcon");
    } else if (input.startsWith("qcon")) {
      queueFilter.setQcon(FilterResult.parseFilter(input, 4, true, -1, null, echo, queueFilter.getQcon()));
    } else if (input.matches("^z\\s*\\d+$")) {
      noLongerSupportedFilter("qsiz");
    } else if (input.startsWith("qsiz")) {
      queueFilter.setQsiz(FilterResult.parseFilter(input, 4, true, -1, null, echo, queueFilter.getQsiz()));
    } else if (input.startsWith("qque")) {
      queueFilter.setQque(FilterResult.parseFilter(input, 4, false, null, null, echo, queueFilter.getQque()));
      filterHistory.add(input);
    } else if (input.startsWith("s")) {
      noLongerSupportedFilter("qsch");
    } else if (input.startsWith("qsch")) {
      queueFilter.setQsch(FilterResult.parseFilter(input, 4, false, null, null, echo, queueFilter.getQsch()));
      filterHistory.add(input);
    } else if (input.startsWith("qlim")) {
      queueFilter.setQlim(FilterResult.parseFilter(input, 1, false, 0, null, echo, queueFilter.getQlim()));
    }

    // Consumer filter commands:
    else if (input.startsWith("csrt")) {
      consumerFilter.setCsrt(FilterResult.parseSortCmd(input, 9, echo));
    } else if (input.startsWith("cscp")) {
      consumerFilter.setCscp(FilterResult.parseFilter(input, 4, false, null, null, echo, consumerFilter.getCscp()));
    } else if (input.startsWith("cage")) {
      consumerFilter.setCage(FilterResult.parseFilter(input, 4, true, -1, null, echo, consumerFilter.getCage()));
    } else if (input.startsWith("csch")) {
      consumerFilter.setCsch(FilterResult.parseFilter(input, 4, false, null, null, echo, consumerFilter.getCsch()));
      filterHistory.add(input);
    } else if (input.startsWith("ctrs")) {
      consumerFilter.setCtrs(FilterResult.parseFilter(input, 4, true, -1, null, echo, consumerFilter.getCtrs()));
    } else if (input.startsWith("cack")) {
      consumerFilter.setCack(FilterResult.parseFilter(input, 4, true, -1, null, echo, consumerFilter.getCack()));
    } else if (input.startsWith("clim")) {
      consumerFilter.setClim(FilterResult.parseFilter(input, 1, false, 0, null, echo, consumerFilter.getClim()));
    }

    // Action commands

    else { // The commands below will perform an action or a listing *depending* upon the queuelist
      if (filterChangedViewNotUpdated)
        System.out.println("Run 'q' command, then try again. Filters may have changed or no queues have been listed yet");
      else if (input.startsWith("p"))
        PurgeQueue.purge(state.getQiList(), config, input, echo);
      else if (input.startsWith("r"))
        RemoveMessages.remove(state.getQiList(), config, input, echo, messageFilter);
      else if (input.startsWith("n"))
        DeleteQueue.delete(state.getQiList(), config, input, echo);
      else if (input.startsWith("m"))
        listMessages(state, config, input, echo, messageFilter);
      else if (input.startsWith("c"))
        listConsumers(state, config, input, echo, consumerFilter);
      else if (input.startsWith("j"))
        printQueueInfoToJsonFile(state.getQiList(), input);
    }
  }

  private static void noLongerSupportedFilter(String newFilter) {
    // TODO Remove in 2027, since this warning/exit will have circulated for at least 1 year. 21/11-2025
    System.out.println("Filter no longer supported - replaced by '" + newFilter + "' - see help (h)");
    if (commands != null) {
      System.out.println("BRS terminates since a non-applied filter could have disastrous consequences");
      System.exit(1);
    }
  }



  /* LIST COMMANDS */

  private static List<QueueInfo> refreshQueueList(Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    List<QueueInfo> qiList;
    qiList = ListQueues.Retrieve.retrieveAndFilter(config, queueFilter); // updates the qiList and prints it
    ListQueues.Print.printToConsole(qiList, echo);
    filterChangedViewNotUpdated = false;
    return qiList;
  }

  private static void filterHistory() {
    for (int i = 0; i < filterHistory.size(); i++) {
      System.out.println((i + 1) + ": " + filterHistory.get(i));
    }
  }

  private static void listMessages(State state, Config config, String input, boolean echo, MessageFilter messageFilter) {
    Integer tmp = SimpleParser.getInt(input, false);
    List<MessageInfo> miList = new ArrayList<>();
    boolean miListRefreshed = false;
    if (tmp == null || tmp < 0) {
      miList = ListMessages.Retrieve.retrieveAndFilter(state.getQiList(), config, messageFilter);
      miListRefreshed = true;
    } else if (tmp > 0 && tmp <= state.getQiList().size()) {
      List<QueueInfo> selectedQIList = new ArrayList<>();
      selectedQIList.add(state.getQiList().get(tmp - 1));
      miList = ListMessages.Retrieve.retrieveAndFilter(selectedQIList, config, messageFilter);
      miListRefreshed = true;
    } else {
      System.out.println("Invalid queue number: " + tmp + ", command ignored");
    }
    if (miListRefreshed) {
      state.setMiList(miList);
      ListMessages.Print.printMI(state.getMiList(), echo);
    }
  }

  private static void listConsumers(State state, Config config, String input, boolean echo, ConsumerFilter consumerFilter) {
    Integer tmp = SimpleParser.getInt(input, false);
    List<ConsumerInfo> ciList;
    if (tmp == null || tmp < 0) {
      ciList = ListConsumers.Retrieve.retrieveAndFilter(state.getQiList(), config, consumerFilter);
    } else if (tmp > 0 && tmp <= state.getQiList().size()) {
      List<QueueInfo> selectedQIList = new ArrayList<>();
      selectedQIList.add(state.getQiList().get(tmp - 1));
      ciList = ListConsumers.Retrieve.retrieveAndFilter(selectedQIList, config, consumerFilter);
    } else {
      System.out.println("Invalid queue number: " + tmp + ", command ignored");
      return;
    }
    ListConsumers.Print.printCI(ciList, echo);
  }

  /* OTHER MISC FUNCTIONS */

  private static void printQueueInfoToJsonFile(List<QueueInfo> qiList, String input) throws IOException {
    String filename = input.substring(1).trim();
    if (filename.isEmpty()) {
      System.out.println("No filename specified, command ignored");
      return;
    }
    File file = Div.getValidFileReferenceForWrite(filename);
    if (file != null) {
      ListQueues.Print.printToJson(qiList, file);
    }
  }

  private static void showSettings(String input) {
    input = input.trim();
    boolean showQ = input.endsWith("q") || input.equals("fl") || input.equals("y");
    boolean showM = input.endsWith("m") || input.equals("fl") || input.equals("y");
    boolean showC = input.endsWith("c") || input.equals("fl") || input.equals("y");
    boolean showAll = input.equals("fl") || input.equals("y");

    // Queue-filter-explanations
    Integer qCol = queueFilter.getQsrt().number();
    String qsrtFilter = qCol == 0 ? "Not specified" : (qCol > 0 ? "A" : "De") + "scending by column " + qCol;
    String qconFilter = queueFilter.getQcon().number() < 0 ? "<ALL>" : toTextWithOp(queueFilter.getQcon(), "consumers");
    String qsizFilter = queueFilter.getQsiz().number() < 0 ? "<ALL>" : toTextWithOp(queueFilter.getQsiz(), "messages");
    String qqueFilter = queueFilter.getQque().arg() == null ? "<ALL>" : queueFilter.getQque().arg();
    String qschFilter = queueFilter.getQsch().arg() == null ? "<ALL>" : toTextWithNeg(queueFilter.getQsch().arg());
    String qlimFilter = queueFilter.getQlim().number() < 1 ? "<ALL>" : "First " + queueFilter.getQlim().number() + " rows";

    // Message-filter-explanations
    Integer mCol = messageFilter.getMsrt().number();
    String msrtFilter = mCol == 0 ? "Not specified" : (mCol > 0 ? "A" : "De") + "scending by column " + mCol;
    String mageFilter = messageFilter.getMage().number() < 0 ? "<ALL>" : toTextWithOp(messageFilter.getMage(), " minutes");
    String mtypFilter = messageFilter.getMtyp().arg() == null ? "<ALL>" : toTextWithNeg(messageFilter.getMtyp().arg());
    String mbidFilter = messageFilter.getMbid().arg() == null ? "<ALL>" : toTextWithNeg(messageFilter.getMbid().arg());
    String msndFilter = messageFilter.getMsnd().arg() == null ? "<ALL>" : toTextWithNeg(messageFilter.getMsnd().arg());
    String mrcvFilter = messageFilter.getMrcv().arg() == null ? "<ALL>" : toTextWithNeg(messageFilter.getMrcv().arg());
    String mlimFilter = messageFilter.getMlim().number() < 1 ? "<ALL>" : "First " + messageFilter.getMlim().number() + " rows";

    // Consumer-filter-explanations
    Integer cCol = consumerFilter.getCsrt().number();
    String csrtFilter = cCol == 0 ? "Not specified" : (cCol > 0 ? "A" : "De") + "scending by column " + cCol;
    boolean scopeLocal = consumerFilter.getCscp().arg() != null && consumerFilter.getCscp().arg().equalsIgnoreCase("loc");
    boolean scopeRemote = consumerFilter.getCscp().arg() != null && consumerFilter.getCscp().arg().equalsIgnoreCase("rem");
    String cscpFilter = !scopeLocal && !scopeRemote ? "<ALL>" : scopeLocal ? "Local" : "Remote";
    String cageFilter = consumerFilter.getCage().number() < 0 ? "<ALL>" : toTextWithOp(consumerFilter.getCage(), "minutes");
    String cschFilter = consumerFilter.getCsch().arg() == null ? "<ALL>" : toTextWithNeg(consumerFilter.getCsch().arg());
    String ctrsFilter = consumerFilter.getCtrs().number() < 0 ? "<ALL>" : toTextWithOp(consumerFilter.getCtrs(), "messages");
    String cackFilter = consumerFilter.getCack().number() < 0 ? "<ALL>" : toTextWithOp(consumerFilter.getCack(), "messages");
    String climFilter = consumerFilter.getClim().number() < 1 ? "<ALL>" : "First " + consumerFilter.getClim().number() + " rows";

    System.out.println("Que/Msg   Type   Operation        Cmd     Comp    : Setting");
    System.out.println("----------------------------------------------------------------");
    if (showQ) {
      System.out.println("Queue     Sort   Column-sort      (qsrt)  (=)     : " + qsrtFilter);
      System.out.println("Queue     Filter No of consumers  (qcon)  (>=<)   : " + qconFilter);
      System.out.println("Queue     Filter Queue-size       (qsiz)  (>=<)   : " + qsizFilter);
      System.out.println("Queue     Filter Queues (list)    (qque)  (or)    : " + qqueFilter);
      System.out.println("Queue     Filter Search           (qsch)  (regex) : " + qschFilter);
      System.out.println("Queue     Filter Limit-N-rows     (qlim)  (=)     : " + qlimFilter);
    }
    if (showAll)
      System.out.println("----------------------------------------------------------------");
    if (showM) {
      System.out.println("Message   Sort   Column-sort      (msrt)  (=)     : " + msrtFilter);
      System.out.println("Message   Filter TTL (minutes)    (mage)  (>=<)   : " + mageFilter);
      System.out.println("Message   Filter Msg/BusinessType (mtyp)  (%)     : " + mtypFilter);
      System.out.println("Message   Filter BAMessageId      (mbid)  (%)     : " + mbidFilter);
      System.out.println("Message   Filter Sender           (msnd)  (%)     : " + msndFilter);
      System.out.println("Message   Filter Receiver         (mrcv)  (%)     : " + mrcvFilter);
      System.out.println("Message   Filter Limit-N-rows     (mlim)  (=)     : " + mlimFilter);
    }
    if (showAll)
      System.out.println("----------------------------------------------------------------");
    if (showC) {
      System.out.println("Consumer  Sort   Column-sort     (csrt)  (=)     : " + csrtFilter);
      System.out.println("Consumer  Filter Scope           (cscp)  (=)     : " + cscpFilter);
      System.out.println("Consumer  Filter Minutes old     (cage)  (>=<)   : " + cageFilter);
      System.out.println("Consumer  Filter Search          (csch)  (regex) : " + cschFilter);
      System.out.println("Consumer  Filter Transit msgs    (ctrs)  (>=<)   : " + ctrsFilter);
      System.out.println("Consumer  Filter Ack msgs        (cack)  (>=<)   : " + cackFilter);
      System.out.println("Consumer  Filter Limit-N-rows    (clim)  (=)     : " + climFilter);
    }
  }

  private static String toTextWithNeg(String searchString) {
    return searchString.startsWith("!") ? searchString.substring(1) + " (negated:true)" : searchString + " (negated:false)";
  }

  private static String toTextWithOp(FilterResult numberOp, String designation) {
    if (designation != null && designation.endsWith("s") && numberOp.number() == 1) {
      designation = designation.substring(0, designation.length() - 1);
    }
    return numberOp.op() + " " + numberOp.number() + (designation == null ? "" : " " + designation);
  }


  // Interactive help
  private static void help() {
    System.out.println(" MISC:");
    System.out.println("   h            - help");
    System.out.println("   x            - exit");
    System.out.println("   e<on|off>    - off: will output only the most necessary information, useful for script-mode");
    System.out.println(" LISTINGS:");
    System.out.println("   q            - view queues (refresh list - subsequent purge/delete will be based on this list)");
    System.out.println("   m[<number>]  - view messages in all queues. If number is given, only messages from that queue will be shown");
    System.out.println("   c[<number>]  - view consumers on the queues. If number is given, only consumers for that queue will be shown");
    System.out.println("   fh           - show filter (search/queues) history");
    System.out.println("   fl[q|m|c]    - show filter/sort settings");
    System.out.println("   fr[q|m|c]    - reset all filters, optional reset only queue-filter 'frq', message-filter 'frm' or consumer-filter 'frc'");
    System.out.println(" QUEUE FILTERS:");
    System.out.println("   qsrt<number>     - sort column ascending using column number 1 to 6, or descending from -1 to -6");
    System.out.println("   qcon<op><number> - filter based on number of consumers. <op> can be >,= or <");
    System.out.println("   qsiz<op><number> - lists queues with size larger than or equal to this number.");
    System.out.println("   qque<queues>     - lists queues with these names (comma-separated).");
    System.out.println("   qsch<string>     - match lines with this regexp/string. Prefixing with ! will negate search.");
    System.out.println("   qlim<number>     - limit output to this number of lines (after filter/sorting is applied).");
    System.out.println(" MESSAGE FILTERS:");
    System.out.println("   msrt<number>     - sort column ascending using column number 1 to 9, or descending from -1 to -9");
    System.out.println("   mage<op><number> - list based on minutes old.  <op> can be >,= or <");
    System.out.println("   mtyp<string>     - list based on messagetype. % allowed for wildcard. Prefix with ! to negate. Example: 'mtypNBM%AOF-1'");
    System.out.println("   mbid<string>     - list based on BAmessageId. % allowed for wildcard. Prefix with ! to negate. Example: 'mbid123%456'");
    System.out.println("   msnd<string>     - list based on sender. % allowed for wildcard. Prefix with ! to negate. Example: 'msnd!50V00000ABC'");
    System.out.println("   mrcv<string>     - list based on receiver. % allowed for wildcard. Prefix with ! to negate. Example: 'mrcv50V00000ABC'");
    System.out.println("   mlim<number>     - limit output to this number of lines (after filter/sorting is applied).");
    System.out.println(" CONSUMER FILTERS:");
    System.out.println("   csrt<number>     - sort column ascending using column number 1 to 9, or descending from -1 to -9");
    System.out.println("   cscp[loc|rem]    - list only loc(al) consumers (127.0.0.1) or only rem(ote) consumers");
    System.out.println("   cage<op><number> - list based on minutes old.  <op> can be >,= or <");
    System.out.println("   csch<string>     - match lines with this regexp/string. Prefixing with ! will negate search.");
    System.out.println("   ctrs<op><number> - list based on numbers of 'in-transit'. <op> can be >,= or <");
    System.out.println("   cack<op><number> - list based on numbers of 'received/ack' by the consumer. <op> can be >,= or <");
    System.out.println("   clim<number>     - limit output to this number of lines (after filter/sorting is applied).");
    System.out.println(" ACTIONS - TAKE CARE!:");
    System.out.println("   p<number>        - purge queue (number is queue number). Purge ALL queues in the printed list if number is -1");
    System.out.println("   r<number>        - same operation as purge (p), but will also apply the message filters to the messages in the queue");
    System.out.println("   n<number>        - nuke/destroy queues (number is queue number). Delete ALL queues in the printed list if number is -1");
    System.out.println("   j<file>          - print queue-info to file in JSON-format");
  }

  // Command line help
  private static void usage() {
    System.out.println("BrokerShell (BRS) " + VERSION + " is made to browse and manage queues on Artemis brokers used by");
    System.out.println("ECP/EDX-installations. The broker must respond on 8161-port (console-api-port) for this tool to work.");
    System.out.println("You can run BRS locally on same host, then things will most likely work. If you want to allow for");
    System.out.println("remote access (which will enable the greatest power of BRS), then change the jolokia-access.xml to this:");
    System.out.println("    <?xml version=\"1.0\" encoding=\"utf-8\"?>");
    System.out.println("    <restrict><cors><allow-origin>*</allow-origin></cors></restrict>");
    System.out.println("If you allow for remote access, then you can specify multiple hosts in the -h option, and BRS will list");
    System.out.println("queues and perform delete/purge operations on all hosts listed. Combined with the -f/-c option + crontab");
    System.out.println("you can automate management of queues on multiple brokers from a single command-line tool.");
    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar BRS [OPTIONS] BROKER\n");
    System.out.println("\n\nThe options and arguments:");
    System.out.println(" OPTIONS       : ");
    System.out.println("                 -f <commands-file> : each line in file specify an interactive command.");
    System.out.println("                 -c <commands>      : a semi-colon separated list of interactive commands. Ignored if -f is specified");
    System.out.println("                 -h <host-file>     : each line: hostname [user password protocol] - omit 3 last args to use URL-data");
    System.out.println(" BROKER        : <URL|PATH> - either specify a broker-URL, or a path for the ECP/EDX-configuration");
    System.out.println("                 URL                : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println("                 PATH               : Specify either the conf-path for ECP or EDX (ex: /etc/ecp-endpoint or /etc/edx-toolbox)");
    System.out.println();
    System.out.println("\nExample 1:  Start BRS in interactive mode");
    System.out.println("\tjava -jar ekit.jar BRS https://endpoint:password@localhost:8161");
    System.out.println("\nExample 2:  Interactive mode - browse and manage queues on multiple hosts");
    System.out.println("\tjava -jar ekit.jar BRS -h hosts.txt https://endpoint:password@remotehost:8161");
    System.out.println("\nExample 3:  Script mode - run commands from file");
    System.out.println("\tjava -jar ekit.jar BRS -f commands.txt https://endpoint:password@localhost:8161");
    System.out.println("\nExample 4:  Script mode - find queues with min 1 msg on queue, then find msg 10min or older and remove those");
    System.out.println("\tjava -jar ekit.jar BRS -c \"eoff;z1;age10;v;r-1\" https://endpoint:password@localhost:8161");
  }
}
