package no.statnett.ecp.brs.actions;

import no.statnett.ecp.brs.Config;
import no.statnett.ecp.brs.Div;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.SimpleParser;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DeleteQueue {
    public static void delete(List<QueueInfo> qiList, Config config, String input, boolean echo) throws NoSuchAlgorithmException, IOException, KeyManagementException {
        Integer tmp = SimpleParser.getInt(input);
        if (tmp == null)
            return;
        if (tmp < 0) {
            int dangerousDelete = dangerousDelete(qiList);
            if (tmp == -1 && dangerousDelete > 0) {
                System.out.println("This deletion of queues is aborted, you're attempting to delete " + dangerousDelete + " queues WITH consumers. If you want to persist, run 'n-2' to confirm");
                return;
            }
            if (echo)
                System.out.println("Deleting all queues in the printed list, since number is -1");
            deleteImpl(qiList, config, echo);
        } else if (tmp > 0 && tmp <= qiList.size()) {
            if (echo)
                System.out.println("Deleting queue number " + tmp);
            List<QueueInfo> selectedQIList = new ArrayList<>();
            selectedQIList.add(qiList.get(tmp - 1));
            deleteImpl(selectedQIList, config, echo);
        } else {
            System.out.println("Invalid queue number: " + tmp + ", command ignored");
        }
    }

    private static void deleteImpl(List<QueueInfo> limitedQIList, Config config, boolean echo) throws NoSuchAlgorithmException, IOException, KeyManagementException {
        int executionsOk = 0;
        int failed = 0;
        if (echo)
            System.out.print("Starting to delete queues.");
        String echoOffOneLinerSummary = "";
        Map<String, String> broker2Encoded = new HashMap<>();
        for (QueueInfo qi : limitedQIList) {
            try {
                if (broker2Encoded.get(qi.getHost()) == null) { // The same broker on the same host, no need to look up again
                    String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(Div.prot(config, qi), qi.getHost(), config.get("port"), Div.user(config, qi), Div.pass(config, qi));
                    broker2Encoded.put(qi.getHost(), brokerNameEncoded);
                }
                boolean success = ArtemisConsoleAPI.destroyQueue(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi), qi.getQueueName());
                if (success) {
                    echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getQueueName() + ":delete-ok] ";
                    executionsOk++;
                } else {
                    echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getQueueName() + ":delete-failed] ";
                    failed++;
                }
                if (echo)
                    System.out.print(".");
            } catch (Exception e) {
                System.out.println("Error occurred: During deletion of queue: " + qi.getQueueName() + " on host " + qi.getHost() + ": " + e.getMessage());
                failed++;
            }
        }
        String failedStr = failed > 0 ? "Failed to delete " + failed + " queues." : "";
        if (echo) {
            System.out.println("\nDelete completed: Deleted " + executionsOk + " out of " + limitedQIList.size() + " queues listed. " + failedStr);
        } else {
            String timestamp = LocalDateTime.now().format(Const.localTmsSec);
            System.out.println(timestamp + " Delete completed: Deleted " + executionsOk + " out of " + limitedQIList.size() + " queues (" + echoOffOneLinerSummary + ")");

        }
    }

    private static int dangerousDelete(List<QueueInfo> qiList) {
        int dangerousCount = 0;
        for (QueueInfo qi : qiList) {
            if (qi.getConsumers() > 0) {
                dangerousCount++;
            }
        }
        return dangerousCount;
    }
}
