package no.statnett.ecp.brs.actions;

import no.statnett.ecp.brs.Config;
import no.statnett.ecp.brs.Div;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.SimpleParser;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PurgeQueue {
    public static void purge(List<QueueInfo> qiList, Config config, String input, boolean echo) {
        Integer tmp = SimpleParser.getInt(input);
        if (tmp == null)
            return;
        if (tmp < 0) {
            int dangerousCount = dangerousPurge(qiList);
            if (tmp == -1 && dangerousCount > 0) {
                System.out.println("This purge is aborted, you're attempting to purge at least " + dangerousCount + " messages which might not be intended. If you want to persist, run 'p-2' to confirm");
                return;
            }
            if (echo)
                System.out.println("Purging all queues in the printed list, since number is -1");
            purgeImpl(qiList, config, echo);
        } else if (tmp > 0 && tmp <= qiList.size()) {
            if (echo)
                System.out.println("Purging queue number " + tmp);
            List<QueueInfo> selectedQIList = new ArrayList<>();
            selectedQIList.add(qiList.get(tmp - 1));
            purgeImpl(selectedQIList, config, echo);
        } else {
            System.out.println("Invalid queue number: " + tmp + ", command ignored");
        }
    }

    private static void purgeImpl(List<QueueInfo> limitedQIList, Config config, boolean echo) {
        int messagesPurged = 0;
        int failed = 0;
        if (echo)
            System.out.print("Starting to purge.");
        String echoOffOneLinerSummary = "";
        Map<String, String> broker2Encoded = new HashMap<>();
        for (QueueInfo qi : limitedQIList) {
            try {
                if (broker2Encoded.get(qi.getHost()) == null) { // The same broker on the same host, no need to look up again
                    String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(Div.prot(config, qi), qi.getHost(), config.get("port"), Div.user(config, qi), Div.pass(config, qi));
                    broker2Encoded.put(qi.getHost(), brokerNameEncoded);
                }
                ArtemisConsoleAPI.ACAPIResult result = ArtemisConsoleAPI.removeAllMessages(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi), qi.getQueueName());
                if (result.getStatus() != null && result.getStatus().equals("200")) {
                    messagesPurged += Integer.parseInt(result.getValue());
                    if (messagesPurged > 0)
                        echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getQueueName() + ":" + messagesPurged + "] ";
                } else {
                  if (echo)
                    System.out.println("Failed to purge queue: " + qi.getQueueName() + " on host: " + qi.getHost() + " - " + result.getRawResponse());
                  failed++;
                  echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getQueueName() + ":failed] ";
                }
                if (echo)
                    System.out.print(".");
            } catch (Exception e) {
                if (echo)
                  System.out.println("Error occurred: During purging/removeAllMessages of queue: " + qi.getQueueName() + " on host " + qi.getHost() + ": " + e.getMessage());
                failed++;
                echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getQueueName() + ":failed] ";
            }
        }
        if (echo) {
            System.out.println("\nPurge completed: " + messagesPurged + " messages from " + limitedQIList.size() + " queues");
        } else { // despite 'echo off' we still write this very important info to stdout
            String timestamp = LocalDateTime.now().format(Const.localTmsSec);
            if (messagesPurged > 0)
              System.out.println(timestamp + " INFO  Purged  " + messagesPurged + " messages from " + limitedQIList.size() + " queues. " + echoOffOneLinerSummary);
            else
              System.out.println(timestamp + " DEBUG Nothing was purged. " + echoOffOneLinerSummary);
            if (failed > 0) {
              System.out.println(timestamp + " WARN  Purge failed on " + failed + " queues");
            }
        }
    }

    private static int dangerousPurge(List<QueueInfo> qiList) {
        int dangerousCount = 0;
        for (QueueInfo qi : qiList) {
            if (qi.getSize() > 0 && qi.getConsumers() > 0)
                dangerousCount += qi.getSize();
            else if (qi.getSize() > 0 && !qi.getQueueName().contains("DLQ") && !qi.getQueueName().contains("Expiry"))
                dangerousCount += qi.getSize();
        }
        return dangerousCount;
    }
}
