package no.statnett.ecp.cds;

import no.statnett.ecp.cds.actions.ListComponents;
import no.statnett.ecp.cds.actions.ListMessagePaths;
import no.statnett.ecp.cds.state.*;
import no.statnett.ecp.utils.Options;
import no.statnett.ecp.utils.SimpleParser;
import no.statnett.ecp.utils.URLParser;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

public class FAMonitor {

  public static final String VERSION = "v0.9.0";

  public static final Pattern versionPattern = Pattern.compile("^\\d+[.\\d+]*$");

  // Options from input-args
  private static List<String> commands = null;

  // Useful global variables, set based on commands
  private static boolean echo = true;
  private static final ComponentFilter componentFilter = new ComponentFilter();
  private static final MessagePathFilter messagePathFilter = new MessagePathFilter();

  private static final List<String> filterHistory = new ArrayList<>();
  private static boolean filterChangedViewNotUpdated = true;

  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String commandsFromFile = Options.parseString(initArgsList, "-f");
    String commandsFromArg = Options.parseString(initArgsList, "-c");
    if (commandsFromFile != null) {
      File commandFile = Options.checkFile(commandsFromFile, false);
      commands = Files.readAllLines(commandFile.toPath());
    } else if (commandsFromArg != null) {
      commands = Arrays.asList(commandsFromArg.split(";"));
    }
    if (commands != null) { // Some commands have been specified, we'll filter out empty lines and comments
      commands = commands.stream().filter(c -> !c.trim().isEmpty()).filter(c -> !c.trim().startsWith("#")).collect(Collectors.toList());
      if (commands.isEmpty()) {
        System.out.println("Error: No commands found in file (-f) or in argument, exiting");
        System.exit(1);
      }
    }
    return initArgsList;
  }


  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    // READ OPTIONS
    List<String> mandatoryArgs = parseOptions(args);
    if (mandatoryArgs.size() != 1) {
      if (!mandatoryArgs.isEmpty()) {
        System.out.println("\nError: Expected exactly 1 (one) mandatory argument: URL to the CD, but received " + mandatoryArgs.size() + "\n");
      }
      if (echo)
        usage();
      System.exit(1);
    }

    // LOAD CONFIG - PREPARE INITIAL STATE
    Map<String, String> cdConfig = URLParser.parse(mandatoryArgs.get(0), false);
    Config config = new Config(cdConfig);
    State state = new State();

    // Establish which ComponentDirectory this is
    state.setDashboard(retrieveDashboard(config));

    // RUN
    executeMonitor(state, config);
  }

  public static Dashboard retrieveDashboard(Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
    Dashboard dashboard = new Dashboard();

    URL url = new URL(protocol, config.get("host"), Integer.parseInt(config.get("port")), "/ECP_MODULE/dashboard");
    String response = getECPHTTResponse("GET", url, config.get("user"), config.get("password"), null);
    if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
      return null;
    }
    JSONObject jsonObject = new JSONObject(response);
        /*
             {
                "activeComponentsCount": 79,
                "certificateExpiration": "2026-06-13",
                "certificateType": "AUTHENTICATION",
                "dateNow": "2025-06-20T15:37:08.311084367",
                "description": "",
                "directoryRestart": null,
                "id": "50V000000000111W",
                "jobs": {
                    "completedJobsCount": 7,
                    "failedJobsCount": 0,
                    "waitingToRunJobsCount": 1
                },
                "requestCount": 0,
                "synchronization": "OK",
                "version": "4.16.0.2186",
                "versionWarning": null

         */

    dashboard.setCdCode(jsonObject.getString("id"));
    dashboard.setVersion(jsonObject.getString("version"));
    dashboard.setActiveComponents(jsonObject.getInt("activeComponentsCount"));
    return dashboard;
  }


  private static void executeCommand(State state, Config config, String input) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    if (input.startsWith("e"))
      echo = !input.substring(1).trim().equalsIgnoreCase("off");
    else if (echo && input.equalsIgnoreCase("h"))
      help();
    else if (echo && input.equalsIgnoreCase("y"))
      showSettings();
    else if (echo && input.equalsIgnoreCase("f"))
      filterHistory();
    else if (input.equalsIgnoreCase("v"))
      state.setCList(refreshComponentList(config, state.getDashboard().getCdCode()));
    else if (input.startsWith("sch"))
      setSearchFilter(input);
    else if (input.startsWith("asc"))
      setAscendingSort(input);
    else if (input.startsWith("dsc"))
      setDescendingSort(input);
    else if (input.startsWith("ver"))
      setVersionFilter(input);
    else if (input.startsWith("val"))
      setValidToFilter(input);
    else if (input.startsWith("scp"))
      setScopeFilter(input);
    else if (input.startsWith("mpc"))
      setMpCountFilter(input);
    else if (input.startsWith("stat"))
      setStatFilter(input);
    else if (input.startsWith("code"))
      setCodeFilter(input);
    else if (input.startsWith("broker"))
      setBrokerFilter(input);
    else if (input.startsWith("type"))
      setMessagePathFilter(input);
    else { // The commands below will perform an action - we'll prevent that action if v-command has not been run
      if (filterChangedViewNotUpdated)
        System.out.println("Run 'v' command, then try again. Filters may have changed or no components have been listed yet");
      else if (input.startsWith("m"))
        listMessagePaths(state, config, input, echo);
      else {
        System.out.println("Unknown command: " + input);
      }
    }
  }

  private static void executeMonitor(State state, Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    // There is at least ONE command - this has been checked in parseOptions()
    long startMs =  System.currentTimeMillis();
    executeCommand(state, config, "sch (FA)");
    while (true) {
      // 1.time the componentFilter.getEndpoints() is empty - will search for all endpoints
      executeCommand(state, config, "v");
      if (state.getCList().isEmpty()) {
        System.out.println("After " + (System.currentTimeMillis() - startMs)/1000 + " seconds, all endpoints have successfully synchronized");
        break;
      }
      // Next round - only search for these endpoints!
      componentFilter.setEndpoints(state.getCList().stream().map(c -> c.getCode()).collect(Collectors.toList()));
      System.out.println("We've run for " + (System.currentTimeMillis() - startMs)/1000 + " seconds and still have " + componentFilter.getEndpoints().size() + " that reports (FA)iled synchronization");
      try {
        Thread.sleep(60000);
      } catch (InterruptedException e) {
        throw new RuntimeException(e);
      }
    }
    if (echo)
      System.out.println("Exiting CDShell");
  }

  private static void interactive(State state, Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    System.out.println("CDShell for ECP CD " + VERSION + " - Interactive mode. Enter h for help");
    Scanner scanner = new Scanner(System.in);
    String input;
    do {
      System.out.print(">");
      input = scanner.nextLine();
      if (!input.trim().isEmpty()) {
        executeCommand(state, config, input);
      }
    } while (!input.equalsIgnoreCase("x") && !input.equalsIgnoreCase("exit") && !input.equalsIgnoreCase("quit"));
    scanner.close();
    System.out.println("Exiting CDShell");
  }


  private static void setDescendingSort(String input) {
    Integer tmp = 0;
    if (input.length() > 3) {
      tmp = SimpleParser.getInt(input);
    }
    if (tmp == null)
      return;
    if (tmp > 8) {
      System.out.println("Sorting by column " + tmp + " is not supported, only 1-8, command is ignored");
    } else if (tmp < 1) {
      if (echo) {
        if (componentFilter.getSortAscending() > 0) {
          System.out.println("Descending sort is turned off, but has no effect since ascending sort of column " + componentFilter.getSortAscending() + " takes precedence");
        } else {
          System.out.println("Descending sort is turned off, will show default listing");
        }
      }
      componentFilter.setSortDescending(0);
    } else {
      componentFilter.setSortDescending(tmp);
      componentFilter.setSortAscending(0);
      if (echo)
        System.out.println("Sorting by column " + tmp + " in descending order");
    }
  }

  private static void setAscendingSort(String input) {
    Integer tmp = 0;
    if (input.length() > 3) {
      tmp = SimpleParser.getInt(input);
    }
    if (tmp == null)
      return;
    if (tmp > 8) {
      System.out.println("Sorting by column " + tmp + " is not supported, only 1-8, command is ignored");
    } else if (tmp < 1) {
      if (echo) {
        if (componentFilter.getSortDescending() > 0) {
          System.out.println("Ascending sort is turned off, will go back to descending sort on column " + componentFilter.getSortDescending());
        } else {
          System.out.println("Ascending sort is turned off, will show default listing");
        }
      }
      componentFilter.setSortAscending(0);
    } else {
      componentFilter.setSortAscending(tmp);
      componentFilter.setSortDescending(0);
      if (echo)
        System.out.println("Sorting by column " + tmp + " in ascending order");
    }
  }

  private static void setSearchFilter(String input) {
    componentFilter.setSearchString(null);
    if (input.length() > 3) {
      componentFilter.setSearchString(input.substring(3).trim());
      if (componentFilter.getSearchString().isEmpty())
        componentFilter.setSearchString(null);
    }
    if (componentFilter.getSearchString() == null) {
      if (echo)
        System.out.println("Filtering by search string is turned off, will match ALL components");
    } else {
      filterHistory.add(input);
      if (echo)
        System.out.println("Search string filter is set to '" + componentFilter.getSearchString() + "' (will be negated: " + componentFilter.getSearchString().startsWith("!") + ")");
    }
    filterChangedViewNotUpdated = true;
  }

  private static void setVersionFilter(String input) {
    componentFilter.setVersion(null);
    componentFilter.setVersionOperator(null);
    if (input.length() >= 5) {
      String operator = input.substring(3, 4);
      if (operator.equals("<") || operator.equals(">") || operator.equals("="))
        componentFilter.setVersionOperator(operator);
      else {
        if (echo)
          System.out.println("Invalid version operator '" + operator + "', will match ALL versions");
        return;
      }
      String version = input.substring(4);
      if (versionPattern.matcher(version).matches()) {
        componentFilter.setVersion(version);
      } else {
        if (echo)
          System.out.println("Invalid version '" + version + "', will match ALL versions");
        return;
      }
    }

    if (componentFilter.getVersion() == null || componentFilter.getVersionOperator() == null) {
      if (echo)
        System.out.println("Filtering by version is turned off, will match ALL versions");
    } else {
      filterHistory.add(input);
      if (echo)
        System.out.println("Version filter is set to " + componentFilter.getVersionOperator() + componentFilter.getVersion());
    }
    filterChangedViewNotUpdated = true;
  }

  private static void setValidToFilter(String input) {
    componentFilter.setValidTo(null);
    componentFilter.setValidToOperator(null);
    if (input.length() >= 5) {
      String operator = input.substring(3, 4);
      if (operator.equals("<") || operator.equals(">") || operator.equals("="))
        componentFilter.setValidToOperator(operator);
      else {
        if (echo)
          System.out.println("Invalid validTo operator '" + operator + "', will match ALL validTo");
        return;
      }
      componentFilter.setValidTo(input.substring(4));
    }
    if (componentFilter.getValidTo() == null || componentFilter.getValidToOperator() == null) {
      if (echo)
        System.out.println("Filtering by validTo is turned off, will match ALL validTo");
    } else {
      filterHistory.add(input);
      if (echo)
        System.out.println("ValidTo filter is set to " + componentFilter.getValidToOperator() + componentFilter.getValidTo());
    }
    filterChangedViewNotUpdated = true;
  }

  private static void setScopeFilter(String input) {
    componentFilter.setScope(null);
    if (input.length() > 3) {
      String type = input.substring(3).trim();
      if (type.equalsIgnoreCase("loc") || type.equalsIgnoreCase("rem") || type.equalsIgnoreCase("all"))
        componentFilter.setScope(type.toLowerCase());
    }
    if (componentFilter.getScope() == null) {
      if (echo)
        System.out.println("Filtering by scope is turned off, will match ALL components");
    } else {
      if (echo)
        System.out.println("Scope filter is set to '" + componentFilter.getScope() + "'");
    }
    filterChangedViewNotUpdated = true;
  }

  private static void setMpCountFilter(String input) {
    Integer tmp = -1;
    if (input.length() > 3) {
      tmp = SimpleParser.getInt(input);
    }
    if (tmp == null)
      return;
    if (tmp == -1) {
      if (echo) {
        System.out.println("Filtering on message path count is turned off, will match ALL message path counts");
      }
      componentFilter.setMpCount(-1);
    } else if (tmp < -1) {
      if (echo) {
        System.out.println("Filtering on message path count " + tmp + " is not supported, will match ALL message path counts");
      }
      componentFilter.setMpCount(-1);
    } else {
      componentFilter.setMpCount(tmp);
      if (echo)
        System.out.println("Filtering on message path count " + tmp + " - only endpoints with this exact count will be listed");
    }
    filterChangedViewNotUpdated = true;
  }

  private static void setStatFilter(String input) {
    messagePathFilter.setStat(null);
    boolean wrongInput = false;
    if (input.length() == 8) {
      String stat = input.substring(4).trim().toLowerCase();
      if (stat.matches("[ayn?]{4}"))
        messagePathFilter.setStat(stat);
      else
        wrongInput = true;
    } else {
      wrongInput = input.length() == 4 ? false : true;
    }
    if (messagePathFilter.getStat() == null) {
      if (echo)
        if (wrongInput)
          System.out.println("Invalid stat filter '" + (input.length() == 4 ? "" : input.substring(4)) + "', will match ALL states (press 'h' for help)");
        else
          System.out.println("Filtering by stat is turned off, will match ALL states");
    } else {
      if (echo)
        System.out.println("Stat filter is set to '" + messagePathFilter.getStat() + "'");
    }
  }

  private static void setCodeFilter(String input) {
    messagePathFilter.setCode(null);
    if (input.length() > 6) {
      messagePathFilter.setCode(input.substring(4).trim());
    }
    if (messagePathFilter.getCode() == null) {
      if (echo)
        System.out.println("Filtering by code is turned off, will match ALL codes");
    } else {
      if (echo)
        System.out.println("Code filter is set to '" + messagePathFilter.getCode() + "'");
    }
  }

  private static void setBrokerFilter(String input) {
    messagePathFilter.setBroker(null);
    if (input.length() > 6) {
      messagePathFilter.setBroker(input.substring(6).trim());
    }
    if (messagePathFilter.getBroker() == null) {
      if (echo)
        System.out.println("Filtering by broker is turned off, will match ALL brokers");
    } else {
      if (echo)
        System.out.println("Broker filter is set to '" + messagePathFilter.getBroker() + "'");
    }
  }

  private static void setMessagePathFilter(String input) {
    messagePathFilter.setMessageType(null);
    if (input.length() > 4) {
      messagePathFilter.setMessageType(input.substring(4).trim());
    }
    if (messagePathFilter.getMessageType() == null) {
      if (echo)
        System.out.println("Filtering by message type is turned off, will match ALL message types");
    } else {
      if (echo)
        System.out.println("Message type filter is set to '" + messagePathFilter.getMessageType() + "'");
    }
  }

  private static List<Component> refreshComponentList(Config config, String cdCode) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    List<Component> cList = ListComponents.Retrieve.retrieveAndFilter(config, componentFilter, messagePathFilter, cdCode); // updates the cList and prints it
    ListComponents.Print.printToConsole(cList, echo);
    filterChangedViewNotUpdated = false;
    return cList;
  }

  private static void listMessagePaths(State state, Config config, String input, boolean echo) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    List<Component> cList = ListMessagePaths.Retrieve.retrieveAndFilter(state.getCList(), config, messagePathFilter);
    ListMessagePaths.Print.printMP(cList, echo);
  }

  private static void addCMP() {
    // POST http://localhost:8080/ECP_MODULE/directory/paths
        /*
             {
                 "applyToEndpoints": [
                     {
                         "applied": true,
                         "canBeRevoked": false,
                         "code": "50V-SN-TEST2-AT7",
                         "company": {
                             "contactEmail": "ecp@statnett.no",
                             "contactPerson": "Morten Simonsen",
                             "contactPhone": "004793480511",
                             "organization": "Statnett (TEST2)"
                         },
                         "componentDirectory": "50V000000000111W",
                         "componentStatus": null,
                         "created": "2024-05-21T15:57:22.527",
                         "envName": null,
                         "id": "50V-SN-TEST2-AT7",
                         "implementationVersion": "4.16.0.2186",
                         "isNotPushed": true,
                         "lastModification": "2025-03-20T15:04:46.387",
                         "location": null,
                         "networks": [
                             "DefaultNetwork"
                         ],
                         "projectName": null,
                         "type": "ENDPOINT",
                         "url": [
                         ],
                         "validTo": "2025-12-02T17:18:11"
                     }
                 ],
                 "broker": "46V000000000015S",
                 "messageType": "TEST",
                 "senders": [
                     ""
                 ],
                 "sendersAll": true,
                 "type": "INDIRECT",
                 "validFrom": "2025-06-26T15:00:00"
             }
         */

  }

  private static void filterHistory() {
    for (int i = 0; i < filterHistory.size(); i++) {
      System.out.println((i + 1) + ": " + filterHistory.get(i));
    }
  }

  // Other methods - not directly related to the main method

  private static void showSettings() {
    String sortSetting = "Not specified";
    if (componentFilter.getSortAscending() > 0) {
      sortSetting = "Ascending by column " + componentFilter.getSortAscending();
    } else if (componentFilter.getSortDescending() > 0) {
      sortSetting = "Descending by column " + componentFilter.getSortDescending();
    }
    String searchStringFilter = componentFilter.getSearchString() == null ? "<ALL>" : prettyprinted(componentFilter.getSearchString());
    String versionFilter = componentFilter.getVersion() == null ? "<ALL>" : componentFilter.getVersionOperator() + componentFilter.getVersion();
    String validToFilter = componentFilter.getValidTo() == null ? "<ALL>" : componentFilter.getValidToOperator() + componentFilter.getValidTo();
    String scopeFilter = componentFilter.getScope() == null || componentFilter.getScope().equals("all") ? "<ALL>" : componentFilter.getScope();
    String mpCountFilter = componentFilter.getMpCount() == -1 ? "<ALL>" : componentFilter.getMpCount() + "";

    String statFilter = messagePathFilter.getStatExplained();
    String codeFilter = messagePathFilter.getCode() == null ? "<ALL>" : messagePathFilter.getCode();
    String brokerFilter = messagePathFilter.getBroker() == null ? "<ALL>" : messagePathFilter.getBroker();
    String messageTypeFilter = messagePathFilter.getMessageType() == null ? "<ALL>" : messagePathFilter.getMessageType();


    System.out.println("Object    Type   Operation        Cmd       Reset Comp      : Setting");
    System.out.println("---------------------------------------------------------------------");
    System.out.println("Comp      Sort   Column-sort      (asc/dsc) ()    N/A       : " + sortSetting);
    System.out.println("Comp      Filter Search (regexp)  (sch)     ()    (match)   : " + searchStringFilter);
    System.out.println("Comp      Filter Version          (ver)     ()    (>,<,=)   : " + versionFilter);
    System.out.println("Comp      Filter ValidTo          (val)     ()    (>,<,=)   : " + validToFilter);
    System.out.println("Comp      Filter Scope            (scp)     ()    (type)    : " + scopeFilter);
    System.out.println("Comp      Filter MsgPath-Count    (mpc)     ()    (exact)   : " + mpCountFilter);
    System.out.println("MsgPath   Filter Stat             (stat)    ()    (a,y,n,?) : " + statFilter);
    System.out.println("MsgPath   Filter Code             (code)    ()    (match)   : " + codeFilter);
    System.out.println("MsgPath   Filter Broker           (broker)  ()    (match)   : " + brokerFilter);
    System.out.println("MsgPath   Filter MessageType      (type)    ()    (match)   : " + messageTypeFilter);
  }

  private static String prettyprinted(String searchString) {
    return searchString.startsWith("!") ? searchString.substring(1) + " (negated:true)" : searchString + " (negated:false)";
  }


  // Interactive help
  private static void help() {
    System.out.println(" MISC:");
    System.out.println("   h            - help");
    System.out.println("   x            - exit");
    System.out.println("   e<on|off>    - off: will output only the most necessary information, useful for script-mode");
    System.out.println(" LISTINGS:");
    System.out.println("   v            - view components - the #MPs column (6) will depend on the message path filters specified");
    System.out.println("   m            - view messagepaths for all components listed with v-command (depends on the component filters specified)");
    System.out.println("   f            - show filter (search) history");
    System.out.println("   y            - show filter/sort settings");
    System.out.println(" COMPONENT FILTERS:");
    System.out.println("   sch<string>  - match lines with this regexp/string. Prefixing with ! will negate search.");
    System.out.println("                  example of searches:");
    System.out.println("                    test            - match rows containing 'test'");
    System.out.println("                    !4.12|4.11      - match rows NOT containing '4.12' nor '4.11'");
    System.out.println("                    4.12.*Stat      - match rows containing '4.12<zero-or-more-char>Stat'");
    System.out.println("   ver<op><ver> - list versions greater/lower/equal to <ver>. <op> can be '>','<' or '='. Drop <op><ver> to list all");
    System.out.println("   val<op><tms> - list validTo greater/lower/equal to <tms>. <op> can be '>','<' or '='. Drop <op><tms> to list all");
    System.out.println("                  example of version-filters:");
    System.out.println("                    >2025");
    System.out.println("                    <2025-01-01");
    System.out.println("                    >2025-01-01T18:00");
    System.out.println("                    =null");
    System.out.println("   scp<type>    - <type> can be 'loc' (for 'local endpoint'), 'rem' (for 'remote endpoint') or 'all' (for both). List all endpoints of this type");
    System.out.println("   mpc<number>  - <number> is the exact number of message path count to filter for. If not set (-1), match all message path counts");
    System.out.println("   asc<number>  - ascending sort by column 1-code 2-type 3-version 4-validTo 5-organization 6-email. Remove sort by setting to 0.");
    System.out.println("   dsc<number>  - descending sort by column 1-code 2-type 3-version 4-validTo 5-organization 6-email. Remove sort by setting to 0.");
    System.out.println(" MESSAGE-PATH FILTERS:");
    System.out.println("   stat<filter> - The 4 columns 'valid', 'connected', 'local' and 'active' is filtered based on the <filter> which is always 4 character long.");
    System.out.println("                  Each character represents one column: 'a' = 'all', 'y' = 'yes/true', 'n' = 'no/false' and '?' = 'unknown'.");
    System.out.println("                  The ? option only makes sense to use for connection-status, since sometimes the status is unknown.");
    System.out.println("                  example of filters: ");
    System.out.println("                      'aaaa' (show all)");
    System.out.println("                      'yayn' (valid paths, connected & disconnected, only local, only inactive)");
    System.out.println("                      'yany' (valid paths, connected & disconnected, only central, only active)");
    System.out.println("                      'a?aa' (show paths with unknown connected status");
    System.out.println("   code<code>   - match endpoint codes for the messagepath. Same logic/syntax as for the sch/search for component filters");
    System.out.println("   broker<code> - match broker codes for the messagepath. Same logic/syntax as for the sch/search for component filters");
    System.out.println("   type<type>   - match messagepath types for the messagepath. Same logic/syntax as for the sch/search for component filters");
    System.out.println(" ACTIONS - TAKE CARE!:");
    System.out.println("   a<number>    - accept request with this number. Accept ALL requests in the printed list if number is -1");
  }

  // Command line help
  private static void usage() {
    System.out.println("CDShell (CDS) " + VERSION + " is made to browse and manage ECP components. Some use cases are listed below, ");
    System.out.println("with command sequence to accomplish it");
    System.out.println();
    System.out.println("1) Find contact email addresses for all endpoints with too old ECP-version");
    System.out.println("   >scploc");
    System.out.println("   >ver<4.14");
    System.out.println("   >v\n");

    System.out.println("2) Find endpoints without active messagepaths");
    System.out.println("   >scploc");
    System.out.println("   >stataaay");
    System.out.println("   >mpc0");
    System.out.println("   >v\n");

    System.out.println("3) Find local endpoints without message path to specific broker");
    System.out.println("   >scploc");
    System.out.println("   >broker50V000000000112U");
    System.out.println("   >mpc0");
    System.out.println("   >v\n");

    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar CDS [OPTIONS] CD-URL\n");
    System.out.println("\n\nThe options and arguments:");
    System.out.println(" OPTIONS       : ");
    System.out.println("                 -f <commands-file> : each line in the file specifies an interactive command.");
    System.out.println("                 -c <commands>      : a semi-colon separated list of interactive commands. Ignored if -f is specified");
    System.out.println(" CD-URL        : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("\nExample 1:  Start CDS in interactive mode");
    System.out.println("\tjava -jar ekit.jar CDS https://admin:password@localhost:8443");
  }
}
