package no.statnett.ecp.cds.actions;

import no.statnett.ecp.cds.Config;
import no.statnett.ecp.cds.state.Component;
import no.statnett.ecp.cds.state.ComponentFilter;
import no.statnett.ecp.cds.state.MessagePathFilter;
import no.statnett.ecp.utils.Const;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;


public class ListComponents {

    public static class Print {
        public static void printToConsole(List<Component> sortedCList, boolean echo) {
            if (echo) {
                String codeHd = "Code (1)";
                String typeHd = "Type (2)";
                String versionHd = "Version (3)";
                String validToHd = "ValidTo (4)";
                String syncTmsHd = "SyncStatus (5)";
                String mpCntHd = "#MPs (6)";
                String orgHd = "Organization (7)";
                String emailHd = "Email (8)";

                int codeWidth = Math.max(sortedCList.stream().mapToInt(c -> c.getCode().length()).max().orElse(0), codeHd.length());
                int typeWidth = Math.max(sortedCList.stream().mapToInt(c -> c.getType().length()).max().orElse(0), typeHd.length());
                int versionWidth = Math.max(sortedCList.stream().mapToInt(c -> ("" + c.getVersion()).length()).max().orElse(0), versionHd.length());
                int validToWidth = Math.max(sortedCList.stream().mapToInt(c -> ("" + c.getValidTo()).length()).max().orElse(0), validToHd.length());
                int syncWidth = 24;
                int mpCntWidth = mpCntHd.length();
                int orgWidth = Math.max(sortedCList.stream().mapToInt(c -> (c.getOrg()).length()).max().orElse(0), orgHd.length());
                int emailWidth = Math.max(sortedCList.stream().mapToInt(c -> (c.getEmail()).length()).max().orElse(0), emailHd.length());
                int totalWidth = codeWidth + typeWidth + versionWidth + validToWidth + syncWidth + mpCntWidth + orgWidth + emailWidth + 4 + 8 * 3;

                System.out.println(
                        String.format(
                                "%-4s | %-" + codeWidth + "s | %-" + typeWidth + "s | %" + versionWidth + "s | %" + validToWidth +
                                "s | %-" + syncWidth + "s | %-" + mpCntWidth + "s | %-" + orgWidth + "s | %-" + emailWidth + "s",
                                "#", codeHd, typeHd, versionHd, validToHd, syncTmsHd, mpCntHd, orgHd, emailHd) +
                                " (" + LocalDateTime.now().format(Const.localTmsSec) + ")");
                System.out.println("-".repeat(totalWidth));

                int i = 1;
                for (Component c : sortedCList) {
                    String sb = String.format("%-4d |", i) +
                            String.format(" %-" + codeWidth + "s |", c.getCode()) +
                            String.format(" %-" + typeWidth + "s |", c.getType()) +
                            String.format(" %" + versionWidth + "s |", c.getVersion()) +
                            String.format(" %" + validToWidth + "s |", c.getValidTo()) +
                            String.format(" %" + syncWidth + "s |", c.getSync()) +
                            String.format(" %-" + mpCntWidth + "s |", c.getMpCount()) +
                            String.format(" %-" + orgWidth + "s |", c.getOrg()) +
                            String.format(" %-" + emailWidth + "s", c.getEmail());
                    System.out.println(sb);
                    i++;
                }
            }

        }
    }

    public static class Retrieve {

        public static List<Component> retrieveAndFilter(Config config, ComponentFilter componentFilter, MessagePathFilter messagePathFilter, String cdCode) throws NoSuchAlgorithmException, IOException, KeyManagementException {
            String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
            List<Component> cList = retrieve(protocol, config.get("host"), config.get("port"), config.get("user"), config.get("password"), cdCode); // Find the main host first
            List<Component> cListWMPath = ListMessagePaths.Retrieve.retrieveAndFilter(cList, config, messagePathFilter);
            List<Component> filteredBySearchString = filterBySearchString(cListWMPath, componentFilter.getSearchString());
            List<Component> filteredByVersion = filterByVersion(filteredBySearchString, componentFilter.getVersion(), componentFilter.getVersionOperator());
            List<Component> filteredByValidTo = filterByValidTo(filteredByVersion, componentFilter.getValidTo(), componentFilter.getValidToOperator());
            List<Component> filteredByScope = filterByScope(filteredByValidTo, componentFilter.getScope(), cdCode);
            List<Component> filteredByMpCount = filterByMpCount(filteredByScope, componentFilter.getMpCount());
            List<Component> sortedCList = sortComponents(filteredByMpCount, componentFilter.getSortAscending(), componentFilter.getSortDescending());
            List<Component> filteredForEndpoints = filterByEndpoints(sortedCList, componentFilter.getEndpoints());

            return filteredForEndpoints;
        }

        private static List<Component> retrieve(String prot, String host, String port, String user, String pw, String cdCode) throws IOException, NoSuchAlgorithmException, KeyManagementException {
            URL url = new URL(prot, host, Integer.parseInt(port), "/ECP_MODULE/components?count=1000&status=ACTIVE");
            String response = getECPHTTResponse("GET", url, user, pw, null);
            if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
                return null;
            }
            JSONObject jsonObject = new JSONObject(response);
            // JsonObject:
            // {
            //     "data": [
            //         {
            //             "applied": false,
            //             "canBeRevoked": false,
            //             "code": "50V-SIRAKVINAATN",
            //             "company": {
            //                 "contactEmail": "finnharald.skille@sirakvina.no",
            //                 "contactPerson": "Finn Harald Skille",
            //                 "contactPhone": "004795964192",
            //                 "organization": "Sira-Kvina Kraftselskap"
            //             },
            //             "componentDirectory": "50V000000000111W",
            //             "componentStatus": null,
            //             "created": "2023-02-16T15:57:17.622",
            //             "envName": null,
            //             "id": "50V-SIRAKVINAATN",
            //             "implementationVersion": "4.12.0.1871",
            //             "isNotPushed": true,
            //             "lastModification": "2024-09-20T15:49:52.585",
            //             "location": "LOCAL",
            //             "networks": [
            //                 "DefaultNetwork"
            //             ],
            //             "projectName": null,
            //             "type": "ENDPOINT",
            //             "url": [
            //             ],
            //             "validTo": "2025-08-14T09:15:03"
            //         },
            //         {
            //         ..... X other objects like the one above
            //         }
            //     ]
            //     "total": 221
            // }


            List<Component> components = new ArrayList<>();
            JSONArray dataArray = jsonObject.getJSONArray("data");
            for (int i = 0; i < dataArray.length(); i++) {
                JSONObject componentObject = dataArray.getJSONObject(i);
                String code = componentObject.getString("code");
                JSONObject companyObject = componentObject.getJSONObject("company");
                String email = companyObject.getString("contactEmail");
                String name = companyObject.getString("contactPerson");
                String phone = companyObject.getString("contactPhone");
                String org = companyObject.getString("organization");
                String created = componentObject.isNull("created") ? null : componentObject.getString("created");
                String version = componentObject.isNull("implementationVersion") ? null : componentObject.getString("implementationVersion");
                String lastModification = componentObject.isNull("lastModification") ? null : componentObject.getString("lastModification");
                String type = componentObject.getString("type");
                if (type.equals("COMPONENT_DIRECTORY")) {
                    type = "DIRECTORY";
                }
                String validTo = componentObject.isNull("validTo") ? null : componentObject.getString("validTo");
                String cdCodeComponent = componentObject.getString("componentDirectory");
                Component component = new Component(code, email, name, phone, org, created, version, lastModification, type, validTo, cdCodeComponent);
                components.add(component);
                if (component.getType().equals("ENDPOINT") && cdCode.equals(cdCodeComponent) && validTo != null) {
                    retrieveStatsPrComp(prot, host, port, user, pw, code, component);
                }
            }
            return components;
        }

        private static void retrieveStatsPrComp(String prot, String host, String port, String user, String pw, String code, Component component) throws NoSuchAlgorithmException, KeyManagementException, IOException {
            URL url2 = new URL(prot, host, Integer.parseInt(port), "/ECP_MODULE/components/" + code + "/statistics");
            String response = getECPHTTResponse("GET", url2, user, pw, null);
            if (!response.startsWith("{"))
              return;
            JSONObject jsonObject = new JSONObject(response);
            /* jsonObject:
            {
                "componentCode": "50V-AGDERENERAT5",
                "lastSynchronizedTime": "2025-06-26T12:22:34.1509427",
                "lastSynchronizationSucceed": true,
                "syncStatus": "SUCCESS",
                "sentMessages": 356624,
                "receivedMessages": 212648,
                "waitingToDeliverMessages": 0,
                "waitingToReceiveMessages": 0,
                "remoteComponentStatistics": [
                    {
                        "componentCode": "44V000000000018F",
                        "connected": true,
                        "messagesIn": 0,
                        "lastOperationIn": null,
                        "messagesOut": 5,
                        "lastOperationOut": "2025-01-24T10:56:21.5031283"
                    },
                    {
                        "componentCode": "50V000000000112U",
                        "connected": true,
                        "messagesIn": 247497,
                        "lastOperationIn": "2025-06-26T14:22:45.3804763",
                        "messagesOut": 413308,
                        "lastOperationOut": "2025-06-26T14:22:46.2664377"
                    }
                    [ more brokers could be listed, almost always at least one broker, unless no message path has
                     ever been created ]
                ],
                "lastUpdate": "2025-06-26T12:23:15.218809915"
            }
            */

            Object timestamp = jsonObject.get("lastSynchronizedTime");
            if (timestamp != null && timestamp instanceof String) {
                // timestamp is local, example:  "2025-06-26T12:22:34.1509427"
                // convert to UTC and set string to the format YYYY-MM-DDTHH:mm:SSZ
                component.setSyncTimestamp(LocalDateTime.parse(timestamp.toString()).format(Const.localTmsSec));
            } else {
                component.setSyncTimestamp(null);
            }
            String syncStatus = jsonObject.getString("syncStatus");
            if (syncStatus.equals("SUCCESS")) {
                component.setSyncStatus("OK");
            } else {
                component.setSyncStatus(syncStatus.substring(0, 2));
            }
            JSONArray remoteComponentStatistics = jsonObject.getJSONArray("remoteComponentStatistics");
            for (int i = 0; i < remoteComponentStatistics.length(); i++) {
                JSONObject remoteComponent = remoteComponentStatistics.getJSONObject(i);
                String brokerCode = remoteComponent.getString("componentCode");
                boolean connected = remoteComponent.getBoolean("connected");
                component.getConnectedMap().put(brokerCode, connected);
            }
        }


        private static List<Component> filterBySearchString(List<Component> CList, String matchStr) {
            if (matchStr == null || matchStr.trim().isEmpty())
                return CList;
            List<Component> filteredCList = new ArrayList<>();
            boolean positive = true;
            if (matchStr.startsWith("!")) {
                positive = false;
                matchStr = matchStr.substring(1);
            }
            Pattern pattern = Pattern.compile(matchStr.toLowerCase());
            for (Component c : CList) {
                Matcher matcher = pattern.matcher(c.toString().toLowerCase());
                if (matcher.find()) {
                    if (positive)
                        filteredCList.add(c);
                } else {
                    if (!positive)
                        filteredCList.add(c);
                }
            }
            return filteredCList;
        }

        private static List<Component> filterByVersion(List<Component> CList, String version, String versionOperator) {
            if (version == null || version.trim().isEmpty())
                return CList;
            List<Component> filteredCList = new ArrayList<>();
            for (Component c : CList) {
                // versionOperator can be '>' or '<' or '=' or '!'
                // version can be <number>[.<number>]* or null
                // Example:
                //   null
                //   4
                //   3.1
                //   4.149.2
                //   1.12.239.3
                String operator = operator(c.getVersion(), version);
                if (operator.equals(versionOperator)) {
                    filteredCList.add(c);
                }
            }
            return filteredCList;
        }

        private static String operator(String cVersion, String version) {
            if (cVersion == null && version == null)
                return "=";
            if (cVersion == null)
                return "<";
            if (version == null)
                return ">";
            int result = new VersionComparator().compare(cVersion, version);
            return result > 0 ? ">" : (result < 0 ? "<" : "=");
        }

        public static List<Component> filterByValidTo(List<Component> CList, String validTo, String validToOperator) {
            if (validTo == null || validTo.trim().isEmpty())
                return CList;
            List<Component> filteredCList = new ArrayList<>();
            for (Component c : CList) {
                String cValidTo = c.getValidTo();
                if (cValidTo == null) {
                    if (validTo.equals("null") && validToOperator.equals("=")) {
                        filteredCList.add(c);
                    } else if (!validTo.equals("null") && validToOperator.equals("<")) {
                        filteredCList.add(c);
                    }
                } else if (validTo.equals("null")) {
                    if (validToOperator.equals(">")) {
                        filteredCList.add(c);
                    }
                } else {
                    int compareResult = cValidTo.compareTo(validTo);
                    if (validToOperator.equals("=") && compareResult == 0) {
                        filteredCList.add(c);
                    } else if (validToOperator.equals(">") && compareResult > 0) {
                        filteredCList.add(c);
                    } else if (validToOperator.equals("<") && compareResult < 0) {
                        filteredCList.add(c);
                    }
                }
            }
            return filteredCList;
        }

        private static List<Component> filterByScope(List<Component> CList, String scope, String cdCode) {
            if (scope == null || scope.trim().isEmpty() || scope.equals("all"))
                return CList;
            List<Component> filteredCList = new ArrayList<>();
            for (Component c : CList) {
                if (c.getCdCode().equals(cdCode) && scope.equals("loc")) {
                    filteredCList.add(c);
                } else if (!c.getCdCode().equals(cdCode) && scope.equals("rem")) {
                    filteredCList.add(c);
                }
            }
            return filteredCList;
        }

        private static List<Component> filterByMpCount(List<Component> CList, int mpCount) {
            if (mpCount == -1)
                return CList;
            List<Component> filteredCList = new ArrayList<>();
            for (Component c : CList) {
                if (c.getMpCount() == mpCount) {
                    filteredCList.add(c);
                }
            }
            return filteredCList;
        }


        private static List<Component> filterByEndpoints(List<Component> CList, List<String> endpoints) {
          List<Component> filteredCList = new ArrayList<>();
          if (endpoints != null && !endpoints.isEmpty()) {
            for (Component c : CList) {
              if (endpoints.contains(c.getCode())) {
                filteredCList.add(c);
              }
            }
          } else {
            filteredCList = CList;
          }
          return filteredCList;
        }

        private static List<Component> sortComponents(List<Component> cList, int sortAscending, int sortDescending) {
            if (sortAscending > 0) {
                switch (sortAscending) {
                    case 1:
                        cList.sort(Comparator.comparing(Component::getCode));
                        break;
                    case 2:
                        cList.sort(Comparator.comparing(Component::getType));
                        break;
                    case 3:
                        cList.sort(Comparator.comparing(Component::getVersion, Comparator.nullsFirst(new VersionComparator())));
                        break;
                    case 4:
                        // ValidTo can sometimes be null
                        cList.sort(Comparator.comparing(Component::getValidTo, Comparator.nullsFirst(Comparator.naturalOrder())));
                        break;
                    case 5:
                        cList.sort(Comparator.comparing(Component::getSync));
                        break;
                    case 6:
                        cList.sort(Comparator.comparingInt(Component::getMpCount));
                        break;
                    case 7:
                        cList.sort(Comparator.comparing(Component::getOrg));
                        break;
                    case 8:
                        cList.sort(Comparator.comparing(Component::getEmail));
                        break;
                }
            } else if (sortDescending > 0) {
                switch (sortDescending) {
                    case 1:
                        cList.sort(Comparator.comparing(Component::getCode).reversed());
                        break;
                    case 2:
                        cList.sort(Comparator.comparing(Component::getType).reversed());
                        break;
                    case 3:
                        cList.sort(Comparator.comparing(Component::getVersion, Comparator.nullsFirst(new VersionComparator())).reversed());
                        break;
                    case 4:
                        // ValidTo can sometimes be null
                        cList.sort(Comparator.comparing(Component::getValidTo, Comparator.nullsFirst(Comparator.naturalOrder())).reversed());
                        break;
                    case 5:
                        cList.sort(Comparator.comparing(Component::getSync).reversed());
                        break;
                    case 6:
                        cList.sort(Comparator.comparingInt(Component::getMpCount).reversed());
                        break;
                    case 7:
                        cList.sort(Comparator.comparing(Component::getOrg).reversed());
                        break;
                    case 8:
                        cList.sort(Comparator.comparing(Component::getEmail).reversed());
                        break;
                }
            }
            return cList;
        }
    }

    public static class VersionComparator implements Comparator<String> {
        @Override
        // Compare 2 version numbers, and add to the list if the first is greater than the second
        // The comparion can not be lexicographic, since the version numbers can be of different length
        // Example:
        // 1.12.239.3 > 1.12.239
        // 1.12.239.3 > 1.9.239.3
        public int compare(String o1, String o2) {
            String[] cParts = o1.split("\\.");
            String[] vParts = o2.split("\\.");
            int maxLength = Math.max(cParts.length, vParts.length);
            for (int i = 0; i < maxLength; i++) {
                int cNum = i < cParts.length ? Integer.parseInt(cParts[i]) : 0;
                int vNum = i < vParts.length ? Integer.parseInt(vParts[i]) : 0;
                if (cNum > vNum) {
                    return 1;
                }
                if (cNum < vNum) {
                    return -1;
                }
            }
            return 0;
        }
    }
}
