package no.statnett.ecp.cds.actions;

import no.statnett.ecp.cds.Config;
import no.statnett.ecp.cds.state.Component;
import no.statnett.ecp.cds.state.MessagePathFilter;
import no.statnett.ecp.cds.state.MessagePathInfo;
import no.statnett.ecp.utils.Const;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

public class ListMessagePaths {

    public static class Print {
        public static void printMP(List<Component> components, boolean echo) {
            int i = 1;

            List<MessagePathInfo> mpiList = components.stream().flatMap(c -> c.getMpiMap().values().stream()).collect(Collectors.toList());
            if (echo) {
                String codeHd = "Code (1)";
                String validHd = "Valid (2)";
                String connectedHd = "Connected (3)";
                String localHd = "Local (4)";
                String activeHd = "Active (5)";
                String brokerHd = "Broker (6)";
                String messageTypeHd = "Message type (7)";
                String sendersHd = "Senders (8)";

                int codeWidth = Math.max(components.stream().mapToInt(m -> m.getCode().length()).max().orElse(0), codeHd.length());
                int validWidth = validHd.length();
                int connectedWidth = connectedHd.length();
                int localWidth = localHd.length();
                int activeWidth = activeHd.length();
                int brokerWidth = Math.max(mpiList.stream().mapToInt(m -> m.getBroker().length()).max().orElse(0), brokerHd.length());
                int msgTypeWidth = Math.max(mpiList.stream().mapToInt(m -> m.getMessageType().length()).max().orElse(0), messageTypeHd.length());
                int sendersWidth = Math.max(mpiList.stream().mapToInt(m -> m.getSenders().length()).max().orElse(0), sendersHd.length());
                int totalWidth = codeWidth + validWidth + connectedWidth + localWidth + activeWidth + brokerWidth + msgTypeWidth + sendersWidth + 4 + 8 * 3;

                System.out.println(String.format("%-4s | %-" + codeWidth + "s | %-" + validWidth + "s | %-" + connectedWidth + "s | %-" + localWidth + "s | %-" + activeWidth + "s | %-" + brokerWidth + "s | %-" + msgTypeWidth + "s | %-" + sendersWidth + "s", "#", codeHd, validHd, connectedHd, localHd, activeHd, brokerHd, messageTypeHd, sendersHd) + " (" + LocalDateTime.now().format(Const.localTmsSec) + ")");
                System.out.println("-".repeat(totalWidth));

                for (Component c : components) {
                    for (MessagePathInfo mpi : c.getMpiMap().values()) {
                        if (!mpi.isSkip()) {
                            String validStr = mpi.isValid() ? "Y" : "N";
                            String connectedStr = mpi.isConnected() == null ? "?" : mpi.isConnected() ? "Y" : "N";
                            String localStr = mpi.isLocal() ? "Y" : "N";
                            String activeStr = mpi.isActive() ? "Y" : "N";
                            String sb = String.format("%-4d |", i) +
                                    String.format(" %-" + codeWidth + "s |", c.getCode()) +
                                    String.format(" %-" + validWidth + "s |", validStr) +
                                    String.format(" %-" + connectedWidth + "s |", connectedStr) +
                                    String.format(" %-" + localWidth + "s |", localStr) +
                                    String.format(" %-" + activeWidth + "s |", activeStr) +
                                    String.format(" %-" + brokerWidth + "s |", mpi.getBroker()) +
                                    String.format(" %-" + msgTypeWidth + "s |", mpi.getMessageType()) +
                                    String.format(" %-" + sendersWidth + "s", mpi.getSenders());
                            System.out.println(sb);
                            i++;
                        }
                    }
                }
            }
        }
    }

    public static class Retrieve {
        public static List<Component> retrieveAndFilter(List<Component> cList, Config config, MessagePathFilter messagePathFilter) throws NoSuchAlgorithmException, IOException, KeyManagementException {
            retrievePaths(cList, config, "/ECP_MODULE/paths?count=9999&page=0", true);
            retrievePaths(cList, config, "/ECP_MODULE/directory/paths?count=9999&page=0", false);

            // Update all message path (determine active mp and use filters)
            updateMPdata(cList, messagePathFilter);

            return cList;
        }

        private static List<Component> retrievePaths(List<Component> cList, Config config, String mpURLPath, boolean local) throws NoSuchAlgorithmException, KeyManagementException, IOException {
            String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
            URL mpURL = new URL(protocol, config.get("host"), Integer.parseInt(config.get("port")), mpURLPath);
            String response = getECPHTTResponse("GET", mpURL, config.get("user"), config.get("password"), null);
            if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
                System.out.println("Error occured when trying to retrieve url: " + mpURL);
                System.out.println("Response:\n" + response);
            }

            JSONObject jsonObject = new JSONObject(response);

            /*
            {
              "total": 650,
              "data": [
                  {
                      "validFrom": "2024-08-01T01:00:00",
                      "validTo": null,
                      "messageType": "*",
                      "type": "INDIRECT",
                      "id": 0,
                      "broker": "50V000000000112U",
                      "status": null,
                      "receiver": "50V-SIRAKVINAATN",
                      "senders": [
                          "All"
                      ]
                  },
                  ...
                  ]
               }
               */


            JSONArray dataArray = jsonObject.getJSONArray("data");

            // Iterate over the data array
            for (int i = 0; i < dataArray.length(); i++) {
                JSONObject dataObject = dataArray.getJSONObject(i);
                Integer id = dataObject.getInt("id");
                String broker = dataObject.getString("broker");
                String messageType = dataObject.getString("messageType");
                String receiver = dataObject.getString("receiver");
                String validFrom = dataObject.isNull("validFrom") ? null : dataObject.getString("validFrom");
                String validTo = dataObject.isNull("validTo") ? null : dataObject.getString("validTo");


                Component component = cList.stream().filter(c -> c.getCode().equals(receiver)).findFirst().orElse(null);

                if (component != null) {
                    JSONArray sendersArray = dataObject.getJSONArray("senders");
                    List<String> senders = new ArrayList<>();
                    for (int j = 0; j < sendersArray.length(); j++) {
                        senders.add(sendersArray.getString(j));
                    }
                    Map<Integer, MessagePathInfo> mpiMap = component.getMpiMap();
                    if (!local) {
                        // Central message paths will be sorted after local message paths, which is what we want in the mpiMap - because they have less priority.
                        // We assume that we will never have more than 100000 local message path and that ids are set sequentially starting from 1.
                        id = id * 100000;
                    }
                    MessagePathInfo mpi = new MessagePathInfo(id, broker, messageType, senders, validFrom, validTo, local);
                    mpiMap.put(id, mpi);
                }
            }
            return cList;
        }

        private static void updateMPdata(List<Component> components, MessagePathFilter messagePathFilter) {
            for (Component c : components) {
                Map<Integer, MessagePathInfo> mpiMap = c.getMpiMap();
                boolean activeStarMP = false;
                for (MessagePathInfo mpi : mpiMap.values()) {
                    mpi.setConnected(c.getConnectedMap().get(mpi.getBroker()));
                    if (!activeStarMP && mpi.getMessageType().equals("*") && mpi.isValid() && mpi.isConnected() != null && mpi.isConnected()) {
                        mpi.setActive(true);
                        activeStarMP = true;
                    }
                }

                Set<String> messageTypeActiveSet = new HashSet<>();
                int mpCount = 0;
                for (MessagePathInfo mpi : mpiMap.values()) {
                    String validStr = mpi.isValid() ? "Y" : "N";
                    String connectedStr = mpi.isConnected() == null ? "?" : mpi.isConnected() ? "Y" : "N";
                    String localStr = mpi.isLocal() ? "Y" : "N";
                    if (!activeStarMP && !messageTypeActiveSet.contains(mpi.getMessageType()) && !mpi.getMessageType().equals("*") && validStr.equals("Y") && connectedStr.equals("Y")) {
                        mpi.setActive(true);
                        messageTypeActiveSet.add(mpi.getMessageType());
                    }
                    String activeStr = mpi.isActive() ? "Y" : "N";
                    boolean skip = skip(messagePathFilter, c, mpi, validStr, connectedStr, localStr, activeStr);
                    mpi.setSkip(skip);
                    mpCount += skip ? 0 : 1;
                }
                c.setMpCount(mpCount);
            }
        }

        private static boolean skip(MessagePathFilter messagePathFilter, Component c, MessagePathInfo mpi, String validStr, String connectedStr, String localStr, String activeStr) {
            // Apply messagepath-filter
            if (messagePathFilter.getStat() != null) {
                String stat = messagePathFilter.getStat().toUpperCase();
                if (!stat.substring(0, 1).equals(validStr) && stat.charAt(0) != 'A') {
                    return true;
                }
                if (!stat.substring(1, 2).equals(connectedStr) && stat.charAt(1) != 'A') {
                    return true;
                }
                if (!stat.substring(2, 3).equals(localStr) && stat.charAt(2) != 'A') {
                    return true;
                }
                if (!stat.substring(3, 4).equals(activeStr) && stat.charAt(3) != 'A') {
                    return true;
                }
            }
            if (messagePathFilter.getCode() != null) {
                if (skipRegexp(messagePathFilter.getCode(), c.getCode())) return true;
            }

            if (messagePathFilter.getBroker() != null) {
                if (skipRegexp(messagePathFilter.getBroker(), mpi.getBroker())) return true;
            }
            if (messagePathFilter.getMessageType() != null) {
                if (skipRegexp(messagePathFilter.getMessageType(), mpi.getMessageType())) return true;
            }
            return false;
        }

        private static boolean skipRegexp(String filterValue, String compValue) {
            boolean negate = false;
            if (filterValue.startsWith("!") && filterValue.length() > 1) {
                negate = true;
                filterValue = filterValue.substring(1);
            }
            Pattern pattern = Pattern.compile(filterValue);
            Matcher matcher = pattern.matcher(compValue);
            boolean skip = false;
            if (!matcher.find()) {
                skip = true;
            }
            return negate ? !skip : skip;
        }


    }
}
