package no.statnett.ecp.div;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.*;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

/**
 * This class is only used for Statnett (or other TSOs) to check whether BSPs have specified the appropriate message paths.
 * No BSP would be interested in this, they can very easily check their own message path in the ECP GUI.
 */
public class ComponentListing {
    public static final String VERSION = "v1.0.2";
    private static URL baseURL;
    private static String user;
    private static String pw;


    private static void parseMandatoryArg(List<String> mandatoryArgs) throws MalformedURLException {
        if (mandatoryArgs.isEmpty()) {
            usage();
            System.exit(1);
        }
        baseURL = new URL(mandatoryArgs.get(0));
        if (baseURL.getUserInfo() != null) {
            user = baseURL.getUserInfo().split(":")[0];
            pw = baseURL.getUserInfo().split(":")[1];
        } else {
            System.out.println("The URL must contain user:password");
            System.exit(1);
        }
    }

    public static void main(String[] args) throws NoSuchAlgorithmException, IOException, KeyManagementException {
        List<String> initArgsList = new ArrayList<>(Arrays.asList(args));
        parseMandatoryArg(initArgsList);
        Map<String, Component> components = components();
        System.out.println("Code;Email;Name;Phone;Org;Created;Version;Last Modification;Type;Valid To");
        assert components != null;
        for (String code : components.keySet()) {
            System.out.println(components.get(code));
        }
    }

    private static Map<String, Component> components() throws IOException, NoSuchAlgorithmException, KeyManagementException {
        URL url = new URL(baseURL.getProtocol(), baseURL.getHost(), baseURL.getPort(), "/ECP_MODULE/components?count=1000&status=ACTIVE");
        String response = getECPHTTResponse("GET", url, user, pw, null);
        if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
            return null;
        }
        JSONObject jsonObject = new JSONObject(response);
        // JsonObject:
        // {
        //     "data": [
        //         {
        //             "applied": false,
        //             "canBeRevoked": false,
        //             "code": "50V-SIRAKVINAATN",
        //             "company": {
        //                 "contactEmail": "finnharald.skille@sirakvina.no",
        //                 "contactPerson": "Finn Harald Skille",
        //                 "contactPhone": "004795964192",
        //                 "organization": "Sira-Kvina Kraftselskap"
        //             },
        //             "componentDirectory": "50V000000000111W",
        //             "componentStatus": null,
        //             "created": "2023-02-16T15:57:17.622",
        //             "envName": null,
        //             "id": "50V-SIRAKVINAATN",
        //             "implementationVersion": "4.12.0.1871",
        //             "isNotPushed": true,
        //             "lastModification": "2024-09-20T15:49:52.585",
        //             "location": "LOCAL",
        //             "networks": [
        //                 "DefaultNetwork"
        //             ],
        //             "projectName": null,
        //             "type": "ENDPOINT",
        //             "url": [
        //             ],
        //             "validTo": "2025-08-14T09:15:03"
        //         },
        //         {
        //         ..... 220 other objects like the one above
        //         }
        //     ]
        //     "total": 221
        // }


        Map<String, Component> components = new TreeMap<>();
        JSONArray dataArray = jsonObject.getJSONArray("data");
        for (int i = 0; i < dataArray.length(); i++) {
            JSONObject componentObject = dataArray.getJSONObject(i);
            String code = componentObject.getString("code");
            JSONObject companyObject = componentObject.getJSONObject("company");
            String email = companyObject.getString("contactEmail");
            String name = companyObject.getString("contactPerson");
            String phone = companyObject.getString("contactPhone");
            String org = companyObject.getString("organization");
            String created = componentObject.getString("created");
            String version = componentObject.getString("implementationVersion");
            String lastModification = componentObject.getString("lastModification");
            String type = componentObject.getString("type");
            String validTo = componentObject.isNull("validTo") ?  null : componentObject.getString("validTo");
            Component component = new Component(code, email, name, phone, org, created, version, lastModification, type, validTo);
            components.put(code, component);
        }
        return components;
    }

    private static void usage() {
        System.out.println("ComponentListing (CL) v" + VERSION + " will list component information");
        System.out.println();
        System.out.println("Usage  : java -jar ekit.jar CL <URL>\n");
        System.out.println("\n\nThe arguments:");
        System.out.println("     <URL>         : Specify url (either to an Endpoint or a CD) on this form: http[s]://user:pass@host:port");
        System.out.println();
        System.out.println("\nExample 1:  List components:");
        System.out.println("\tjava -jar ekit.jar CL https://admin:password@remote-host:8443");
    }

    public static class Component {

        private final String code;
        private final String email;
        private final String name;
        private final String phone;
        private final String org;
        private final String created;
        private final String version;
        private final String lastModification;
        private final String type;
        private final String validTo;

        public Component(String code, String email, String name, String phone, String org, String created, String version, String lastModification, String type, String validTo) {
            this.code = code;
            this.email = email;
            this.name = name;
            this.phone = phone;
            this.org = org;
            this.created = created;
            this.version = version;
            this.lastModification = lastModification;
            this.type = type;
            this.validTo = validTo;
        }

        public String toString() {
            return code + ";" + email + ";" + name + ";" + phone + ";" + org + ";" + created + ";" + version + ";" + lastModification + ";" + type + ";" + validTo;
        }

    }

}
