package no.statnett.ecp.div;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

/**
 * This class is only used for Statnett (or other TSOs) to check whether BSPs have specified the appropriate message paths.
 * No BSP would be interested in this, they can very easily check their own message path in the ECP GUI.
 */
public class MessagePathListing {
  public static final String VERSION = "v1.0.4";
  private static List<String> codes;
  private static URL baseURL;
  private static String user;
  private static String pw;


  private static void parseMandatoryArg(List<String> mandatoryArgs) throws MalformedURLException {
    if (mandatoryArgs.size() < 2) {
      usage();
      System.exit(1);
    }
    codes = Arrays.asList(mandatoryArgs.get(0).split(","));
    for (String code : codes) {
      // Each code should be on the form \d\dV[0-9A-Z-]+
      if (!code.matches("\\d\\dV[0-9A-Z-]+")) {
        System.out.println("The code " + code + " is not on the form \\d\\dV[0-9A-Z-]+");
        System.exit(1);
      }
    }
    baseURL = new URL(mandatoryArgs.get(1));
    if (baseURL.getUserInfo() != null) {
      user = baseURL.getUserInfo().split(":")[0];
      pw = baseURL.getUserInfo().split(":")[1];
    } else {
      System.out.println("The URL must contain user:password");
      System.exit(1);
    }
  }

  public static void main(String[] args) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(args));
    parseMandatoryArg(initArgsList);
    Map<String, MessagePaths> messagePaths = messagePaths();

    statistics(messagePaths);
    for (String code : messagePaths.keySet()) {
      System.out.println(code + " : [" + messagePaths.get(code) + "]");
    }
  }

  public static Map<String, MessagePaths> messagePaths() throws NoSuchAlgorithmException, IOException, KeyManagementException {
    // Sort the map according to the list of codes:
    Map<String, MessagePaths> messagePathsMap = new TreeMap<>(Comparator.comparing(codes::indexOf));

    URL mpURL = new URL(baseURL.getProtocol(), baseURL.getHost(), baseURL.getPort(), "/ECP_MODULE/paths?count=9999&page=0");
    String response = getECPHTTResponse("GET", mpURL, user, pw, null);
    if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
      System.out.println("Error occured when trying to retrieve url: " + mpURL);
      System.out.println("Response:\n" + response);
      System.exit(1);
    }

    JSONObject jsonObject = new JSONObject(response);
    JSONArray dataArray = jsonObject.getJSONArray("data");

    // Iterate over the data array
    for (int i = 0; i < dataArray.length(); i++) {
      JSONObject dataObject = dataArray.getJSONObject(i);
      String broker = dataObject.getString("broker");
      String messageType = dataObject.getString("messageType");
      String receiver = dataObject.getString("receiver");
      JSONArray sendersArray = dataObject.getJSONArray("senders");
      if (codes.contains(receiver)) {
        List<String> senders = new ArrayList<>();
        for (int j = 0; j < sendersArray.length(); j++) {
          senders.add(sendersArray.getString(j));
        }
        MessagePathInfo messagePathInfo = new MessagePathInfo(broker, messageType, senders);
        if (!messagePathsMap.containsKey(receiver)) {
          messagePathsMap.put(receiver, new MessagePaths());
        }
        messagePathsMap.get(receiver).getBrokerMap().put(broker, messagePathInfo);
      }
    }
    return messagePathsMap;
  }

  private static void statistics(Map<String, MessagePaths> messagePaths) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    for (String endpointCode : messagePaths.keySet()) {
      URL statURL = new URL(baseURL.getProtocol(), baseURL.getHost(), baseURL.getPort(), "/ECP_MODULE/components/" + endpointCode + "/statistics");
      String response = getECPHTTResponse("GET", statURL, user, pw, null);
      if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
        return;
      }
      JSONObject jsonObject = new JSONObject(response);
      // JsonObject:
      // {"lastSynchronizedTime":"2024-09-30T14:54:35.5098683","waitingToDeliverMessages":0,"componentCode":"50V-SIRAKVINAATN","waitingToReceiveMessages":0,"lastUpdate":"2024-09-30T14:55:15.193690806","lastSynchronizationSucceed":true,"receivedMessages":138,"sentMessages":80,"syncStatus":"SUCCESS","remoteComponentStatistics":[{"connected":true,"messagesOut":0,"componentCode":"44V000000000018F","messagesIn":0,"lastOperationOut":null,"lastOperationIn":null},{"connected":true,"messagesOut":0,"componentCode":"45V000000000058P","messagesIn":0,"lastOperationOut":null,"lastOperationIn":null},{"connected":true,"messagesOut":0,"componentCode":"46V000000000015S","messagesIn":0,"lastOperationOut":null,"lastOperationIn":null},{"connected":true,"messagesOut":80,"componentCode":"50V000000000112U","messagesIn":138,"lastOperationOut":"2024-09-20T17:29:15.4008938","lastOperationIn":"2024-09-27T16:34:00.6788648"}]}
      // Read out list of brokers and their connected status:
      JSONArray remoteComponentStatistics = jsonObject.getJSONArray("remoteComponentStatistics");
      for (int i = 0; i < remoteComponentStatistics.length(); i++) {
        JSONObject remoteComponent = remoteComponentStatistics.getJSONObject(i);
        String brokerCode = remoteComponent.getString("componentCode");
        boolean connected = remoteComponent.getBoolean("connected");
        if (messagePaths.get(endpointCode).getBrokerMap().containsKey(brokerCode))
          messagePaths.get(endpointCode).getBrokerMap().get(brokerCode).setConnected(connected);
      }
    }
  }

  private static void usage() {
    System.out.println("MessagePathListing (MPL) v" + VERSION + " will list message path for certain endpoints.");
    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar MPL <CODE-LIST> <ECP-URL>\n");
    System.out.println("\n\nThe arguments:");
    System.out.println("     <CODE-LIST>       : List of ECP codes for which message paths will be listed (comma-separated, no spaces)");
    System.out.println("     <ECP-URL>         : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("If you specify a CD as the host-argument, you'll also get the connected status for each MP/Broker on each endpoint, if");
    System.out.println("not the connected status will be '?'.");
    System.out.println("\nExample 1:  List message path check towards two endpoint from your endoint (at <remote-host>)");
    System.out.println("\tjava -jar ekit.jar MPL 50V000000000115O,50V-SN-TEST2-AT7 https://admin:password@remote-host:8443");
  }

  public static class MessagePaths {
    Map<String, MessagePathInfo> messagePathPrBroker;

    public MessagePaths() {
      // Sort the map according to the order keys are added:
      messagePathPrBroker = new LinkedHashMap<>();
    }

    public Map<String, MessagePathInfo> getBrokerMap() {
      return messagePathPrBroker;
    }

    public String toString() {
      return messagePathPrBroker.values().stream().map(MessagePathInfo::toString).collect(Collectors.joining(", "));
    }

  }

  public static class MessagePathInfo {
    private String broker;
    private String messageType;
    private String senders;
    private Boolean connected;

    public MessagePathInfo(String broker, String messageType, List<String> senders) {
      this.broker = broker;
      this.messageType = messageType;
      this.senders = senders.stream().collect(Collectors.joining(","));
    }

    public String getBroker() {
      return broker;
    }

    public String getMessageType() {
      return messageType;
    }

    public String getSenders() {
      return senders;
    }

    public boolean isConnected() {
      return connected;
    }

    public void setConnected(boolean connected) {
      this.connected = connected;
    }

    public String toString() {
      // <messagetype> from <sender> to <broker> (connected: <status>)
      String connStatus = connected == null ? " ? " : connected ? "yes" : "no ";
      return messageType + " from " + senders + " to " + broker + " (connected: " + connStatus + ")";
    }
  }
}
