package no.statnett.ecp.edx;

import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.LogOut;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.time.ZonedDateTime;
import java.util.*;

public class Flow {
  // This class will store the traffic (PLAIN MSG/STANDARD MESSAGES) passing through ECP and EDX in ONE direction, divided in
  // "measurements". Each measurement is populated by reading the edx and ecp log, divided by minute.

  // Once all measurements are populated, the calculate-method will be called to make the comparison of most recent measurement.
  // The main idea is that the traffic going through both ECP and EDX should be the same, otherwise there is a problem. There could of course
  // be a minute-shift between the two, but that should be the only difference - and we must be able to detect that (and ignore it).

  private final boolean snd;
  private final Map<Long, Measurement> allMeasurements = new HashMap<>();
  private Measurement main;
  private int diff = 0;

  public Flow(boolean snd) {
    this.snd = snd;
  }

  public void addEcpId(long yyyyMMddHHmm, String ecpId, String logLine) {
    Measurement measurement = getOrCreateMeasurement(yyyyMMddHHmm);
    measurement.addEcpId(ecpId, logLine);
  }

  public void addEdxId(long yyyyMMddHHmm, String edxId, String logLine) {
    Measurement measurement = getOrCreateMeasurement(yyyyMMddHHmm);
    measurement.addEdxId(edxId, logLine);
  }

  public void removeEdxId(String edxMessageId) {
    for (Measurement m : allMeasurements.values()) {
      m.removeEdxId(edxMessageId);
    }
  }

  public void correlate(long yyyyMMddHHmm, String edxMessageId, String ecpMessageId) {
    Measurement measurement = getOrCreateMeasurement(yyyyMMddHHmm);
    measurement.correlate(edxMessageId, ecpMessageId);
  }

  public void addEcpExpired(long yyyyMMddHHmm, String expiration) {
    Measurement measurement = getOrCreateMeasurement(yyyyMMddHHmm);
    measurement.addEcpExpired(expiration);
  }

  public void addNegativeACK(long yyyyMMddHHmm, String negativeACK) {
    Measurement measurement = getOrCreateMeasurement(yyyyMMddHHmm);
    measurement.addNegativeACK(negativeACK);
  }

  private Measurement getOrCreateMeasurement(long yyyyMMddHHmm) {
    Measurement measurement = allMeasurements.get(yyyyMMddHHmm);
    if (measurement == null) {
      measurement = new Measurement(yyyyMMddHHmm, snd);
      allMeasurements.put(yyyyMMddHHmm, measurement);
    }
    return measurement;
  }

  private Measurement getMeasurement(long yyyyMMddHHmm) {
    return allMeasurements.get(yyyyMMddHHmm) == null ? new Measurement(yyyyMMddHHmm, snd) : allMeasurements.get(yyyyMMddHHmm);
  }

  // We try all combinations of the measurements, to see if there is a combindation of minutes
  // which gives 0 or positive diff. If that is not possible, we'll return the diff of the main measurement and
  // admit we have a real problem where either EDX sends more than ECP send or ECP receives more than EDX receives.
  public void findBestDiff(ZonedDateTime nowZDT, int toleranceBandMinute) {

    // Retrieve the measurements for the main measurement, and the measurements in the tolerance band:
    List<Measurement> measurements = new ArrayList<>();
    int expSize = 0;
    int negativeACKSize = 0;
    int completeIntervalMinutes = 1 + 2 * toleranceBandMinute; // 1 minute main measurement + N minute tolerance before/after
    for (int i = 0; i < completeIntervalMinutes; i++) {
      long yyyyMMddHHmm = Long.parseLong(nowZDT.minusMinutes(i + 1).format(Const.tmsNumberMinute));
      Measurement measurement = getMeasurement(yyyyMMddHHmm);
      if (!snd)
        expSize += measurement.getEcpExpired().size();
      else
        negativeACKSize += measurement.getNegativeACK().size();
      measurements.add(measurement);
    }

    // The middle measurement is the main measurement. If tolerance-band is 2, then 0,1 and 3,4 are the tolerance-band measurements
    main = measurements.get(toleranceBandMinute);

    // If mainDiff >= 0, we have no problem
    if (main.diff() >= 0) {
      this.diff = main.diff();
      return;
    }

    // If that is not the case, we try various combinations of measurements find a 0-or-greater-diff.
    // We start with combinations of 2 measurements, then 3, then 4, up to all measurements
    int bestCombinationDiff = Integer.MIN_VALUE;
    for (int length = 2; length <= completeIntervalMinutes; length++) {
      List<Set<Integer>> combinationSets = Combination.getCombinationSets(completeIntervalMinutes, length);
      for (Set<Integer> combination : combinationSets) {
        if (!combination.contains(toleranceBandMinute)) {
          // if a combinationSet does not include the main measurement, we skip it
          continue;
        }
        int combinationDiff = combination.stream().mapToInt(measureIndex -> measurements.get(measureIndex).diff()).sum();
        if (combinationDiff >= bestCombinationDiff) {
          bestCombinationDiff = combinationDiff;
        }

        if (combinationDiff >= 0) {
          // We found a combination which gives a diff of 0 or greater, so we're happy with that
          this.diff = combinationDiff;
          return;
        }
      }
    }

    // At this point we have a negative diff, having tried all combinations. But - there could still be a valid reason for
    // this diff:
    // *  Because of "message expiration". This information is not so easy to correlate to the measurements, but we
    // collect "expiration" messages by the minute. If we have enough such expiration messages in the tolerance band to
    // counter the diff - we'll accept that as a valid reason for the diff. We'll only do this for the rcv-diff, because
    // this is where we expect message expiration to occur.
    // * Because of "NegativeACK". This can happen when an EDX is about to send a message to ECP, but then for some reason
    // cannot process it. Again we do the same as for the expired situation in ECP - we try to counter this against diff
    // we have. However, these negative ACK situation is *really* failed messages, although not in such a way that EDX and
    // ECP have lost communication (which is the worst scenario, worthy of an alarm). This scenario usually happens in TEST
    // when BA send some new ECP-code that doesn't exist etc. So it's more of a WARN situation than an ERROR

    if (!snd && expSize > 0) {
      if (main.diff() + expSize >= 0) {
        this.diff = main.diff() + expSize;
        return;
      } else if (bestCombinationDiff + expSize >= 0) {
        this.diff = bestCombinationDiff + expSize;
        return;
      }
    } else if (snd && negativeACKSize > 0) {
      if (main.diff() + negativeACKSize >= 0) {
        this.diff = main.diff() + negativeACKSize;
        return;
      } else if (bestCombinationDiff + negativeACKSize >= 0) {
        this.diff = bestCombinationDiff + negativeACKSize;
        return;
      }
    }



    // If no combination with diff >= 0 was found, we will use main.diff() as the diff - which will then be negative
    this.diff = main.diff();
  }

  public int diff() {
    return diff;
  }

  public Measurement getMain() {
    return main;
  }

  public void debug(File debugFile) {
    // Print the loglines for EDX and ECP for the main measurement
    try {
      Files.writeString(debugFile.toPath(), "Negative diff: " + diff() + " occurred - see loglines below:\n", StandardOpenOption.APPEND);
      String direction = snd ? "\tSND" : "\tRCV";
      if (snd) { // We have improved the diff-logging for SND-direction - we'll only show EDX SND lines *without* corresponding ECP-messageId
        for (String diffLine : main.getSndDiff()) {
          Files.writeString(debugFile.toPath(), direction + " EDX " + diffLine + "\n", StandardOpenOption.APPEND);
        }
      } else {
        for (String logLine : main.getEcpLogLines())
          Files.writeString(debugFile.toPath(), direction + " ECP " + logLine + "\n", StandardOpenOption.APPEND);
        for (String logLine : main.getEdxLogLines())
          Files.writeString(debugFile.toPath(), direction + " EDX " + logLine + "\n", StandardOpenOption.APPEND);
        for (String expiration : main.getEcpExpired())
          Files.writeString(debugFile.toPath(), direction + " EXP " + expiration + "\n", StandardOpenOption.APPEND);
      }
    } catch (IOException e) {
      System.err.println(LogOut.e() + " Cannot write to debug-file " + debugFile.getAbsolutePath() + ": " + e.getMessage());
    }
  }

}
