package no.statnett.ecp.eft;

import no.statnett.ecp.utils.Options;

import java.io.File;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import static java.time.ZoneOffset.UTC;

public class EDXFailTrash {
  public static final String VERSION = "v0.9.1";

  private static File failDir;
  private static File trashDir;
  private static int hours = 1;

  private static List<String> parseOptions(String[] initialArgs) {
    // No options so far - all args are mandatory
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    hours = Options.parseInt(initArgsList, "-h", 1, 168, 1);
    return initArgsList;
  }

  public static void main(String[] args) {
    List<String> mandatoryArgs = parseOptions(args);
    if (mandatoryArgs.size() != 2) {
      usage();
      System.exit(1);
    } else {
      String failDirStr = mandatoryArgs.get(0);
      String trashDirStr = mandatoryArgs.get(1);
      if (new File(failDirStr).isDirectory()) {
        failDir = Options.checkDirectory(failDirStr, false);
      }
      if (new File(trashDirStr).isDirectory()) {
        trashDir = Options.checkDirectory(trashDirStr, false);
      }
    }
    ZonedDateTime now = ZonedDateTime.now();
    ZonedDateTime endOfPeriod = now.withMinute(0).withSecond(0).withNano(0);
    ZonedDateTime startOfPeriod = endOfPeriod.minusHours(hours);
    List<File> failFiles = scanForFiles(failDir, startOfPeriod, endOfPeriod);
    List<File> trashFiles = scanForFiles(trashDir, startOfPeriod, endOfPeriod);
    printFiles(failFiles, "FAIL");
    printFiles(trashFiles, "TRASH");
  }

  private static List<File> scanForFiles(File directory, ZonedDateTime startOfPeriod, ZonedDateTime endOfPeriod) {
    List<File> files = new ArrayList<>();
    for (File f : Objects.requireNonNull(directory.listFiles())) {
      long modifiedSec = f.lastModified() / 1000;
      if (startOfPeriod.toEpochSecond() <= modifiedSec && modifiedSec < endOfPeriod.toEpochSecond()) {
        files.add(f);
      }
    }
    return files;
  }

  private static void printFiles(List<File> files, String errorType) {
    // Print for each file this:
    // Timestamp of file (in UTC) errorType=<errorType> bSn=<bSn>, code=<code>, msgType=<msgType>, id=<id>
    for (File file : files) {
      String filename = file.getName();
      String[] fields = filename.split("_");
      ZonedDateTime fileZDT = ZonedDateTime.ofInstant(Instant.ofEpochMilli(file.lastModified()), UTC);
      String bSn = fields[0] == null || fields[0].equals("") ? "Unknown," : fields[0] + ",";
      String code = fields[1]+",";
      String msgType = fields[2]+",";
      String id = fields[4];
      String logLine = String.format(fileZDT + " errorType=%-6s bSn=%-20s code=%s msgType=%-35s id=%s", errorType,bSn,code,msgType,id);
      System.out.println(logLine);


    }
  }


  private static void usage() {
    System.out.println("EDXFailTrash (EFT) " + VERSION + " is a simple tool to monitor two folders in which EDX is expected");
    System.out.println("to output files that either FAIL (the fail attribute in edx.yml) or was not routed anywhere and");
    System.out.println("therefore was routed to the default endpoint in edx.yml - AKA as TRASH");
    System.out.println("The expected way to run the tool is by using a crontab to run every hour");
    System.out.println("\nUsage  : java -jar ekit.jar EFT [OPTION] <fail-directory> <trash-directory>\n");
    System.out.println(" OPTION:");
    System.out.println("     -h <hours>   : Default i 1 hour, but increase if you want to run this at a rarer interval");
    System.out.println("\nExample - default");
    System.out.println("java -jar ekit.jar EFT /opt/fssf/failed /opt/fssf/trash");
    System.out.println("\nExample - same as above, but excludes a businessType that never goes through both ECP and EDX");
  }


}
