package no.statnett.ecp.em;

import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.DebugLog;
import no.statnett.ecp.utils.Div;
import no.statnett.ecp.utils.EcpEdxLog;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;

import static java.time.ZoneOffset.UTC;

public class EcpLog2MessageParser {

  private String ecpLogDirectoryStr;
  private DebugLog debugLog;
  private int noAckLimitSec = 60;

  // Important timestamps to keep track of what to log
  private long oldestEcpTms = Long.MAX_VALUE;
  private long newestEcpTms = 0;
  private long oldestModifiedLogFile;
  private long newestModifiedLogFile;

  // Datastructures for message data
  private Map<String, Message> sndMsgMap = new HashMap<>();
  private Map<String, Message> rcvMsgMap = new HashMap<>();
  private Map<String, Long> mapRCplusGTMStoPayloadSize = new HashMap<>();

  public EcpLog2MessageParser(String ecpLogDirectoryStr, DebugLog debugLog, int noAckLimitSec) {
    this.ecpLogDirectoryStr = ecpLogDirectoryStr;
    this.debugLog = debugLog;
    this.noAckLimitSec = noAckLimitSec;
  }

  public long getOldestEcpTms() {
    return oldestEcpTms;
  }

  public long getNewestEcpTms() {
    return newestEcpTms;
  }

  public long getOldestModifiedLogFile() {
    return oldestModifiedLogFile;
  }

  public long getNewestModifiedLogFile() {
    return newestModifiedLogFile;
  }

  public Map<String, Message> getSndMsgMap() {
    return sndMsgMap;
  }

  public Map<String, Message> getRcvMsgMap() {
    return rcvMsgMap;
  }

  public Map<String, Long> getMapRCplusGTMStoPayloadSize() {
    return mapRCplusGTMStoPayloadSize;
  }

  public void processDirectory() throws IOException {
    File fileDir = new File(ecpLogDirectoryStr);
    int counter = 0;
    if (fileDir.isDirectory()) {
      // Process all ecp.log/gz-files in directory in cronological order, starting with the oldest
      List<Path> logFilesOrderedByModifiedAscending = Files.list(Path.of(ecpLogDirectoryStr)).filter(p -> p.getFileName().toString().matches("ecp.(log|.*gz)|ecp-audit(.*log|.*gz)")).sorted((o1, o2) -> (int) (o1.toFile().lastModified() - o2.toFile().lastModified())).collect(Collectors.toList());
      if (logFilesOrderedByModifiedAscending.size() > 0) {
        oldestModifiedLogFile = logFilesOrderedByModifiedAscending.get(0).toFile().lastModified();
      }
      for (Path p : logFilesOrderedByModifiedAscending) {
        counter++;
        if (p.toFile().lastModified() >= newestModifiedLogFile) {
          newestModifiedLogFile = p.toFile().lastModified();
        } else {
          continue; // Skip all log-files which have processed before - this will lead to the effect that we for the most part only process the ecp.log and no gz-files (older/history)
        }
        Map<LogLine, Integer> llMap;
        if (p.toString().endsWith("gz")) {
          llMap = processGZIP(p);
        } else {
          llMap = processPlainText(p.toString());
        }
        String debugMsg = "Process " + p.getFileName() + " found " + llMap.values().stream().mapToInt(Integer::intValue).sum() + " log-lines: " +
            "snd (init/payl-ack/later_ack): " + llMap.getOrDefault(LogLine.SND, 0) + "/" + llMap.getOrDefault(LogLine.SND_PAYLOAD, 0) + "-" + llMap.getOrDefault(LogLine.SND_ACK, 0) + "/" + llMap.getOrDefault(LogLine.SND_LATER_ACK, 0) + ", " +
            "rcv (init/comp/comp_fail): " + llMap.getOrDefault(LogLine.RCV, 0) + "/" + llMap.getOrDefault(LogLine.RCV_COMP, 0) + "/" + llMap.getOrDefault(LogLine.RCV_COMP_FAILED, 0) + ", " +
            "else (uninteresting/already_processed/error): " + llMap.getOrDefault(LogLine.UNINTERESTING, 0) + "/" + llMap.getOrDefault(LogLine.ALREADY_PROCESSED, 0) + "/" + llMap.getOrDefault(LogLine.ERROR, 0);
        debugLog.debug(debugMsg);
      }
    } else {
      System.err.println("Input argument " + fileDir + " is not a directory - exiting");
      System.exit(1);
    }
    if (counter == 0) {
      System.err.println("No files found in directory " + fileDir + " that matches the excpected filename pattern 'ecp.(log|.*gz)|ecp-audit(.*log|.*gz)' - exiting");
      System.exit(1);
    }
  }

  private Map<LogLine, Integer> processGZIP(Path p) {
    Map<LogLine, Integer> countPrLogLine = new HashMap<>();
    try {
      InputStream fileStream = new FileInputStream(p.toFile());
      InputStream gzipStream = new GZIPInputStream(fileStream);
      Reader decoder = new InputStreamReader(gzipStream, StandardCharsets.UTF_8);
      new BufferedReader(decoder).lines().forEach(s ->
      {
        LogLine ll = processLogLine(s);
        countPrLogLine.put(ll, countPrLogLine.getOrDefault(ll, 0) + 1);
      });
    } catch (Exception e) {
      debugLog.debug("Error occurred during processing of gzip-log " + p.getFileName(), e);
      // We log and ignore this problem
    }
    return countPrLogLine;
  }

  private Map<LogLine, Integer> processPlainText(String log) {
    Map<LogLine, Integer> countPrLogLine = new HashMap<>();
    try (Stream<String> stream = Files.lines(Paths.get(log), StandardCharsets.UTF_8)) {
      stream.forEach(s -> {
        LogLine ll = processLogLine(s);
        countPrLogLine.put(ll, countPrLogLine.getOrDefault(ll, 0) + 1);
      });
    } catch (IOException e) {
      debugLog.debug("Error occurred during processing of plain-text log " + log, e);
      // We log and ignore this problem
    }
    return countPrLogLine;
  }

  // s is the line we'll investigate/parse
  private LogLine processLogLine(String s) {
    try {
      LogLine ll = LogLine.UNINTERESTING;
      if (s != null && s.length() > 20 && s.startsWith("202")) { // 2023-02-27 12:48:12.996
        long tms = Long.parseLong(s.substring(0, 23).replaceAll("[^\\d]", ""));
        if (tms < newestEcpTms - 1000) {
          // We will read the same logs over and over, thus processing the same lines over and over. To speed up the process
          // we'll ignore lines that we think are already processed. Usually it's that simple, that if newestEcpTms is set, we need
          // only process newer log-entries than that. But! sometimes log-entries do not follow *strict* chronological order, and we
          // therefore have to allow for some margin. The margin is set to 1 second (1000 millisec), hopefully this is enough.

          return LogLine.ALREADY_PROCESSED;
        }

        // Man finner først MSG-01 (men den inneholder ikke msgId), deretter MSG-11 (m/ msgId) (ECP forsøker å sende en melding), deretter MSG-09/10 (ACK som kommer tilbake m/msgId)
        // MSG-01-Example ECP 4.10: 2024-06-18 06:58:39.195  INFO 2454074 --- [Camel (camel-1) thread #14 - JmsConsumer[ecp.endpoint.outbox]] EcpAuditLog                              : ComponentCode=50V0000000009005, ComponentType=ECP Endpoint, RecordID=f0799318-6714-40c5-a81a-1460ab731be6, ID=MSG-01, Category=MSG, Text=Sending message [messageID=<null>,internalType=STANDARD_MESSAGE,senderCode=50V0000000009005,receiverCode=48V000000000009U,businessType=HVDC-NSL-A26,baMessageID=20240618-001,senderApplication=HVDC,generated=2024-06-18T08:58:39.141+02:00[Europe/Brussels]] with payloadSize=30292., Direction=OUT, User=SYSTEM, Severity=0
        // MSG-01-Example ECP 4.12: 2024-06-13 09:00:24.020  INFO 1238105 --- [Camel (camel-1) thread #24 - JmsConsumer[ecp.endpoint.outbox]] EcpAuditLog                              : ComponentCode=50V000000000251G, ComponentType=ECP Endpoint, RecordID=ece8c5cd-bc10-4aaf-9d73-26da7e0413aa, ID=MSG-01, Category=MSG, Text=Sending message [messageID=<null>,internalType=STANDARD_MESSAGE,senderCode=50V000000000251G,receiverCode=50V0000000000881,businessType=NBM-MFRREAM-CIM-PTA47-MTA41-ACK,baMessageID=674beb43063948be8391f0716f0c9360,senderApplication=fifty-bspi-electronic-ordering-bsp-heartbeat,generated=2024-06-13T11:00:23.982+02:00[Europe/Brussels]] with payloadSize=2273., Direction=OUT, User=SYSTEM, Severity=0
        if (s.contains("MSG-01") && s.contains("STANDARD_MESSAGE") && s.contains("payloadSize=")) { // 1.
          long payloadSize = EcpEdxLog.findPayloadSizeInLogLine(s);
          long generatedTms = EcpEdxLog.findGeneratedTmsInLogLine(s);
          String receiverCode = EcpEdxLog.findReceiverCodeInLogLine(s);
          String key = generatedTms + receiverCode;
          mapRCplusGTMStoPayloadSize.put(key, payloadSize); // nøkkelen "receiverCode + generatedTms" brukes for å koble payloadSize til riktig messageID i MSG-11-seksjonen
          ll = LogLine.SND_PAYLOAD;
        }
        // MSG-11-Example ECP 4.10: 2024-06-18 07:00:50.303  INFO 2454074 --- [Camel (camel-1) thread #14 - JmsConsumer[ecp.endpoint.outbox]] EcpAuditLog                              : ComponentCode=50V0000000009005, ComponentType=ECP Endpoint, RecordID=43298e9b-2b7b-460b-8bb3-7b1d343ace97, ID=MSG-11, Category=MSG, Text=Message [messageID=d45db43c-48a3-4e87-a2aa-6b5523c7e635,internalType=STANDARD_MESSAGE,senderCode=50V0000000009005,receiverCode=48V000000000009U,businessType=HVDC-NSL-A26-ACK,baMessageID=20240618-002,senderApplication=HVDC,generated=2024-06-18T09:00:50.287+02:00[Europe/Brussels]] has been uploaded to 48V000000000045Q., Direction=OUT, User=SYSTEM, Severity=0
        // MSG-11-Example ECP 4.12: 2024-06-13 09:00:24.254  INFO 1238105 --- [Camel (camel-1) thread #24 - JmsConsumer[ecp.endpoint.outbox]] EcpAuditLog                              : ComponentCode=50V000000000251G, ComponentType=ECP Endpoint, RecordID=da7bacb9-e042-45f4-a710-287fdcf6088a, ID=MSG-11, Category=MSG, Text=Message [messageID=4fc44a7b-c43d-4960-b588-ecbe0eea290f,internalType=STANDARD_MESSAGE,senderCode=50V000000000251G,receiverCode=50V000000000022V,businessType=NBM-MFRREAM-CIM-PTA47-MTA41-ACK,baMessageID=71e3539551054d5c8b1202415f0e9cc2,senderApplication=fifty-bspi-electronic-ordering-bsp-heartbeat,generated=2024-06-13T11:00:24.177+02:00[Europe/Brussels]] has been uploaded to 50V000000000119G., Direction=OUT, User=SYSTEM, Severity=0
        else if (s.contains("MSG-11") && s.contains("STANDARD_MESSAGE")) { // 1.
          String messageId = EcpEdxLog.findMessageIDInLogLine(s);
          Message message = new Message(messageId, true);
          message.setAckStatus("Y");
          message.setTmpTms(LocalDateTime.parse("" + tms, Const.tmsNumberMillisec).toInstant(UTC).toEpochMilli());
          message.setBroker(EcpEdxLog.findUploadBrokerInLogLine(s));
          long generatedTms = EcpEdxLog.findGeneratedTmsInLogLine(s);
          String receiverCode = EcpEdxLog.findReceiverCodeInLogLine(s);
          message.setReceiver(receiverCode);
          message.setSender(EcpEdxLog.findSenderCodeInLogLine(s));
          message.setMessageType(EcpEdxLog.findBusinessTypeInLogLine(s));
          message.setBaMessageId(EcpEdxLog.findBaMessageIdInLogLine(s));
          message.setSenderApplication(EcpEdxLog.findSenderApplicationInLogLine(s));
          String key = generatedTms + receiverCode;
          Long payloadSize = mapRCplusGTMStoPayloadSize.get(key); // finner nøkkelen "receiverCode + generatedTms" og bruker den til å slå opp payloadSize som ble laget over (i MSG-01-seksjonen) (vi er da noenlunde trygge på at dette gjelder samme msgId)
          if (payloadSize != null) {
            message.setBytes(payloadSize);
          }
          sndMsgMap.put(messageId, message);
          ll = LogLine.SND;
        }
        // MSG-09-Example ECP 4.10: 2024-06-18 07:00:51.475  INFO 2454074 --- [Camel (camel-1) thread #12 - JmsConsumer[ecp.endpoint.download]] EcpAuditLog                              : ComponentCode=50V0000000009005, ComponentType=ECP Endpoint, RecordID=fa8e96a1-6a89-4794-89be-7c1dc057feb2, ID=MSG-09, Category=MSG, Text=Message [messageID=d45db43c-48a3-4e87-a2aa-6b5523c7e635,internalType=STANDARD_MESSAGE,senderCode=50V0000000009005,receiverCode=48V000000000009U,businessType=HVDC-NSL-A26-ACK,baMessageID=20240618-002,senderApplication=HVDC,generated=<null>] has been delivered., Direction=OUT, User=SYSTEM, Severity=0
        // MSG-09-Example ECP 4.12: 2024-06-13 08:55:25.768  INFO 1238105 --- [Camel (camel-1) thread #20 - JmsConsumer[ecp.endpoint.download]] EcpAuditLog                              : ComponentCode=50V000000000251G, ComponentType=ECP Endpoint, RecordID=6e3828a6-e8c5-45c9-bf01-7cd5be3f89db, ID=MSG-09, Category=MSG, Text=Message [messageID=545bed33-fbae-4e69-8c46-2748ac52342e,internalType=STANDARD_MESSAGE,senderCode=50V000000000251G,receiverCode=50V000000000045J,businessType=NBM-MFRREAM-CIM-PTA47-MTA41-ACK,baMessageID=0189077a91e04cb4961097c98f6a376e,senderApplication=fifty-bspi-electronic-ordering-bsp-heartbeat,generated=<null>] has been delivered., Direction=OUT, User=SYSTEM, Severity=0
        // MSG-10-Example ECP 4.12: 2024-06-13 09:00:05.194  INFO 1238105 --- [Camel (camel-1) thread #20 - JmsConsumer[ecp.endpoint.download]] EcpAuditLog                              : ComponentCode=50V000000000251G, ComponentType=ECP Endpoint, RecordID=d6059406-286e-44be-914b-713b8db731ed, ID=MSG-10, Category=MSG, Text=Message [messageID=d12470cc-5e5a-49ed-913c-62781962e81f,internalType=STANDARD_MESSAGE,senderCode=50V000000000251G,receiverCode=50V000000000024R,businessType=NBM-MFRREAM-CIM-PTA47-MTA39,baMessageID=bf098d70-7004-3a7a-8dd3-1b37aaa9f81f,senderApplication=fifty-bspi-electronic-ordering-bsp-heartbeat,generated=<null>] has been received., Direction=OUT, User=SYSTEM, Severity=0
        else if (s.contains("MSG-09") || s.contains("MSG-10")) { // 2.
          String messageId = EcpEdxLog.findMessageIDInLogLine(s);
          Message message = sndMsgMap.get(messageId);
          if (message != null && message.getTms() == 0L) { // update with first ACK only - will not overwrite if we have set 'No-ACK' (which is a minor flaw in this code, but simpler and not terribly wrong, it will only happen if we have passed the expire-limit
            message.setTms(LocalDateTime.parse("" + tms, Const.tmsNumberMillisec).toInstant(UTC).toEpochMilli());
            message.setMs(message.getTms() - message.getTmpTms());
            ZonedDateTime tmsZDT = ZonedDateTime.ofInstant(Instant.ofEpochMilli(message.getTms()), UTC);
            message.setTmsHour(Long.parseLong(tmsZDT.format(Const.tmsNumberHour)));
            message.setTmsMinute(Long.parseLong(tmsZDT.format(Const.tmsNumberMinute)));
            message.setQuarterHour(Div.calcQH(message.getTms()));  // QuarterHour is calucated so that the first QH of 2020 UTC is 0, the second is 1, etc.
            message.setWeek(Div.calcWeek(tmsZDT));
            ll = LogLine.SND_ACK;
          } else {
            ll = LogLine.SND_LATER_ACK;
          }
        }
        // Man finner først MSG-08 (her finner vi brokerId + messageId), deretter kompletteres mottaket av en melding med MSG-02 - denne er litt mer straight-forward enn når man sender en melding ut, fordi vi har meldingsId+payloadSize sammen
        // MSG-08-Example ECP 4.12: 2024-06-13 09:00:23.421  INFO 1238105 --- [Camel (camel-1) thread #25 - JmsConsumer[ecp.endpoint.receive.50V000000000251G]] EcpAuditLog                              : ComponentCode=50V000000000251G, ComponentType=ECP Endpoint, RecordID=11093f1a-4ceb-41a5-9a7c-5d3257eae1ab, ID=MSG-08, Category=MSG, Text=Message [messageID=d9ed1f78-e74a-4ac6-b6a8-3609a0b1e482,internalType=STANDARD_MESSAGE,senderCode=50V000000000022V,receiverCode=50V000000000251G,businessType=NBM-MFRREAM-CIM-PTA47-MTA41,baMessageID=RESPAD340303,senderApplication=SKSRKv2,generated=2024-06-13T11:00:21.317+02:00[GMT+02:00]] has been received from 50V000000000119G., Direction=IN, User=SYSTEM, Severity=0
        if (s.contains("MSG-08") && s.contains("STANDARD_MESSAGE")) {
          String messageId = EcpEdxLog.findMessageIDInLogLine(s);
          Message message = new Message(messageId, false);
          message.setAckStatus("-");
          rcvMsgMap.put(messageId, message);
          message.setBroker(EcpEdxLog.findDownloadBrokerInLogLine(s));
          ll = LogLine.RCV;
        }
        // MSG-02-Example ECP 4.12: 2024-06-13 09:00:23.967  INFO 1238105 --- [Camel (camel-1) thread #20 - JmsConsumer[ecp.endpoint.download]] EcpAuditLog                              : ComponentCode=50V000000000251G, ComponentType=ECP Endpoint, RecordID=0cf2898d-e105-400b-8508-b6b5d24edb92, ID=MSG-02, Category=MSG, Text=Receiving message [messageID=8356331f-3d77-4df7-a757-17b327390c8b,internalType=STANDARD_MESSAGE,senderCode=50V000000000101Z,receiverCode=50V000000000251G,businessType=NBM-MFRREAM-CIM-PTA47-MTA39-ACK,baMessageID=c136aeac-b302-4b3f-9fa0-4d2cdc68a104,senderApplication=PowerPlan,generated=2024-06-13T11:00:23.720+02:00[GMT+02:00]] with payloadSize=1273., Direction=IN, User=SYSTEM, Severity=0
        else if (s.contains("MSG-02") && s.contains("STANDARD_MESSAGE") && s.contains("payloadSize=")) { // 3.
          String messageId = EcpEdxLog.findMessageIDInLogLine(s);
          Message message = rcvMsgMap.get(messageId);
          if (message != null) {
            // Time data
            long rcvStartEpochMs = EcpEdxLog.findGeneratedTmsInLogLine(s);
            message.setTmpTms(rcvStartEpochMs);
            long rcvEndEcpochMs = LocalDateTime.parse("" + tms, Const.tmsNumberMillisec).toInstant(UTC).toEpochMilli();
            message.setTms(rcvEndEcpochMs);
            message.setMs(message.getTms() - message.getTmpTms());
            ZonedDateTime tmsZDT = ZonedDateTime.ofInstant(Instant.ofEpochMilli(message.getTms()), UTC);
            message.setTmsHour(Long.parseLong(tmsZDT.format(Const.tmsNumberHour)));
            message.setTmsMinute(Long.parseLong(tmsZDT.format(Const.tmsNumberMinute)));
            message.setQuarterHour(Div.calcQH(message.getTms())); // QuarterHour is calucated so that the first QH of 2020 UTC is 0, the second is 1, etc.
            String weekNo = Div.calcWeek(tmsZDT);
            message.setWeek(weekNo);

            // Various other data
            message.setReceiver(EcpEdxLog.findReceiverCodeInLogLine(s));
            message.setSender(EcpEdxLog.findSenderCodeInLogLine(s));
            message.setMessageType(EcpEdxLog.findBusinessTypeInLogLine(s));
            message.setBaMessageId(EcpEdxLog.findBaMessageIdInLogLine(s));
            message.setSenderApplication(EcpEdxLog.findSenderApplicationInLogLine(s));
            message.setBytes(EcpEdxLog.findPayloadSizeInLogLine(s));
            ll = LogLine.RCV_COMP;
          } else {
            ll = LogLine.RCV_COMP_FAILED;
          }
        }

        if (tms < oldestEcpTms) {
          oldestEcpTms = tms;
        }
        if (tms > newestEcpTms)
          newestEcpTms = tms;
      }
      return ll;
    } catch (NumberFormatException |
             DateTimeParseException ex) {
      debugLog.debug("Error occurred: " + ex + " with input text: " + s);
      return LogLine.ERROR;
    }
  }

  // Process all un-acknowledged messages in the time period startZDT to endZDT. The rules for the un-acknowledged messages (msg.getMs() == O signals this) are
  // 1. If msg.tmpTms + noAckLimitMs < startZDT --> they have (or should have if everything is ok) already have been logged --> dismiss
  // 2. Else if msg.tmpTms + noAckLimitMs > endZDT --> we're still waiting for an ACK --> dismiss/skip for now
  // 3. Else log msg with timestamp tmpTms + noAckLimitMs and set AckStatus = 'N'

  // NoAckLimitMs is not the same as expireLim (which is the limit where we *count* a message as 'expired' whether or not it actually was received). NoAckLimitMs
  // is the maximum time to wait before you decide to log the message that was sent (but never got an ack back). The downside of a high noAckLimit can be illustrated
  // with an example: If you set noAckLimit to 30 min, then a message you sent at 12:00 (and never received an ack) will be logged with AckStatus='N' at 12:30. This
  // can be confusing. The noAckLimit I suggest is 15 min, which should cover most burst-delays and give a reliable status of the message (although not 100% accurate,
  // since the messages could have been received, and the ack could come a second or a minute later).
  public void processNoACKOnMessages(Map<String, Message> sndMsgMap, ZonedDateTime startZDT, ZonedDateTime endZDT) {
    // Utils.find all messages that are not ACK'ed
    Set<String> msgMsWith0L = sndMsgMap.entrySet().stream().filter(m -> m.getValue().getMs() == 0L).map(Map.Entry::getKey).collect(Collectors.toSet());

    int counter = 0;
    Set<String> alreadyProcessedNoACKs = new HashSet<>();
    debugLog.debug("Processing " + msgMsWith0L.size() + " NoAck-messages, and see if they fall with the time period from " + startZDT + " to " + endZDT);
    for (String messageId : msgMsWith0L) {
      Message message = sndMsgMap.get(messageId);
      long noAckTms = message.getTmpTms() + noAckLimitSec * 1000L;
      String timestampExplained = "tmpTms " + ZonedDateTime.ofInstant(Instant.ofEpochMilli(message.getTmpTms()), UTC) + " + noAckLimit (" + noAckLimitSec + "s)";

      if (startZDT != null && noAckTms < startZDT.toInstant().toEpochMilli()) {
        alreadyProcessedNoACKs.add(messageId);
        debugLog.debug("Message " + messageId + " with " + timestampExplained + " is before startZDT " + startZDT + " - therefore already processed - do nothing");
      } else if (noAckTms > endZDT.toInstant().toEpochMilli()) {
        debugLog.debug("Message " + messageId + " with " + timestampExplained + " is after  endZDT   " + endZDT +   " - have patience, wait for ACK - do nothing");
      } else {
        message.setTms(noAckTms);
        message.setMs(message.getTms() - message.getTmpTms());
        ZonedDateTime tmsZDT = ZonedDateTime.ofInstant(Instant.ofEpochMilli(message.getTms()), UTC);
        message.setTmsHour(Long.parseLong(tmsZDT.format(Const.tmsNumberHour)));
        message.setTmsMinute(Long.parseLong(tmsZDT.format(Const.tmsNumberMinute)));
        message.setQuarterHour(Div.calcQH(message.getTms())); // QuarterHour is calucated so that the first QH of 2020 UTC is 0, the second is 1, etc.
        message.setWeek(Div.calcWeek(tmsZDT));
        message.setAckStatus("N");
        debugLog.debug("Message " + messageId + " with " + timestampExplained + " is before endZDT   " + endZDT +   " - get tms " + tmsZDT + " and AckStatus 'N'");
        counter++;
      }
    }
    if (!alreadyProcessedNoACKs.isEmpty()) {
      debugLog.debug("Found " + alreadyProcessedNoACKs.size() + " messages that are not ACK'ed, but have already been processed in previous runs - they will be removed");
      sndMsgMap.keySet().removeAll(alreadyProcessedNoACKs);
    }
    if (counter > 0)
      debugLog.debug(counter + " messages have gotten AckStatus 'N'");
  }
}
