package no.statnett.ecp.emd;

import no.statnett.ecp.em.EcpLog2MessageParser;
import no.statnett.ecp.em.Message;
import no.statnett.ecp.utils.DebugLog;
import no.statnett.ecp.utils.Options;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.time.ZoneOffset.UTC;


public class ECPMsgDelay {

  public static final String VERSION = "v1.12.1";
  private static final DateTimeFormatter tmsF = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");
  private static final DateTimeFormatter tmsHourF = DateTimeFormatter.ofPattern("yyyyMMddHH");
  private static final DateTimeFormatter dateF = DateTimeFormatter.ofPattern("yyyyMMdd");
  private static final DateTimeFormatter jsonF = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");

  private static int delayLim = 10;
  private static int expireLim = 60; // 1 min
  private static int noAckLim = 900; // 15 min
  private static int noDaysSummary = 7;
  private static boolean snd = false;
  private static boolean rcv = false;
  private static File limitFile;
  private static File clockSkewFile;
  private static File oppositeFile;
  private static File summaryFile;
  private static DebugLog debugLog = null;
  private static int offsethour = 1;


  public static void main(String[] args) {
    List<String> mandatoryArgs = parseOptions(args);
    try {

      if (summaryFile == null) { // normal behavior
        String ecpLogDirectory = null;
        if (mandatoryArgs.size() != 1) {
          usage();
          System.exit(1);
        } else {
          ecpLogDirectory = mandatoryArgs.get(0);
        }
        EcpLog2MessageParser ecpLog2MessageParser = new EcpLog2MessageParser(ecpLogDirectory, debugLog, noAckLim);
        ecpLog2MessageParser.processDirectory();

        Long newestTms = ecpLog2MessageParser.getNewestEcpTms();
        ZonedDateTime newest = LocalDateTime.parse("" + newestTms, tmsF).atZone(UTC);
        ZonedDateTime endZDT = newest.truncatedTo(ChronoUnit.MINUTES);


        ZonedDateTime runTms = Instant.now().atZone(UTC).minusHours(offsethour).truncatedTo(ChronoUnit.HOURS);

        if (snd) {

          ZonedDateTime previousEndZDT = newest.truncatedTo(ChronoUnit.HOURS);
          if (offsethour > 0) {
            previousEndZDT = runTms.truncatedTo(ChronoUnit.HOURS);
          }
          ecpLog2MessageParser.processNoACKOnMessages(ecpLog2MessageParser.getSndMsgMap(), previousEndZDT, endZDT);
        }
        printNormalOutput(ecpLog2MessageParser, runTms);
      } else { // if -x is specified
        // Read first line of summary file
        String firstLine = Files.lines(summaryFile.toPath()).findFirst().orElse(null);
        if (firstLine == null) {
          System.err.println("Summary file is empty");
          System.exit(1);
        } else if (firstLine.contains("endpoint")) {
          processOppositeSummary(summaryFile);
        } else {
          processStandardSummary(summaryFile);
        }
      }
    } catch (Throwable t) {
      System.err.println("Error occurred: " + t);
    }
  }

  private static List<String> parseOptions(String[] initialArgs) {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String debugFilename = Options.parseString(initArgsList, "-z");
    File debugFile = null;
    if (debugFilename != null) {
      debugFile = Options.checkFile(debugFilename, true);
    }
    debugLog = new DebugLog(debugFile, false);
    delayLim = Options.parseInt(initArgsList, "-d", 1, 3600, 10);
    expireLim = Options.parseInt(initArgsList, "-e", 1, 3600, 60);
    if (delayLim >= expireLim) {
      System.err.println("Delay limit must be less than expire limit - exiting");
      System.exit(1);
    }
    String clockSkewFileName = Options.parseString(initArgsList, "-c");
    if (clockSkewFileName != null) {
      clockSkewFile = Options.checkFile(clockSkewFileName, true);
    }
    String limitFileName = Options.parseString(initArgsList, "-o");
    if (limitFileName != null) {
      limitFile = Options.checkFile(limitFileName, true);
    }
    String oppositeFileName = Options.parseString(initArgsList, "-p");
    if (oppositeFileName != null) {
      oppositeFile = Options.checkFile(oppositeFileName, true);
    }
    String summaryFileName = Options.parseString(initArgsList, "-x");
    if (summaryFileName != null) {
      summaryFile = Options.checkFile(summaryFileName, true);
    }
    noDaysSummary = Options.parseInt(initArgsList, "-y", 1, 365, 7);
    rcv = Options.parseBoolean(initArgsList, "-r");
    snd = Options.parseBoolean(initArgsList, "-s");
    offsethour = Options.parseInt(initArgsList, "-t", 0, 168 * 2, 1);
    if (rcv && snd || !rcv && !snd) { // default is snd = true, rcv= f
      snd = true;
      rcv = false;
    }
    return Options.removeLeftoverOptions(initArgsList);
  }


  private static void usage() {
    System.out.println("ECPMsgDelay (EMD) " + VERSION + ", a tool for analyzing message delays when sending/receiving from other ECP-endpoints");
    System.out.println("Usage  : java -jar ekit.jar EMD [OPTION] <ecp-log-directory>\n");
    System.out.println(" OPTION:");
    System.out.println("     -s             : process only 'sent messages' - this is default behaviour, these timestamps are most trustworthy");
    System.out.println("     -r             : process only 'received messages', -s  will override -r");
    System.out.println("     -d<sec>        : The delay-limit columns will show the number based on this limit in seconds. Default is 10");
    System.out.println("     -e<sec>        : The expire-limit columns will show the number based on this limit in seconds. Default is 60. Disabled if lower or equals to -d");
    System.out.println("     -o <filename>  : The entries that exceed the delay limit will be appended to this file");
    System.out.println("     -c <filename>  : Received entries that has been received before sent, indicating clock skew (only with -r)");
    System.out.println("     -p <filename>  : Append summary pr opposite endpoint pr hour to this file");
    System.out.println("     -x <filename>  : Produces monthly summary of the file specified. The file must contain the std-out or the opposite file (-p).");
    System.out.println("     -y<nodays>     : Only valid with -x. Specifies the number of days to include in the summary. Default is last 7 days.");
    System.out.println("     -t<hours>      : Default i 1, which means previous hour will be processed. Any other numbers is meant to be used for testing purposes.");
    System.out.println("     -z <filename>  : Debug-file will contain information about the ECPMsgDelay processing, in case something fails.");
    System.out.println("\nExample 1: Summary of all sent messages, delay/expire-limit is 10/60 sec, storing output to stdout.json (which is used in Ex 5)");
    System.out.println("\tjava -jar ekit.jar EMD /var/log/ecp-endpoint > stdout.json");
    System.out.println("Example 2: Summary of last hour received messages, delay/expire-limit is 10/60 sec:");
    System.out.println("\tjava -jar ekit.jar EMD -r /var/log/ecp-endpoint");
    System.out.println("Example 3: Summary of last hour received messages + 3 more files for different purposes, delay/expire-limit is 20/60 sec:");
    System.out.println("\tjava -jar ekit.jar EMD -r -d20 -c neg.json -o lim.json -p opp.json /var/log/ecp-endpoint");
    System.out.println("Example 4: Summary of last hour sent messages + 2 files for different purposes, delay/expire-limit is 10/60 sec:");
    System.out.println("\tjava -jar ekit.jar EMD -h -o lim.json -p opp.json /var/log/ecp-endpoint");
    System.out.println("Example 5: Summary of summary, gives daily + monthly summary based on the standard output (see Ex 1):");
    System.out.println("\tjava -jar ekit.jar EMD -x stdout.json -y7");
    System.out.println("Example 6: Summary of summary, gives daily + monthly summary based on the opposite output (see Ex 3 and 4):");
    System.out.println("\tjava -jar ekit.jar EMD -x opp.json -y7");
    System.out.println("\nThe delay analyzer is compatible with ECP 4.9-4.15 - it depends upon certain log events. Other");
    System.out.println("versions of ECP might not give any useful results. It calculates the delays of the message");
    System.out.println("transmissions in both directons, SEND and RECEIVE:");
    System.out.println("  * SEND delay is recorded when first DELIVERED or RECEVIED ACK is received and then compared with SEND tms.");
    System.out.println("  * RECEIVE delay is recorded when STANDARD message is recevied and compared with 'Generated' tms in the msg");
    System.out.println("\nThe main output is a series of json-objects, one pr hour with a summary of the delays/sizes.");
    System.out.println("The output is meant to be easy to read for humans and easy to parse for Splunk, etc. The OPTIONS allow you ");
    System.out.println("to investigate the delays in more detail.");
    System.out.println("\nSince generated tms inside the message (RECEIVE case) can come from an endpoint with a clock skewed");
    System.out.println("compared to this endpoint, we might get negative RECIEVE delays. Those are not counted, but can be");
    System.out.println("listed using the -c option. The -o option will list all messages that exceed the limits in seconds.");
    System.out.println("The -p option will give a summary of the traffic for each opposite endpoint pr hour. ");
    System.out.println("\nExplanations of the various terms used in the output:");
    System.out.println("\ttimstamp     Timestamp for the start of the time period of measurements");
    System.out.println("\tdirection    Direction, snd or rcv");
    System.out.println("\tendpoint     Sending or receiving endpoint");
    System.out.println("\tmeasureEP    The local endpoint, the endpoint we're measuring from");
    System.out.println("\toppositeEP   The opposite/remote endpoint, the endpoint we're measuring to/from");
    System.out.println("\tbroker       The ECP broker that transferred the message");
    System.out.println("\tmsgId        The ECP message id");
    System.out.println("\tcnt          Total number of messages");
    System.out.println("\tcntDel       Number of messages that exceed the delay limit and below expire limit");
    System.out.println("\tcntExp       Number of messages that exceed the expire limit");
    System.out.println("\tcntLim       Number of messages that exceed the delay limit");
    System.out.println("\tms           Delay for the message in milliseconds");
    System.out.println("\tavgMs        Average delay for all messages");
    System.out.println("\tavgMsDel     Average delay for messages that exceed the delay limit and below expire limit");
    System.out.println("\tavgMsExp     Average delay for messages that exceed the expire limit");
    System.out.println("\tavgMsLim     Average delay for messages that exceed the delay limit");
    System.out.println("\tKB           Size of the message in kilobytes");
    System.out.println("\tavgKB        Average size for all messages");
    System.out.println("\tavgKBDel     Average size for messages that exceed the delay limit and below the expire limit");
    System.out.println("\tavgKBExp     Average size for messages that exceed the expire limit");
    System.out.println("\tavgKBLim     Average size for messages that exceed the delay limit");
    System.out.println("\tpct          Percent of messages compared to all messages");
    System.out.println("\tpctLim       Percent of delayed messages compared to all delayed messages");
    System.out.println("\tpctLimSelf   Percent of delayed messages compared to all messages from this endpoint");
    System.out.println("\ttroubleIndex The higher the index, the worse the problem is. It is calculated as:");
    System.out.println("\t             overrepresentation * pctLimSelf * cntLim * avgMsLimIndex / 100");
    System.out.println("\t                 overrepresentation is 100 if limits are not overrepresented nor underrepresented");
    System.out.println("\t                 pctLimSelf is 100 if all traffic to the endpoint exceed the limit");
    System.out.println("\t                 limitCount is simply the number of messages that exceed the limit");
    System.out.println("\t                 avgMsLimIndex is 0 if avgMsLim < delayLimit, 1 if avgMsLim == delayLimit, rising to 100 if avgMsLim >= expireLimit");
  }


  private static void printNormalOutput(EcpLog2MessageParser ecpLog2MessageParser, ZonedDateTime runTms) throws IOException {
    long hourKey = Long.parseLong(runTms.format(tmsHourF));
    Map<String, Message> msgMap = snd ? ecpLog2MessageParser.getSndMsgMap() : ecpLog2MessageParser.getRcvMsgMap();

    // Main map with all messages
    Map<String, Message> hourMap = msgMap.values().stream().filter(m -> m.getTmsHour() == hourKey && m.getMs() >= 0).collect(Collectors.toMap(Message::getMessageId, m -> m));
    // Main map with all messages that are above the delayLimit
    Map<String, Message> delayLimitMap = hourMap.values().stream().filter(m -> m.getMs() >= delayLim * 1000L).collect(Collectors.toMap(Message::getMessageId, m -> m));
    Map<String, Message> expireLimitMap = expireLim == -1 ? new HashMap<>() : hourMap.values().stream().filter(m -> m.getMs() >= expireLim * 1000L).collect(Collectors.toMap(Message::getMessageId, m -> m));

    // Default output to stdout
    System.out.print(logHOURSummary(runTms, hourMap, delayLimitMap, expireLimitMap, snd));

    if (limitFile != null) { // Optional output to file - delayLimits
      logDelayLimits(delayLimitMap);
    }
    if (oppositeFile != null) { // Optional output to file - opposite
      logOpposite(runTms, hourMap, delayLimitMap);
    }

    if (clockSkewFile != null && rcv) { // Optional output to file - clock skew
      Map<String, Message> rcvNegMap = hourMap.values().stream().filter(m -> m.getMs() < 0).collect(Collectors.toMap(Message::getMessageId, m -> m));
      logClockSkew(rcvNegMap);
    }
  }

  private static StringBuffer logHOURSummary(ZonedDateTime tms, Map<String, Message> hourMap, Map<String, Message> delayLimitMap, Map<String, Message> expireLimitMap, boolean snd) {
    long cntTot = hourMap.size();
    long cntExp = expireLimitMap.size();
    long cntDel = delayLimitMap.size() - cntExp;

    long avgMs = (long) hourMap.values().stream().mapToLong(Message::getMs).average().orElse(0L);
    long msExp = expireLimitMap.values().stream().mapToLong(Message::getMs).sum();
    long msDel = delayLimitMap.values().stream().mapToLong(Message::getMs).sum();
    long avgMsDel = cntDel == 0 ? 0 : (msDel - msExp) / cntDel;
    long avgMsExp = cntExp == 0 ? 0 : msExp / cntExp;

    long avgKB = (long) hourMap.values().stream().mapToLong(Message::getKB).average().orElse(0L);
    long kbExp = expireLimitMap.values().stream().mapToLong(Message::getKB).sum();
    long kbDel = delayLimitMap.values().stream().mapToLong(Message::getKB).sum();
    long avgKBDel = cntDel == 0 ? 0 : (kbDel - kbExp) / cntDel;
    long avgKBExp = cntExp == 0 ? 0 : kbExp / cntExp;

    long maxMs = hourMap.values().stream().mapToLong(Message::getMs).max().orElse(0L);
    long maxKB = hourMap.values().stream().mapToLong(Message::getKB).max().orElse(0L);

    StringBuffer sb = new StringBuffer();
    sb.append("{ \"timestamp\": \"").append(tms.format(jsonF)).append("\", ").
        append("\"direction\": \"").append(snd ? "snd" : "rcv").append("\", ").
        append("\"cnt\": ").append(String.format("%4d", cntTot)).append(", ").
        append("\"cntDel\": ").append(String.format("%4d", cntDel)).append(", ").
        append("\"cntExp\": ").append(String.format("%4d", cntExp)).append(", ").

        append("\"avgMs\": ").append(String.format("%9d", avgMs)).append(", ").
        append("\"avgMsDel\": ").append(String.format("%5d", avgMsDel)).append(", ").
        append("\"avgMsExp\": ").append(String.format("%9d", avgMsExp)).append(", ").

        append("\"avgKB\": ").append(String.format("%5d", avgKB)).append(", ").
        append("\"avgKBDel\": ").append(String.format("%5d", avgKBDel)).append(", ").
        append("\"avgKBExp\": ").append(String.format("%5d", avgKBExp)).append(", ").

        append("\"maxMs\": ").append(String.format("%10d", maxMs)).append(", ").
        append("\"maxKB\": ").append(String.format("%5d", maxKB)).
        append(" }\n");
    return sb;
  }

  private static void logDelayLimits(Map<String, Message> delayLimitMap) throws IOException {
    StringBuffer sb = new StringBuffer();
    for (Message message : delayLimitMap.values().stream().sorted(Comparator.comparingLong(Message::getTms)).collect(Collectors.toList())) {
      // convert tms to a string on the form of "yyyy-MM-dd HH:mm:ss"
      sb.append("{ \"timestamp\": \"").append(ZonedDateTime.ofInstant(Instant.ofEpochMilli(message.getTms()), UTC).format(jsonF)).append("\", ").
          append("\"direction\": \"").append(snd ? "snd" : "rcv").append("\", ").
          append("\"msgId\": \"").append(message.getMessageId()).append("\", ").
          append("\"ms\": ").append(String.format("%7d", message.getMs())).append(", ").
          append("\"KB\": ").append(String.format("%5d", message.getKB())).append(", ").
          append("\"measureEP\": \"").append(message.getMeasureEP()).append("\", ").
          append("\"broker\": \"").append(message.getBroker()).append("\", ").
          append("\"oppositeEP\": \"").append(message.getOppositeEP()).append("\"").
          append(" }\n");
    }
    Files.writeString(limitFile.toPath(), sb.toString(), StandardOpenOption.APPEND);
  }

  private static void logClockSkew(Map<String, Message> negMap) throws IOException {
    if (negMap.size() > 0) {
      StringBuffer sb = new StringBuffer();
      for (Message message : negMap.values().stream().sorted(Comparator.comparingLong(Message::getTms)).collect(Collectors.toList())) {
        // convert tms to a string on the form of "yyyy-MM-dd HH:mm:ss"
        sb.append("{ \"timestamp\": \"").append(ZonedDateTime.ofInstant(Instant.ofEpochMilli(message.getTms()), UTC).format(jsonF)).append("\", ").
            append("\"direction\": \"").append(message.getDirection()).append("\", ").
            append("\"msgId\": \"").append(message.getTms()).append("\", ").
            append("\"ms\": ").append(String.format("%6d", message.getMs())).append(", ").
            append("\"measureEP\": \"").append(message.getMeasureEP()).append("\", ").
            append("\"oppositeEP\": \"").append(message.getOppositeEP()).append("\"").
            append(" }\n");
      }
      Files.writeString(clockSkewFile.toPath(), sb.toString(), StandardOpenOption.APPEND);
    }
  }

  private static void logOpposite(ZonedDateTime tms, Map<String, Message> hourMap, Map<String, Message> delayLimitMap) throws IOException {

    // Prepare summary maps (counts, delay, sizes) for all messages in an hour
    Map<String, Integer> hourCountMap = hourMap.values().stream().collect(Collectors.groupingBy(Message::getOppositeEP, Collectors.summingInt(m -> 1)));
    int totalCount = hourCountMap.values().stream().mapToInt(Integer::intValue).sum();
    Map<String, Long> hourMsMap = hourMap.values().stream().collect(Collectors.groupingBy(Message::getOppositeEP, Collectors.summingLong(Message::getMs)));
    Map<String, Long> hourKBMap = hourMap.values().stream().collect(Collectors.groupingBy(Message::getOppositeEP, Collectors.summingLong(Message::getKB)));

    // Prepare summary maps (counts, delay, sizes) for all messages in an hour that are above the delayLimit
    Map<String, Integer> delayLimitCountMap = delayLimitMap.values().stream().collect(Collectors.groupingBy(Message::getOppositeEP, Collectors.summingInt(m -> 1)));
    int totalDelayLimitCount = delayLimitCountMap.values().stream().mapToInt(Integer::intValue).sum();
    Map<String, Long> delayLimitMsMap = delayLimitMap.values().stream().collect(Collectors.groupingBy(Message::getOppositeEP, Collectors.summingLong(Message::getMs)));
    Map<String, Long> delayLimitKBMap = delayLimitMap.values().stream().collect(Collectors.groupingBy(Message::getOppositeEP, Collectors.summingLong(Message::getKB)));


    StringBuffer sb = new StringBuffer();
    for (String oppositeEP : hourCountMap.keySet()) {
      long count = hourCountMap.get(oppositeEP); // count always greater than 0
      long avgMs = hourMsMap.get(oppositeEP) / count;
      long avgKB = hourKBMap.get(oppositeEP) / count;
      long pct = (100L * count) / totalCount; // totalCount always greater than 0 - range 0-100

      long countLim = delayLimitCountMap.getOrDefault(oppositeEP, 0); // how many delayLimits to this endpoint?
      long avgMsLim = countLim == 0 ? 0 : delayLimitMsMap.getOrDefault(oppositeEP, 0L) / countLim;
      long avgKBLim = countLim == 0 ? 0 : delayLimitKBMap.getOrDefault(oppositeEP, 0L) / countLim;

      long pctLim = totalDelayLimitCount == 0 ? 0 : (100L * countLim) / totalDelayLimitCount;
      long delayLimitOverrepresentation = pct == 0 ? 0 : (100L * pctLim) / pct; // are there more delayLimits than to be expected based on the overall traffic to this endpoint? (100% = as many delayLimits as expected)
      long pctLimSelf = (100L * countLim) / count; // how much delayLimits compared to all messages to this endpoint (100% = as many delayLimits as messages)
      int delayLimitMs = delayLim * 1000;
      long avgLimLimDiff = avgMsLim - delayLimitMs; // difference between the average of those messages exceeding the delayLimit and the delayLimit itself
      long maxExpire = Math.max(expireLim, delayLim) * 1000L;
      long delayLimitAvgIndex = avgLimLimDiff <= 0 ? 0 : (avgLimLimDiff >= maxExpire ? 100 : 1 + ((99L * avgLimLimDiff) / maxExpire)); // 0 = no delay, 1 = average delay equals delayLimit, 100 = 10 minutes delay or more on average (for those exceeding the delayLimit)
      long troubleIndex = (delayLimitOverrepresentation * pctLimSelf * countLim * delayLimitAvgIndex) / 100L;

      sb.append("{ \"timestamp\": \"").append(tms.format(jsonF)).append("\", ").
          append("\"endpoint\": \"").append(oppositeEP).append("\", ").
          append("\"direction\": \"").append(snd ? "snd" : "rcv").append("\", ").
          append("\"cnt\": ").append(String.format("%4d", count)).append(", ").
          append("\"avgMs\": ").append(String.format("%7d", avgMs)).append(", ").
          append("\"avgKB\": ").append(String.format("%5d", avgKB)).append(", ").
          append("\"pct\": ").append(String.format("%3d", pct)).append(", ").

          append("\"cntLim\": ").append(String.format("%4d", countLim)).append(", ").
          append("\"avgMsLim\": ").append(String.format("%7d", avgMsLim)).append(", ").
          append("\"avgKBLim\": ").append(String.format("%5d", avgKBLim)).append(", ").
          append("\"pctLim\": ").append(String.format("%3d", pctLim)).append(", ").
          append("\"pctLimSelf\": ").append(String.format("%3d", pctLimSelf)).append(", ").

          append("\"troubleIndex\": ").append(String.format("%9d", troubleIndex)).
          append(" }\n");
    }
    Files.writeString(oppositeFile.toPath(), sb.toString(), StandardOpenOption.APPEND);
  }

  private static void processStandardSummary(File summaryFile) {
    Locale.setDefault(Locale.US); // Sørger for . som desimalskilletegn

    TreeMap<String, StandardSummary> monthSummaryMap = new TreeMap<>(Comparator.naturalOrder());
    TreeMap<String, StandardSummary> dateSummaryMap = new TreeMap<>(Comparator.naturalOrder());
    try (Stream<String> stream = Files.lines(Paths.get(summaryFile.getPath()), StandardCharsets.UTF_8)) {
      stream.forEach(s -> {
        processSummaryLine(s, monthSummaryMap, dateSummaryMap);
      });

      // Skriver ut alle måneder
      for (String month : monthSummaryMap.keySet()) {
        System.out.print(monthSummaryMap.get(month).print());
      }

      // Litt logikk for å skrive ut de 7 siste dagene i den siste måneden
      List<StandardSummary> dateSummaryList = new ArrayList<>(dateSummaryMap.values());
      // Sorter listen basert på Summary.time-feltet
      dateSummaryList.sort(Comparator.comparing(StandardSummary::getTime));
      while (dateSummaryList.size() > noDaysSummary) {
        // Fjern første element i listen
        dateSummaryList.remove(0);
      }
      for (StandardSummary dateSummary : dateSummaryList) {
        System.out.print(dateSummary.print());
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  private static void processSummaryLine(String s, Map<String, StandardSummary> monthSummaryMap, Map<String, StandardSummary> dateSummaryMap) {
    if (s.contains("avgKBExp")) { // Inneholder nyeste versjon (1.0.0+) av ECPMsgDelay-output
      // { "timestamp": "2023-06-22 23:00:00.000Z", "direction": "snd", "cnt": 1527, "cntDel":    0, "cntExp":    0, "avgMs":    762, "avgMsDel":      0, "avgMsExp":      0, "avgKB":     6, "avgKBDel":     0, "avgKBExp":     0, "maxMs":    4625, "maxKB":    10 }
      int timestampStartPos = s.indexOf("timestamp") + 13;
      String month = s.substring(timestampStartPos, timestampStartPos + 7); // yyyy-MM
      String date = s.substring(timestampStartPos, timestampStartPos + 10); // yyyy-MM-dd
      int directionStartPos = s.indexOf("direction") + 13;
      String direction = s.substring(directionStartPos, directionStartPos + 3); // snd/rcv
      String line = s.substring(s.indexOf("cnt")); // resten av linjen sendes inn i Summary.add()
      monthSummaryMap.merge(month, new StandardSummary(month, direction, line), (oldSummary, newSummary) -> oldSummary.add(line));
      dateSummaryMap.merge(date, new StandardSummary(date, direction, line), (oldSummary, newSummary) -> oldSummary.add(line));
    }
  }

  private static void processOppositeSummary(File summaryFile) {
    Locale.setDefault(Locale.US); // Sørger for . som desimalskilletegn

    TreeMap<String, OppositeSummary> monthSummaryMap = new TreeMap<>(Comparator.naturalOrder());
    TreeMap<String, OppositeSummary> dateSummaryMap = new TreeMap<>(Comparator.naturalOrder());
    try (Stream<String> stream = Files.lines(Paths.get(summaryFile.getPath()), StandardCharsets.UTF_8)) {
      stream.forEach(s -> {
        processOppositeSummaryLine(s, monthSummaryMap, dateSummaryMap);
      });

      // Skriver ut alle måneder
      for (String month : monthSummaryMap.keySet()) {
        System.out.print(monthSummaryMap.get(month).print());
      }

      // Litt logikk for å skrive ut de 7 siste dagene i den siste måneden
      List<OppositeSummary> dateSummaryList = new ArrayList<>(dateSummaryMap.values());
      // Sorter listen basert på Summary.time-feltet
      dateSummaryList.sort(Comparator.comparing(OppositeSummary::getTime));
      while (dateSummaryList.size() > noDaysSummary) {
        // Fjern første element i listen
        dateSummaryList.remove(0);
      }
      for (OppositeSummary dateSummary : dateSummaryList) {
        System.out.print(dateSummary.print());
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  private static void processOppositeSummaryLine(String s, Map<String, OppositeSummary> monthSummaryMap, Map<String, OppositeSummary> dateSummaryMap) {
    if (s.contains("direction")) { // Inneholder nyeste versjon (1.5.1+) av ECPMsgDelay-output
      // { "timestamp": "2023-06-09 12:00:00.000Z", "direction": "snd", "endpoint": "50V000000000113S", "cnt":    1, "avgMs":    609, "avgKB":     0, "avgKBs":     0, "freq":   1, "cntLim":    0, "avgMsLim":      0, "avgKBLim":     0, "avgKBsLim":     0, "freqLim":   0, "freqLimSelf":   0, "troubleIndex":        0 }
      int timestampStartPos = s.indexOf("timestamp") + 13;
      int directionStartPos = s.indexOf("direction") + 13;
      String direction = s.substring(directionStartPos, directionStartPos + 3); // snd/rcv

      String month = s.substring(timestampStartPos, timestampStartPos + 7); // yyyy-MM
      String date = s.substring(timestampStartPos, timestampStartPos + 10); // yyyy-MM-dd

      String endpoint = s.substring(s.indexOf("endpoint") + 12, s.indexOf("endpoint") + 28); // 50V000000000113S
      String line = s.substring(s.indexOf("cnt")); // resten av linjen sendes inn i Summary.add()
      monthSummaryMap.merge(month, new OppositeSummary(month, direction, endpoint, delayLim, expireLim, line), (oldSummary, newSummary) -> oldSummary.add(endpoint, line));
      dateSummaryMap.merge(date, new OppositeSummary(date, direction, endpoint, delayLim, expireLim, line), (oldSummary, newSummary) -> oldSummary.add(endpoint, line));
    }
  }

}

