package no.statnett.ecp.eml;

import no.statnett.ecp.em.EcpLog2MessageParser;
import no.statnett.ecp.em.Message;
import no.statnett.ecp.utils.*;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

import static java.time.ZoneOffset.UTC;


public class ECPMsgLogger {

  public static final String VERSION = "v1.6.3";

  // Timestamp to track when to run process again
  private static boolean noNewMessages = false;

  private static File logFile;
  private static String shutdownFilename;
  private static int delayLim = 10;
  private static int expireLim = 60; // 1 min
  private static int noAckLim = 900; // 15 min
  private static String rotationType = null;
  private static int noOfLogFiles = 0;
  private static int noOfDaysDelayRotation = 0;
  private static boolean compress = false;
  private static DebugLog debugLog = null;

  private static void usage() {
    System.out.println("ECPMsgLogger (EML) " + VERSION);
    System.out.println("\nECPMsgLogger is a tool to log ECP-messages to stdout. It is useful to get an overview of the ECP-message flow and");
    System.out.println("detect delays and missing ACKs (check -d and -e options below) and much more. ");
    System.out.println("EML has been made to run continuously, but with a low footprint on the system. EML will output the ECP-messages");
    System.out.println("approximately 5s after the minute shift, so it should never lag long behind the actual ecp-logs. EML supports");
    System.out.println("'restart' in the sense that it can continue logging from the last timestamp without losing any messages nor");
    System.out.println("duplicating any messages. ");
    System.out.println("EML output is very easy to read and parse and contains the essentials of what goes on in ECP-traffic.");
    System.out.println("EML is compatible with ECP 4.12+, but might work with other versions of ECP as well.");

    System.out.println("\nUsage  : java -jar ekit.jar EML [OPTION] <ecp-log-directory> <eml-log-filename>\n");
    System.out.println(" OPTION:");
    System.out.println("     -s <filename>     : if file exists, EML stop logging *gracefully* within 10s max, remove the file and exit");
    System.out.println("     -d<sec>           : When transmission time is above -d<sec> but below -e<sec> the transmission is considered");
    System.out.println("                         'slow' or 'delayed'. The delay column in the output will have status 'DL'. Default value");
    System.out.println("                         is 10 seconds, but can be set from 1s to 2 weeks (1209600s)");
    System.out.println("     -e<sec>           : When transmision time is above -e<sec> the transmission is considered 'expired'. The delay");
    System.out.println("                         column in the output will have status 'EX'. If no ACK is received within this time, the");
    System.out.println("                         ack-column will be set to 'N' and transmission will be set to the expired limit. There is");
    System.out.println("                         always a chance that an ACK will arrive after this limit, but this ACK is ignored. Default");
    System.out.println("                         value is 60 seconds, but can be set from 1s to 2 weeks (1209600s)");
    System.out.println("     -r day|week|month : Rotate log-file every day, week or month");
    System.out.println("     -m<noLogFiles>    : Allow maximum number of log-files. Default is 10.");
    System.out.println("     -n<noDays>        : Do not rotate data younger than noDays. Default is 0. If set higher, the output log will");
    System.out.println("                         always contain the last noDays of data, even if the log-file is rotated. This is useful");
    System.out.println("                         for analysis of the last days of data (GapMonitor)");
    System.out.println("     -z                : Compress rotated logs");
    System.out.println("     -x <filename>     : Logs debug-info about EML itself. If created, it is safe to delete, it will be re-created.");

    System.out.println("\nExample 1: Default settings, no options - process/print all based on ecp-logs founds in <ecp-log-directory>");
    System.out.println("\tjava -jar ekit.jar EML /var/log/ecp-endpoint eml.log");
    System.out.println("\nExample 2: Same as example 1, but add possibility to shutdown EML gracefully");
    System.out.println("\tjava -jar ekit.jar EML -s eml.shutdown /var/log/ecp-endpoint eml.log");
    System.out.println("\nExample 3: Same as example 2, but change delay/expire limits to 5s and 30s");
    System.out.println("\tjava -jar ekit.jar EML -s eml.shutdown -d5 -e30 /var/log/ecp-endpoint eml.log");
    System.out.println("\nExample 4: Same as example 1, but rotate log-file every day and keep 5 log-files and compress them");
    System.out.println("\tjava -jar ekit.jar EML -r day -m5 -z /var/log/ecp-endpoint eml.log");
    System.out.println("\nExample 5: Same as example 1, but add debug-file for debugging");
    System.out.println("\tjava -jar ekit.jar EML -x eml.debug /var/log/ecp-endpoint eml.log");
    System.out.println("\nExample 6: A production scenario which also support GapMonitor's (GM) need for the last 7+1 days of data");
    System.out.println("\tjava -jar ekit.jar EML -s eml.shutdown -r week -n8 -z -x eml.debug /var/log/ecp-endpoint eml.log");

    System.out.println("\nThe simplest way to run this tool in a stable fashion is to establish two cron-job that run every night");
    System.out.println("at 00:04 and 00:05 to stop and start EML every night. Even if the server is restarted or there are downtimes,");
    System.out.println("EML will in the end always start and catch up all the logs that have been missed. The cron-jobs could look like this:");
    System.out.println("\n# Stops EML at 00:04 by creating a shutdown-file, EML stops within 10 sec");
    System.out.println("4 0 * * * touch eml.shutdown");
    System.out.println("# Starts EML at 00:05, will continue from eml.log's last timestamp and append all new messages to eml.log");
    System.out.println("5 0 * * * java -jar ekit.jar EML -s eml.shutdown -r week -n8 -z -x eml.debug /var/log/ecp-endpoint eml.log");
    System.out.println("\nHaving a stable, always running EML with at most 65 sec delay from the ecp-logs is a good starting point for");
    System.out.println("further analysis of the ECP-message flow. Quarter-hours are becoming important in the ECP-world, and EML can");
    System.out.println("help 'pin' messages to a certain QH and find gaps in the message flow.");

    System.out.println("\nThe transmission time (the 'ms' column in the output) is calculated as follows:");
    System.out.println("  * SEND delay is recorded when ACK is received and compared with SEND tms.");
    System.out.println("  * RECEIVE delay is recorded when STD message is recevied and compared with 'Generated' tms in the msg");

    System.out.println("\nThe output is a log output with the following columns.");
    System.out.println("  * TMS: The timestamp of the message in ISO UTC-format");
    System.out.println("  * WEEK_NO: The week number of the message on the form yyyy'W'ww (ISO format)");
    System.out.println("  * Q_HOUR_NO: The quarter hour number of the message, starting from 0 at 2020-01-01 00:00:00");
    System.out.println("  * DIRECTION: The direction of the message, either 'SND' or 'RCV'");
    System.out.println("  * ACK_STATUS: The status of the message, for SND 'Y' or 'N', but for RCV always '-'");
    System.out.println("  * DELAY_STATUS: Based on transmissiontime: t<0 (NG), t<delayLim (OK), t<expireLim (DL), t>=expireLim (EX)");
    System.out.println("  * TRANSMISSION_TIME: The time in milliseconds as explained above for SEND and RECEIVE");
    System.out.println("  * REMOTE_EP: The remote endpoint of the message (either sent to or received from)");
    System.out.println("  * BROKER_EP: The broker the message was transmitted via");
    System.out.println("  * LOCAL_EP: The local endpoint of the message (either sent from or received at)");
    System.out.println("  * MSG_ID: The message ID of the message");
    System.out.println("  * SIZE: The size of the payload in KB");
    System.out.println("  * MESSAGE_TYPE: The type of the message");
    System.out.println("  * BA_MESSAGE_ID: The business application message ID");

    System.out.println("The log output can serve as a stepping stone to further analysis of the ECP message flow or as a simple 'ECP-GUI'");
    System.out.println("One example is to find if messages are sent every quarter-hour, or if there are gaps");
  }

  private static List<String> parseOptions(String[] initialArgs) {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String debugFilename = Options.parseString(initArgsList, "-x");
    File debugFile = null;
    if (debugFilename != null) {
      debugFile = Options.checkFile(debugFilename, true);
    }
    debugLog = new DebugLog(debugFile, true);
    shutdownFilename = Options.parseString(initArgsList, "-s");
    if (shutdownFilename != null && new File(shutdownFilename).exists()) {
      debugLog.debug("Shutdown-file " + shutdownFilename + " existed on startup - will try to remove it");
      try {
        Files.delete(Path.of(shutdownFilename));
      } catch (IOException e) {
        String errorMsg = "The shutdown-file (" + shutdownFilename + ") existed upon startup and should be removed, but could not be removed: " + e.getMessage() + " will prevent EML from working";
        debugLog.debug(errorMsg);
        System.err.println(errorMsg);
        System.exit(1);
      }
    }
    delayLim = Options.parseInt(initArgsList, "-d", 1, 2 * 168 * 86400, 10);
    expireLim = Options.parseInt(initArgsList, "-e", 1, 2 * 168 * 86400, 60);
    if (delayLim >= expireLim) {
      System.err.println("Delay limit must be less than expire limit - exiting");
      System.exit(1);
    }
    rotationType = Options.parseString(initArgsList, "-r");
    if (rotationType != null && !rotationType.matches("^(day|week|month)$")) {
      System.err.println("Rotate option (-r) must be either 'day', 'week' or 'month' - exiting");
      System.exit(1);
    }
    noOfLogFiles = Options.parseInt(initArgsList, "-m", 1, 1000, 10);
    noOfDaysDelayRotation = Options.parseInt(initArgsList, "-n", 0, 1000, 0);
    compress = Options.parseBoolean(initArgsList, "-z");
    return Options.removeLeftoverOptions(initArgsList);
  }

  private static void loopControl(EcpLog2MessageParser ecpLog2MessageParser) throws InterruptedException {
    long sleepMs = 60000; // Default sleep-interval if no messages are found last minute
    if (!noNewMessages) {
      // The time difference in the logs since last run
      LocalDateTime newestTmsLDT = LocalDateTime.parse("" + ecpLog2MessageParser.getNewestEcpTms(), Const.tmsNumberMillisec);
      LocalDateTime latestMinuteShiftTmsLDT = newestTmsLDT.truncatedTo(ChronoUnit.MINUTES);
      long diffSecInLog = newestTmsLDT.toEpochSecond(ZoneOffset.UTC) - latestMinuteShiftTmsLDT.toEpochSecond(ZoneOffset.UTC); // diff between how far the log has come and the latest minute shift
      diffSecInLog = diffSecInLog > 60 ? 60 : diffSecInLog < 0 ? 0 : diffSecInLog; // diffSecInLog should be between 0 and 60

      // If logging is rare, we might sleep until the next minute starts with a 5 second margin
      sleepMs = (60 - diffSecInLog + 5) * 1000;
    }
    while (sleepMs > 0) {
      if (sleepMs > 10000)
        Thread.sleep(10000);
      else {
        Thread.sleep(sleepMs);
      }
      // Check if shutdown-file exists. If it exists, remove it and and exit program
      if (shutdownFilename != null && new File(shutdownFilename).exists()) {
        debugLog.debug("Shutdown-file " + shutdownFilename + " exists - exiting gracefully after having deleted the file.");
        try {
          Files.delete(Path.of(shutdownFilename));
        } catch (IOException e) {
          debugLog.debug("The shutdown-file " + shutdownFilename + " could not be removed: " + e.getMessage() + " will prevent continuous running of EML", e);
          System.err.println("The shutdown-file " + shutdownFilename + " could not be removed: " + e.getMessage() + " will prevent continuous running of EML");
        } finally {
          debugLog.debug("Exiting");
          System.exit(1);
        }
      }
      sleepMs -= 10000; // Always subtract 10s, until sleepMs < 0 and we exit the loop
    }
  }





  /* MAIN PROGRAM */

  public static void main(String[] args) throws IOException {
    // Print a statement like this:
    // 2024-10-01T00:04:07.121 DEBUG  EML Startup
    System.out.println(LocalDateTime.now().format(Const.localTmsMillisec) + " DEBUG  EML " + VERSION + " Startup");
    try {
      List<String> mandatoryArgs = parseOptions(args);
      String ecpLogDirectory = null;
      if (mandatoryArgs.size() != 2) {
        usage();
        System.exit(1);
      } else {
        ecpLogDirectory = mandatoryArgs.get(0);

        String logFilename = mandatoryArgs.get(1);
        if (!logFilename.contains(".")) {
          logFilename += ".log"; // To make 100% sure we have a '.' in the filename - necessary for rotation-logic
        }
        logFile = Options.checkFile(logFilename, true);
      }

      boolean sleepToNextMinute = false;
      ZonedDateTime endOfPreviousRunZDT = findPreviousRunEndLoggingMinuteTms(logFile); // Can be 0 if no or empty file is specified with the -i option
      long previousEndLoggingMinuteTms = endOfPreviousRunZDT == null ? 0 : endOfPreviousRunZDT.toInstant().toEpochMilli();

      EcpLog2MessageParser ecpLog2MessageParser = new EcpLog2MessageParser(ecpLogDirectory, debugLog, noAckLim);
      while (true) {
        try {
          if (sleepToNextMinute) {
            loopControl(ecpLog2MessageParser); // May shutdown the program - graceful exit if a shutdown-file is found
          } else {
            sleepToNextMinute = true;
          }
          if (rotationType != null && ecpLog2MessageParser.getNewestEcpTms() > 0) {
            ZonedDateTime newest = LocalDateTime.parse("" + ecpLog2MessageParser.getNewestEcpTms(), Const.tmsNumberMillisec).atZone(UTC);
            LogRotator logRotator = new LogRotator(logFile, newest, noOfDaysDelayRotation, debugLog);
            logRotator.rotateLogs(rotationType);
            logRotator.deleteExcessLogs(noOfLogFiles);
            if (compress)
              logRotator.compress();
          }


          // NEW RUN of process - will either process the last minute of the log or the entire log

          ecpLog2MessageParser.processDirectory(); // read all log-files in directory or files that has been modified since last run

          // Parse oldestTms and newestTms into ZonedDateTime
          ZonedDateTime oldestUTC = LocalDateTime.parse("" + ecpLog2MessageParser.getOldestEcpTms(), Const.tmsNumberMillisec).atZone(UTC);
          ZonedDateTime startLoggingMinuteZDT = oldestUTC.truncatedTo(ChronoUnit.MINUTES); // This timestamp will be used only upon startup, otherwise it will be set to the previousEndLoggingMinuteTms (see further down)
          ZonedDateTime newestUTC = LocalDateTime.parse("" + ecpLog2MessageParser.getNewestEcpTms(), Const.tmsNumberMillisec).atZone(UTC);
          ZonedDateTime endLoggingMinuteZDT = newestUTC.truncatedTo(ChronoUnit.MINUTES);

          // Any new entries in log?
          if (endLoggingMinuteZDT.toInstant().toEpochMilli() <= previousEndLoggingMinuteTms) {
            noNewMessages = true;
            continue; // Skip if we have already processed this minute
          } else {
            noNewMessages = false;
          }

          // Snd-message without ACK that are older than this timestamp will be set to N
          ecpLog2MessageParser.processNoACKOnMessages(ecpLog2MessageParser.getSndMsgMap(), endOfPreviousRunZDT, endLoggingMinuteZDT);


          // Housekeeping - remove old log-entries from memory
          int removedFromSnt = 0;
          int removedFromRcv = 0;
          long approxLoglengthInMs = ecpLog2MessageParser.getNewestModifiedLogFile() - ecpLog2MessageParser.getOldestModifiedLogFile() < 3600000 ? 3600000 : ecpLog2MessageParser.getNewestModifiedLogFile() - ecpLog2MessageParser.getOldestModifiedLogFile(); // At least wait 1 hour before an ACK is declared missing, but usually wait for as long as the logs cover
          ZonedDateTime oldestTmsWeKeepInMemory = newestUTC.minus(approxLoglengthInMs, ChronoUnit.MILLIS); // We will keep all messages that are newer than this timestamp in memory
          if (previousEndLoggingMinuteTms > 0) { // Don't remove old log-entries upon first run
            removedFromSnt = removeOldLogEntries(ecpLog2MessageParser.getSndMsgMap(), oldestTmsWeKeepInMemory);
            removedFromRcv = removeOldLogEntries(ecpLog2MessageParser.getRcvMsgMap(), oldestTmsWeKeepInMemory);
          }
          debugLog.debug("Current size of sndMsgMap: " + ecpLog2MessageParser.getSndMsgMap().size() + " (removed " + removedFromSnt + ") and rcvMsgMap: " + ecpLog2MessageParser.getRcvMsgMap().size() + " (removed " + removedFromRcv + ") ranging from " + oldestTmsWeKeepInMemory + " to " + newestUTC);

          // Decide which range of log-entries to print
          if (previousEndLoggingMinuteTms > 0) {
            startLoggingMinuteZDT = ZonedDateTime.ofInstant(Instant.ofEpochMilli(previousEndLoggingMinuteTms), UTC); // Will always be a starting on the minute
          }
          debugLog.debug("Print from " + startLoggingMinuteZDT + " to " + endLoggingMinuteZDT);
          print(startLoggingMinuteZDT, endLoggingMinuteZDT, ecpLog2MessageParser.getSndMsgMap(), ecpLog2MessageParser.getRcvMsgMap()); // process the ONE minute of the log
          previousEndLoggingMinuteTms = endLoggingMinuteZDT.toInstant().toEpochMilli();
        } catch (Exception e) {
          debugLog.debug("Error occurred, EML will sleep and start again in a minute - hopefully recover correctly: ", e);
        }
      }
    } catch (Throwable t) {
      System.out.println(LogOut.e() + " EML failed: " + t.getMessage() + ", Stacktrace: " + Div.onelinerStacktrace(t.getStackTrace(), "no.statnett.ecp"));
    }
  }

  private static int removeOldLogEntries(Map<String, Message> msgMap, ZonedDateTime oldestEcpTmsZDT) {
    // Remove all log-entries that are older than oldestEcpTms and count the number of removed entries
    long oldestEcpTms = oldestEcpTmsZDT.toInstant().toEpochMilli();
    Set<String> msgIdSet = new HashSet<>();
    for (String msgId : msgMap.keySet()) {
      Message msg = msgMap.get(msgId);
      if (msg.getTms() > 0 && msg.getTms() < oldestEcpTms) {
        msgIdSet.add(msgId);
      }
    }
    msgIdSet.forEach(msgMap::remove);
    return msgIdSet.size();
  }

  private static void log(String message) {
    try {
      Files.writeString(logFile.toPath(), message + "\n", StandardOpenOption.APPEND);
    } catch (IOException e) {
      debugLog.debug("Error occurred during printing to log-file " + logFile.getAbsoluteFile() + ": " + e.getMessage());
    }
  }

  private static ZonedDateTime findPreviousRunEndLoggingMinuteTms(File startupFile) throws IOException {
    // If no file is specified return 0
    // Read last line of file and find timestamp on the form 2024-04-26 13:48:34.031Z
    // If timestamp is found, return the epochTms of the begninning of next minute after this timestamp
    if (startupFile == null) {
      return null;
    }
    List<String> logArr = Files.readAllLines(startupFile.toPath());
    if (logArr.isEmpty())
      return null;
    String lastLine = logArr.get(logArr.size() - 1);
    if (lastLine.isEmpty())
      return null;
    if (lastLine.length() < 24) // Some content but not enough - triggers an error on System.exit()
      startupFileError(startupFile, lastLine);
    try {
      return ZonedDateTime.parse(lastLine.substring(0, 24)).truncatedTo(ChronoUnit.MINUTES).plusMinutes(1);
    } catch (RuntimeException re) {
      startupFileError(startupFile, lastLine); // Some parsing of log fails - triggers an error on System.exit()
      return null; // This is never reached
    }
  }

  private static void startupFileError(File startupFile, String lastLine) {
    System.err.println("Not possible to parse last line of the " + startupFile.getName() + " into a timestamp on the form 2024-04-26 13:48:34.031Z");
    System.err.println("The line looks like this:");
    System.err.println(lastLine);
    System.err.println("exiting");
    System.exit(1);
  }


  private static void print(ZonedDateTime startTms, ZonedDateTime endTms, Map<String, Message> sndMsgMap, Map<String, Message> rcvMsgMap) {
    // We print the content of the sndMsgMap and rcvMsgMap in cronological order for the given timeframe by startTms and endTms
    // The content is printed in standard log-fashion: TMS DIRECTION ACKSTATUS SENDER/RECEIVER BROKER MESSAGEID TRANSMISSIONTIME PAYLOADSIZE MESSAGETYPE
    for (ZonedDateTime tms = startTms; tms.compareTo(endTms) < 0; tms = tms.truncatedTo(ChronoUnit.MINUTES).plusMinutes(1)) {
      long minuteKey = Long.parseLong(tms.format(Const.tmsNumberMinute));
      Map<String, Message> msgMap = sndMsgMap.values().stream().filter(m -> m.getTmsMinute() == minuteKey && m.getMs() >= 0).collect(Collectors.toMap(Message::getMessageId, m -> m));
      msgMap.putAll(rcvMsgMap.values().stream().filter(m -> m.getTmsMinute() == minuteKey && m.getMs() != 0).collect(Collectors.toMap(Message::getMessageId, m -> m)));
      for (Message m : msgMap.values().stream().sorted(getComparing()).collect(Collectors.toList())) {
        String delayStatus = m.getMs() >= expireLim * 1000L ? "EX" : m.getMs() >= delayLim * 1000L ? "DL" : m.getMs() < 0 ? "NG" : "OK";
        ZonedDateTime tmsMsg = ZonedDateTime.ofInstant(Instant.ofEpochMilli(m.getTms()), UTC);
        // The output is meant to be easy to read for humans and easy to parse for Splunk, etc.
        String sb = makeLogline(
            tmsMsg,
            m.getWeek(),
            m.getQuarterHour(),
            m.getDirection(),
            m.getAckStatus(),
            delayStatus,
            m.getMs(),
            m.getOppositeEP(),
            m.getBroker(),
            m.getMeasureEP(),
            m.getMessageId(),
            m.getKB(),
            m.getMessageType(),
            m.getBaMessageId(),
            m.getSenderApplication());
        log(sb);
      }
    }
  }

  public static String makeLogline(ZonedDateTime tmsMsg, String week, int quarterHour, String direction, String ackStatus, String delayStatus, long ms, String oppositeEP, String broker, String measureEP, String messageId, long kb, String messageType, String baMessageId, String senderApplication) {
    String sb =
        tmsMsg.format(Const.utcTmsMillisec) + " " +
            "wk=" + week + ", " + // 7 digits
            "qh=" + quarterHour + ", " + // 6 digits for the next 20 years
            "dir=" + direction + " " + // 7 char - comma is included in getDirection()
            "ack=" + ackStatus + ", " + // 1 char
            "stat=" + delayStatus + ", " + // 2 char
            "ms=" + String.format("%-11s", ms + ",") + " " + // room for up to 10 digit ms (= 115.74 days) - max expire limit is 2 weeks
            "rem=" + oppositeEP + ", " + // Always 16 char
            "bro=" + broker + ", " + // Always 16 char
            "loc=" + measureEP + ", " + // Always 16 char
            "mId=" + messageId + ", " + // UUID
            "sz=" + String.format("%-7s", kb + ",") + " " + // room for up 6 digit KB (= 999999 KB ~ 1 GB) - max limit in ECP in default 50 MB, but can be changed to higher limits in ecp.properties
            "type=" + String.format("%-36s", messageType + ",") + " " + // room for 35 char message type - covers 99% of all message types without changing the "fixed columns"
            "bId=" + String.format("%-42s", baMessageId + ", ") + " " +  // room for 41 char baMessageId - covers 99% of all baMessageId without changing the "fixed columns"
            "bSn=" + senderApplication;

    return sb;
  }

  private static Comparator<Message> getComparing() {
    // Make comparator that sort on timestamp (getTms()) ascending then sorts on transmission time descending (getMs())
    return Comparator.comparing(Message::getTms).thenComparing(Comparator.comparing(Message::getMs).reversed());
  }

}

