package no.statnett.ecp.epa;

import com.sun.management.OperatingSystemMXBean;

import java.io.*;
import java.lang.management.ManagementFactory;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;


public class ECPPerfAnalyzer {


  public static final String VERSION = "v2.0.2";
  private static final DateTimeFormatter tmsF = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
  private static final DateTimeFormatter dateF = DateTimeFormatter.ofPattern("yyyyMMdd");
  private static TreeMap ecp2BroMap = new TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, Integer>>>>();
  private static TreeMap ecp2EdxMap = new TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, Integer>>>>();
  private static TreeMap edx2EcpMap = new TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, Integer>>>>();
  private static TreeMap edx2BapMap = new TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, Integer>>>>();
  private static long oldestEcpTms = Long.MAX_VALUE;
  private static long newestEcpTms = 0;
  private static long oldestEdxTms = Long.MAX_VALUE;
  private static long newestEdxTms = 0;
  private static LocalDateTime oldestEcpLdt = null;
  private static LocalDateTime oldestEdxLdt = null;
  private static long ecpLogSize = 0;
  private static long edxLogSize = 0;


  public static void main(String[] args) throws IOException, InterruptedException {
    if (args.length < 2 || args.length > 3) {
      usage();
      System.exit(1);
    }
    boolean acks = false;
    boolean tail = false;
    boolean latest = false;
    String ecpLog = args[0];
    String edxLog = args[1];
    if (args[0].startsWith("-")) {
      if (args[0].contains("a"))
        acks = true;
      if (args[0].contains("f"))
        tail = true;
      if (args[0].contains("l")) {
        latest = true;
      }
      ecpLog = args[1];
      edxLog = args[2];
    }

    ecpLogSize = processDirectory(ecpLog, true, acks, latest);
    edxLogSize = processDirectory(edxLog, false, acks, latest);

    Long oldestTms = Math.min(oldestEcpTms, oldestEdxTms);
    Long newestTms = Math.max(newestEcpTms, newestEdxTms);


    LocalDateTime oldest = LocalDateTime.parse("" + oldestTms, tmsF);
    LocalDateTime newest = LocalDateTime.parse("" + newestTms, tmsF);
    LocalDateTime startTms = oldest.truncatedTo(ChronoUnit.MINUTES);
    LocalDateTime endTms = newest.truncatedTo(ChronoUnit.MINUTES);

    // Fjerner overskytende av siste minutt
    edx2EcpMap = removeExcessFromMap(edx2EcpMap, oldest, endTms);
    ecp2BroMap = removeExcessFromMap(ecp2BroMap, oldest, endTms);
    ecp2EdxMap = removeExcessFromMap(ecp2EdxMap, oldest, endTms);
    edx2BapMap = removeExcessFromMap(edx2BapMap, oldest, endTms);

    heading();

    Integer nextHourToPrint = print(startTms, endTms, oldest.getHour(), false, false, 5, null, null);

    LocalDateTime minuteStart = endTms;
    while (tail) {
      long diffSec = LocalDateTime.parse("" + Math.max(newestEcpTms, newestEdxTms), tmsF).toEpochSecond(ZoneOffset.UTC) - minuteStart.toEpochSecond(ZoneOffset.UTC);
      diffSec = diffSec > 60 ? 60 : diffSec < 0 ? 0 : diffSec;
      long sleepMs = (60 - diffSec + 5) * 1000;
      OperatingSystemMXBean mxBean = (OperatingSystemMXBean) ManagementFactory.getOperatingSystemMXBean();
      Thread.sleep(sleepMs);
      long cpuPercent = (long) (100 * mxBean.getSystemCpuLoad());
      double loadAverage = mxBean.getSystemLoadAverage();
      LocalDateTime minuteEnd = minuteStart.plusMinutes(1);
      long nyEcpLogSize = 0;
      if ((new File(ecpLog + "/ecp.log")).exists())
        nyEcpLogSize = processPlainText(ecpLog + "/ecp.log", Long.parseLong(minuteStart.format(tmsF)), Long.parseLong(minuteEnd.format(tmsF)), true, acks);
      else if ((new File(ecpLog + "/ecp-endpoint.log")).exists())
        nyEcpLogSize = processPlainText(ecpLog + "/ecp-endpoint.log", Long.parseLong(minuteStart.format(tmsF)), Long.parseLong(minuteEnd.format(tmsF)), true, acks);
      long nyEdxLogSize = 0;
      if ((new File(edxLog + "/edx.log")).exists())
        nyEdxLogSize = processPlainText(edxLog + "/edx.log", Long.parseLong(minuteStart.format(tmsF)), Long.parseLong(minuteEnd.format(tmsF)), false, acks);
      else if ((new File(edxLog + "/edx-toolbox.log")).exists())
        nyEdxLogSize = processPlainText(edxLog + "/edx-toolbox.log", Long.parseLong(minuteStart.format(tmsF)), Long.parseLong(minuteEnd.format(tmsF)), false, acks);

      nextHourToPrint = print(minuteStart, minuteEnd, nextHourToPrint, nyEcpLogSize < ecpLogSize, nyEdxLogSize < edxLogSize, diffSec, cpuPercent, loadAverage);
      minuteStart = minuteEnd;
      ecpLogSize = nyEcpLogSize;
      edxLogSize = nyEdxLogSize;
    }
  }

  private static void usage() {
    System.out.println("ECPPerfAnalyzer (EPA) " + VERSION);
    System.out.println("Usage  : java -jar ekit.jar EPA [-afl] <ecp-log-directory> <edx-log-directory>\n");
    System.out.println("Example: java -jar ekit.jar EPA /var/log/ecp-endpoint /var/log/edx-toolbox");
    System.out.println("Example: java -jar ekit.jar EPA -af /var/log/ecp-endpoint /var/log/edx-toolbox");
    System.out.println("\nThe analyzer is compatible with ECP 4.14+ - it depends upon certain log events. Older");
    System.out.println("versions of ECP/EDX is not tested as of now. By default it only counts PLAIN/STANDARD msg.");
    System.out.println("Numbers in parenthesis denotes an incomplete count (not a full minute/hour or log-rotation).\n");
    System.out.println("Option: a     Also count the ACKs, not just PLAIN/STANDARD msg");
    System.out.println("Option: f     Tailing the logs");
    System.out.println("Option: l     Only process the newest logfile (the gz-files will be skipped)");
  }

  private static long processDirectory(String dir, boolean ecp, boolean acks, boolean latest) throws IOException {
    File fileDir = new File(dir);
    long logSize = 0L;
    if (fileDir.isDirectory()) {
      for (Path p : Files.list(Path.of(dir)).filter(p -> p.getFileName().toString().matches("ecp.(log|.*gz)|edx(.*log|.*gz)")).sorted((o1, o2) -> (int) (o1.toFile().lastModified() - o2.toFile().lastModified())).collect(Collectors.toList())) {
        try {
          if (p.toString().endsWith("gz")) {
            if (!latest)
              processGZIP(ecp, p, acks);
          } else {
            logSize = processPlainText(p.toString(), null, null, ecp, acks);
          }
        } catch (FileNotFoundException e) {
          e.printStackTrace();
        } catch (IOException e) {
          e.printStackTrace();
        }
      }
    } else {
      System.err.println("Input argument " + fileDir + " is not a directory - exiting");
      System.exit(1);
    }
    return logSize;
  }

  private static void processGZIP(boolean ecp, Path p, boolean acks) throws IOException {
    InputStream fileStream = new FileInputStream(p.toFile());
    InputStream gzipStream = new GZIPInputStream(fileStream);
    Reader decoder = new InputStreamReader(gzipStream, StandardCharsets.UTF_8);
    new BufferedReader(decoder).lines().forEach(s -> processLogLine(null, null, s, ecp, acks));
  }

  private static long processPlainText(String log, Long startProcessTms, Long endProcessTms, boolean ecp, boolean acks) {
    try (Stream<String> stream = Files.lines(Paths.get(log), StandardCharsets.UTF_8)) {
      stream.forEach(s -> {
        processLogLine(startProcessTms, endProcessTms, s, ecp, acks);
      });
    } catch (IOException e) {
      e.printStackTrace();
    }
    return new File(log).length();
  }

  private static void processLogLine(Long startProcessTms, Long endProcessTms, String s, boolean ecp, boolean acks) {
    try {
      if (s != null && s.length() > 20 && (s.startsWith("2024") || s.startsWith("2025") || s.startsWith("2026"))) {
        long tms = Long.parseLong(s.substring(0, 19).replaceAll("[^\\d]", ""));
        if (startProcessTms != null && tms < startProcessTms)
          return;
        if (endProcessTms == null || tms < endProcessTms) {
          if (ecp) {
            if (acks && s.contains("SendEvent for message")) { // ECP -> EDX (4.14+)
              populateMap(s, ECPPerfAnalyzer.ecp2EdxMap);
            } else if (s.contains("ID=MSG-08") && s.contains("STANDARD_MESSAGE")) { // ECP -> EDX (4.14+)
              populateMap(s, ECPPerfAnalyzer.ecp2EdxMap);
            } else if (s.contains("ID=MSG-01")) { // ECP -> BRO (4.14+)
              if (acks) {
                populateMap(s, ECPPerfAnalyzer.ecp2BroMap);
              } else if (s.contains("STANDARD_MESSAGE")) {
                populateMap(s, ECPPerfAnalyzer.ecp2BroMap);
              }
            }
          } else {
            if (s.contains("routed to endpoint") && !s.contains("toolbox-gateway")) { // EDX -> BA
              populateMap(s, ECPPerfAnalyzer.edx2BapMap);
            } else if (acks && s.contains("ACK for message:")) { // EDX -> BA
              populateMap(s, ECPPerfAnalyzer.edx2BapMap);
            } else if (s.contains("ID=MSG-01")) { // EDX -> ECP (4.14+)
              // no ACKS are being sent from EDX, the ACKS are all generated in ECP
              populateMap(s, ECPPerfAnalyzer.edx2EcpMap);
            }
          }
        }
        if (ecp) {
          if (tms < oldestEcpTms) {
            oldestEcpLdt = LocalDateTime.parse("" + tms, tmsF);
            oldestEcpTms = tms;
          }
          if (tms > newestEcpTms)
            newestEcpTms = tms;
        } else {
          if (tms < oldestEdxTms) {
            oldestEdxLdt = LocalDateTime.parse("" + tms, tmsF);
            oldestEdxTms = tms;
          }
          if (tms > newestEdxTms)
            newestEdxTms = tms;
        }
      }
    } catch (NumberFormatException |
             DateTimeParseException ex) {
      System.err.println("Error occurred: " + ex + " with input text: " + s);
    }
  }

  private static void populateMap(String s, TreeMap logMap) {
    TreeMap dateMap = (TreeMap) logMap.merge(Integer.parseInt(s.substring(0, 10).replaceAll("-", "")), new TreeMap<>(Comparator.naturalOrder()), (a, b) -> a != null ? a : b);
    TreeMap hourMap = (TreeMap) dateMap.merge(Integer.parseInt(s.substring(11, 13)), new TreeMap<>(Comparator.naturalOrder()), (a, b) -> a != null ? a : b);
    TreeMap minuteMap = (TreeMap) hourMap.merge(Integer.parseInt(s.substring(14, 16)), new TreeMap<Integer, Integer>(Comparator.naturalOrder()), (a, b) -> a != null ? a : b);
    minuteMap.merge(Integer.parseInt(s.substring(17, 19)), 1, (a, b) -> Integer.sum((Integer) a, (Integer) b));
  }

  private static TreeMap removeExcessFromMap(TreeMap logMap, LocalDateTime oldest, LocalDateTime newest) {
    TreeMap newLogMap = new TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, TreeMap<Integer, Integer>>>>();
    for (LocalDateTime tms = oldest; tms.compareTo(newest) < 0; tms = tms.plusSeconds(1)) {
      int date = Integer.parseInt(tms.format(dateF));
      int hour = tms.getHour();
      int minute = tms.getMinute();
      int second = tms.getSecond();
      TreeMap hourMap = (TreeMap) logMap.get(date);
      if (hourMap != null) {
        TreeMap minuteMap = (TreeMap) hourMap.get(hour);
        if (minuteMap != null) {
          TreeMap secondMap = (TreeMap) minuteMap.get(minute);
          if (secondMap != null) {
            Integer antPrSec = (Integer) secondMap.get(second);
            if (antPrSec != null) {
              TreeMap newDateMap = (TreeMap) newLogMap.merge(date, new TreeMap<>(Comparator.naturalOrder()), (a, b) -> a != null ? a : b);
              TreeMap newHourMap = (TreeMap) newDateMap.merge(hour, new TreeMap<>(Comparator.naturalOrder()), (a, b) -> a != null ? a : b);
              TreeMap newMinuteMap = (TreeMap) newHourMap.merge(minute, new TreeMap<Integer, Integer>(Comparator.naturalOrder()), (a, b) -> a != null ? a : b);
              newMinuteMap.merge(second, antPrSec, (a, b) -> Integer.sum((Integer) a, (Integer) b));
            }
          }
        }
      }
    }
    return newLogMap;
  }


  private static void heading() {
    System.out.println("-".repeat(115));
    System.out.println(String.format("%30s%30s  |%30s  |%20s", "Time", "SEND         ", "RECEIVE       ", "     CPU         "));
    System.out.println(String.format("%30s%10s%10s%10s  |%10s%10s%10s  |%10s%10s", "", "EDX", "ECP", "Diff", "ECP", "EDX", "Diff", " PERCENT ", "   LOAD "));
    System.out.println("-".repeat(115));
  }

  private static Integer print(LocalDateTime startTms, LocalDateTime endTms, Integer hourToPrint, boolean ecpRotate, boolean edxRotate, long diffSec, Long cpuPercent, Double loadAvg) {
    for (LocalDateTime tms = startTms; tms.compareTo(endTms) < 0; tms = tms.truncatedTo(ChronoUnit.MINUTES).plusMinutes(1)) {
      int tmsDate = Integer.parseInt(tms.format(dateF));
      int tmsHour = tms.getHour();
      int tmsMinute = tms.getMinute();

      String adjustStr = diffSec == 5 ? "" : "" + (diffSec - 5);
      if (hourToPrint < tmsHour || hourToPrint - 22 > tmsHour) {
        StringBuilder hourPrint = new StringBuilder();
        hourPrint.append(String.format("%30s", "   HOUR: " + tmsDate + "-" + String.format("%02d   ", hourToPrint)));
        String hEDX2ECP = hCount(edx2EcpMap, oldestEdxLdt, tms.minusHours(1));
        String hECP2BRO = hCount(ecp2BroMap, oldestEcpLdt, tms.minusHours(1));
        String hECP2EDX = hCount(ecp2EdxMap, oldestEcpLdt, tms.minusHours(1));
        String hEDX2BAP = hCount(edx2BapMap, oldestEdxLdt, tms.minusHours(1));
        String sendDiff = "N/A";
        String receiveDiff = "N/A";
        if (hEDX2ECP.trim().matches("[0-9]+") && hECP2BRO.trim().matches("[0-9]+"))
          sendDiff = "" + (Integer.parseInt(hEDX2ECP.trim()) - Integer.parseInt(hECP2BRO.trim()));
        if (hECP2EDX.trim().matches("[0-9]+") && hEDX2BAP.trim().matches("[0-9]+"))
          receiveDiff = "" + (Integer.parseInt(hECP2EDX.trim()) - Integer.parseInt(hEDX2BAP.trim()));
        hourPrint.append(String.format("%10s%10s%10s  |", hEDX2ECP, hECP2BRO, sendDiff));
        hourPrint.append(String.format("%10s%10s%10s  |", hECP2EDX, hEDX2BAP, receiveDiff));
        System.out.println(hourPrint);
        heading();
        hourToPrint = tmsHour;
      }
      StringBuilder minutePrint = new StringBuilder();
      minutePrint.append(String.format("%30s", adjustStr + " MINUTE: " + tmsDate + "-" + String.format("%02d:%02d", tmsHour, tmsMinute)));
      String cEDX2ECP = mCount(edx2EcpMap, oldestEdxLdt, tms, edxRotate);
      String cECP2BRO = mCount(ecp2BroMap, oldestEcpLdt, tms, ecpRotate);
      String cECP2EDX = mCount(ecp2EdxMap, oldestEcpLdt, tms, ecpRotate);
      String cEDX2BAP = mCount(edx2BapMap, oldestEdxLdt, tms, edxRotate);
      String sendDiff = "N/A";
      String receiveDiff = "N/A";
      if (cEDX2ECP.trim().matches("[0-9]+") && cECP2BRO.trim().matches("[0-9]+"))
        sendDiff = "" + (Integer.parseInt(cEDX2ECP.trim()) - Integer.parseInt(cECP2BRO.trim()));
      if (cECP2EDX.trim().matches("[0-9]+") && cEDX2BAP.trim().matches("[0-9]+"))
        receiveDiff = "" + (Integer.parseInt(cECP2EDX.trim()) - Integer.parseInt(cEDX2BAP.trim()));
      minutePrint.append(String.format("%10s%10s%10s  |", cEDX2ECP, cECP2BRO, sendDiff));
      minutePrint.append(String.format("%10s%10s%10s  |", cECP2EDX, cEDX2BAP, receiveDiff));
      minutePrint.append(String.format("%10s%10s   ", (cpuPercent == null ? "" : cpuPercent == -1 ? "N/A " : cpuPercent + " "), (loadAvg == null ? "" : loadAvg == -1d ? "N/A  " : String.format("%.2f  ", loadAvg))));
      System.out.println(minutePrint);
    }
    return hourToPrint;
  }

  private static String mCount(TreeMap map, LocalDateTime oldestTms, LocalDateTime tmsStart, boolean rotate) {
    if (oldestTms.compareTo(tmsStart.plusMinutes(1)) >= 0)
      return "- ";
    boolean complete = oldestTms.compareTo(tmsStart) > 0 ? false : true;

    Integer count = 0;
    TreeMap hourMap = (TreeMap) map.get(Integer.parseInt(tmsStart.format(dateF)));
    if (hourMap != null) {
      TreeMap minuteMap = (TreeMap) hourMap.get(tmsStart.getHour());
      if (minuteMap != null) {
        TreeMap secondMap = (TreeMap) minuteMap.get(tmsStart.getMinute());
        if (secondMap != null) {
          for (int second = 0; second < 60; second++) {
            Integer counter = (Integer) secondMap.get(second);
            if (counter != null)
              count += counter;
          }
        }
      }
    }
    return complete && !rotate ? count + " " : "(" + count + ")";
  }

  private static String hCount(TreeMap map, LocalDateTime oldestTms, LocalDateTime tmsStart) {
    if (oldestTms.compareTo(tmsStart.plusHours(1)) >= 0)
      return "- ";
    boolean complete = oldestTms.compareTo(tmsStart) > 0 ? false : true;

    Integer count = 0;
    TreeMap hourMap = (TreeMap) map.get(Integer.parseInt(tmsStart.format(dateF)));
    if (hourMap != null) {
      TreeMap minuteMap = (TreeMap) hourMap.get(tmsStart.getHour());
      if (minuteMap != null) {
        for (int minute = 0; minute < 60; minute++) {
          TreeMap secondMap = (TreeMap) minuteMap.get(minute);
          if (secondMap != null) {
            for (int second = 0; second < 60; second++) {
              Integer counter = (Integer) secondMap.get(second);
              if (counter != null)
                count += counter;
            }
          }
        }
      }
    }
    return complete ? count + " " : "(" + count + ")";
  }
}

