package no.statnett.ecp.pf;

import jakarta.jms.*;
import no.statnett.ecp.utils.Broker;

import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class Producer implements Runnable {

    private final SecureRandom rnd = new SecureRandom();


    private Map<String, String> amqpConfig;
    private String queueName;
    private Connection connection;
    private Session session;
    private MessageProducer producer;
    private byte[] randomData;
    private int queueSize = -1; // This value is updated from the controller (Performance-class)
    private int maxDiff = 0;
    private int ttl;
    private int msgSizeB;
    private int maxMBNotConsumed;
    private boolean persistence;
    private ProduceCounter produceCounter;


    public Producer(Map<String, String> amqpConfig, String queueName, int msgSizeB, int ttl, boolean persistence, ProduceCounter produceCounter, int maxMBNotConsumed) throws JMSException {
        this.amqpConfig = amqpConfig;
        this.queueName = queueName;
        init(amqpConfig, queueName);
        this.ttl = ttl;
        this.maxMBNotConsumed = maxMBNotConsumed;
        this.persistence = persistence;
        this.msgSizeB = msgSizeB;
        randomData = new byte[msgSizeB];
        rnd.nextBytes(randomData);
        this.produceCounter = produceCounter;
    }

    private void init(Map<String, String> amqpConfig, String queueName) throws JMSException {
        connection = Broker.createAndStartConnection(amqpConfig);
        session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        producer = session.createProducer(session.createQueue(queueName));
    }

    public void setRegulationInfo(int queueSize, int diff) {
        this.maxDiff = Math.max(queueSize, diff);
        this.queueSize = queueSize;
    }

    public void setMsgSizeKB(int msgSizeB) {
        this.randomData = new byte[msgSizeB];
        new SecureRandom().nextBytes(randomData);

    }

    // We measure two metrics in order to determine the speed of our production (the production is measured as 'transmitted' (completion), not just adding to local buffer)
    // 1. QueueSize
    // 2. QueueSizeBytes
    // 2. Bytes produced
    // We aim at not exceeding maxQueueSize,

    public void run() {
        int messageCount = 0;
        while (true) {
            try {
                long byteNotConsumed = (long) maxDiff * randomData.length;
                boolean initialization = queueSize < 0;
                boolean queueFull = ((long) queueSize * randomData.length) >= Performance.maxQueueSizeB;
                boolean bufferAboveLimit = produceCounter.aboveLimit(Performance.bufferLimitB);
                if (queueSize < 10000 && maxDiff < 10000 && byteNotConsumed < maxMBNotConsumed * 1024L * 1024L && !initialization && !queueFull && !bufferAboveLimit) { // Will stop producing as soon as we are above certain limits
                    messageCount = produce(messageCount);
                } else {
                    Thread.sleep(1);
                    produceCounter.increaseSkip();
                }
            } catch (Throwable t) {
                System.out.println("Producer experienced an error " + t + ", but why quit? The producer will re-create the connection and try again. But first sleep 30 seconds..");
                try {
                    Thread.sleep(30000);
                    if (this.producer != null) {
                        producer.close();
                    }
                    if (this.session != null) {
                        session.close();
                    }
                    if (this.connection != null) {
                        connection.close();
                    }
                    init(amqpConfig, queueName);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private int produce(int messageCount) throws JMSException {
        BytesMessage message = session.createBytesMessage();
        for (int i = 0; i < 4; i++) {
            int index = rnd.nextInt(randomData.length);
            randomData[index] = (byte) rnd.nextInt(256);
        }
        message.writeBytes(randomData);
        producer.send(message, persistence ? DeliveryMode.PERSISTENT : DeliveryMode.NON_PERSISTENT, Message.DEFAULT_PRIORITY, ttl, produceCounter);
        messageCount++;
        produceCounter.messageSent();
        return messageCount;
    }
}
