/*
 * Copyright (C) Red Hat, Inc.
 * http://www.redhat.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package no.statnett.ecp.qc;

import jakarta.jms.Queue;
import jakarta.jms.*;
import no.statnett.ecp.EKit;
import no.statnett.ecp.utils.*;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.time.LocalDateTime;
import java.util.*;

public class QueueCleaner {
    public static final String VERSION = "v3.0.3";


    private static boolean consume = false; // default
    private static boolean continuous = false;
    private static String filterInternalType;
    private static String filterMessageType;
    private static String queueNames;
    private static String filterSender;
    private static String filterReceiver;
    private static int filterTTL;
    private static File outputFile;

    public static void main(String[] args) throws Exception {

        List<String> mandatoryArgs = parseOptions(args);
        if (mandatoryArgs.isEmpty()) {
            usage();
            System.exit(1);
        }
        String logFilename = null;
        Map<String, String> config = null;
        if (mandatoryArgs.size() == 1) { // Documented - the proper way to use it, running ekit in same host as broker/ECP
            Options.checkDirectory(mandatoryArgs.get(0), false);
            config = URLParser.parse(mandatoryArgs.get(0), true);
            logFilename = config.get("log");
        } else { // Undocumented - this is used for dev/testing, by copying the ecp.log and connecting remotely to AMQP-broker
            logFilename = mandatoryArgs.get(0);
            Options.checkFile(logFilename, false); // Check that the log-file exists (and is readable
            config = URLParser.parse(mandatoryArgs.get(1), true);
        }

        boolean firstRound = true;
        do {
            if (firstRound) {
                firstRound = false;
            } else {
                Thread.sleep(10000);
            }
            String status = "OK";
            Map<String, Integer> noOfMsgPrQueue = new TreeMap<>();
            String[] queues = null;
            if (queueNames != null) { // Use queue names specified from options, without checking if they are valid - assume user knows
                queues = queueNames.split(",");
            } else {
                queues = QueueNameFinder.findQueueNamesFromECP414AndAbove(logFilename, config);
            }
            Connection connection = null;
            String qn = "Not-initialized";
            try {
                connection = Broker.createAndStartConnection(config);
                for (String queueName : queues) {
                    qn = queueName;
                    Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
                    Queue queue = session.createQueue(queueName);
                    String action = consume ? "Consumed" : "Browsed";
                    int noOfMsg = 0;
                    if (consume) {
                        MessageConsumer consumer = session.createConsumer(queue, Broker.buildSelector(filterMessageType, "" + filterTTL, filterSender, filterReceiver, filterInternalType));
                        while (true) {
                            Message message = consumer.receive(100);
                            if (message == null)
                                break;
                            message.acknowledge();
                            logMessage(message, queueName);
                            noOfMsg++;
                            noOfMsgPrQueue.merge(queueName, 1, Integer::sum);
                        }
                    } else {
                        QueueBrowser browser = session.createBrowser(queue, Broker.buildSelector(filterMessageType, "" + filterTTL, filterSender, filterReceiver, filterInternalType));
                        Enumeration<?> enum1 = browser.getEnumeration();
                        while (enum1.hasMoreElements()) {
                            Object melding = enum1.nextElement();
                            logMessage((Message) melding, queueName);
                            noOfMsg++;
                            noOfMsgPrQueue.merge(queueName, 1, Integer::sum);
                        }
                        browser.close();
                    }
                    System.out.println(info(queueName) + action + " " + noOfMsg + " messages with filter -m=" + filterMessageType + ", -t=" + filterTTL + ", -s=" + filterSender + ", -r=" + filterReceiver + ", -i=" + filterInternalType);
                    session.close();
                }
            } catch (Exception t) {
                System.err.println(error(qn) + "Error occurred: " + t);
                status = t.getMessage().replaceAll("\"", "");
            } finally {
                if (connection != null) {
                    try {
                        connection.close();
                    } catch (JMSException e) {
                        System.err.println(error(qn) + "Error occurred while close JMS-connection:" + e);
                    }
                }
            }
            logTotalSummary(status, noOfMsgPrQueue);
        } while (continuous);
    }


    private static List<String> parseOptions(String[] initialArgs) {
        List<String> argsList = new ArrayList<>(Arrays.asList(initialArgs));
        consume = Options.parseBoolean(argsList, "-c");
        continuous = Options.parseBoolean(argsList, "-f");
        filterInternalType = Options.parseString(argsList, "-i");
        filterMessageType = Options.parseString(argsList, "-m");
        queueNames = Options.parseString(argsList, "-q");
        filterSender = Options.parseString(argsList, "-s");
        filterReceiver = Options.parseString(argsList, "-r");
        filterTTL = Options.parseInt(argsList, "-t", 0, 86400 * 14, 0);
        String outputFileName = Options.parseString(argsList, "-o");
        if (outputFileName != null) {
            outputFile = Options.checkFile(outputFileName, true);
        }
        return argsList;
    }

    private static void logTotalSummary(String status, Map<String, Integer> antallMeldingerPrQueue) throws IOException {
        if (outputFile != null && outputFile.exists()) {
            StringBuilder sb = new StringBuilder();
            sb.append("{ \"statustime\": \"").append(LocalDateTime.now().format(Const.localTmsMillisec)).append("\", ");
            sb.append("\"ekit\": \"").append(EKit.VERSION).append("\", ");
            sb.append("\"status\": \"").append(status).append("\", ");
            sb.append("\"queues\": [");
            List<String> queueStrList = new ArrayList<>();
            for (String queueName : antallMeldingerPrQueue.keySet()) {
                queueStrList.add("{ \"queue\": \"" + queueName + "\", \"no_of_messages\": " + antallMeldingerPrQueue.get(queueName) + " }");
            }
            sb.append(String.join(",", queueStrList));
            sb.append("]}\n");
            Files.writeString(outputFile.toPath(), sb.toString(), StandardOpenOption.APPEND);
        }
        int numberOfMsgTotal = antallMeldingerPrQueue.values().stream().mapToInt(i -> i).sum();
        System.out.println(info("Summary") + String.format("%5s", numberOfMsgTotal) + " messages was " + (consume ? "consumed" : "browsed") + " in total");
    }


    private static String error(String queue) {
        return LogOut.e() + " [queue=" + String.format("%-55s", queue) + "] ";
    }

    private static String info(String queue) {
        return LogOut.i() + " [queue=" + String.format("%-55s", queue) + "] ";
    }

    private static String debug(String queue) {
        return LogOut.d() + " [queue=" + String.format("%-55s", queue) + "] ";
    }


    private static void logMessage(Message msg, String queueName) throws JMSException {
        String mt = EcpMsg.findMessageType(msg);
        Long ageSeconds = EcpMsg.calculateAgeSeconds(msg);
        String recevierCode = EcpMsg.getReceiver(msg);
        String sender = EcpMsg.getSender(msg);
        String senderBA = EcpMsg.getSenderBA(msg);
        String internalType = EcpMsg.getInternalType(msg);

        long hours = ageSeconds / 3600;
        long mins = (ageSeconds % 3600) == 0 ? 0 : (ageSeconds % 3600) / 60;
        String hourStr = hours + "h " + String.format("%02d", mins) + "m";
        String status = consume ? "Consumed" : "Browsed";

        System.out.println(debug(queueName) + "      " + status + ": MT=" + mt + ", Age=" + ageSeconds + " sec (" + hourStr + "), Sender=" + sender + ", Receiver=" + recevierCode + ", SenderBA=" + senderBA + ", InternalType=" + internalType);
    }

    private static void usage() {
        System.out.println("QueueCleaner " + VERSION + " is a tool to browse or consume messages from all queues on ECP or EDX - it can be used to monitor or to");
        System.out.println("maintain the queues on brokers of ECP and EDX. It must be have access to the confg+logs of ECP/EDX in order to determine the queues.");
        System.out.println("It can filter for certain messages, and by so doing, be able to browse/consume certain messages that you know of or get rid of.\n\n");
        System.out.println("Usage  : java -jar QueueCleaner.jar [OPTION] PATH\n");
        System.out.println("\n\nThe options and arguments:");
        System.out.println(" OPTION        : ");
        System.out.println("                 -c                 : consume/delete messges - default is browse mode");
        System.out.println("                 -f                 : continuous operation - will not exit, but repeat browse/consume every 10 seconds.");
        System.out.println("                 -i <internaltype>  : consume messages of a specific internalType (if not specified, all internalTypes will be consumed)");
        System.out.println("                 -m <messagetype>   : consume messages of a specific messageType (if not specified, all messageTypes will be consumed)");
        System.out.println("                 -q <queueNames>    : consume messages from a specific queues (comma-sep) (if not specified, all queues will be consumed)");
        System.out.println("                 -r <receiverCode>  : consume messages from a specific receiver (if not specified, all receivers will be consumed)");
        System.out.println("                 -s <senderCode>    : consume messages from a specific sender (if not specified, all senders will be consumed)");
        System.out.println("                 -t<sec>            : consume messages older than <sec> seconds (if not specified, msg of all ages will be consumed)");
        System.out.println("                 -o <filename>      : will append a one-liner summary in JSON-format to the <filename>");
        System.out.println(" PATH                               : Specify either the conf-path for ECP or EDX (ex: /etc/ecp-endpoint or /etc/edx-toolbox)");

        System.out.println("\nExample 1:  Consume all messages from standard ActiveMQ DLQ:");
        System.out.println("\tjava -jar QueueCleaner.jar -c -q ActiveMQ.DLQ /etc/ecp-endpoint");

        System.out.println("\nExample 2:  Consume older than 1h messages on edx.endpoint.reply (case: typically leftover msg that BA is not consuming):");
        System.out.println("\tjava -jar QueueCleaner.jar -c -t3600 -q edx.endpoint.reply /etc/edx-toolbox");

        System.out.println("\nExample 3:  Consume older than 1h messages on ecp.endpoint.inbox (case: msg stuck on-route from ECP to EDX):");
        System.out.println("\tjava -jar QueueCleaner.jar -c -t3600 -q ecp.endpoint.inbox /etc/ecp-endpoint");

        System.out.println("\nExample 4:  Browse first 400 messages from standard ActiveMQ DLQ:");
        System.out.println("\tjava -jar QueueCleaner.jar -q ActiveMQ.DLQ /etc/ecp-endpoint");

        System.out.println("\nExample 5:  Browse first 400 messages from all queues on EDX:");
        System.out.println("\tjava -jar QueueCleaner.jar /etc/edx-toolbox");

        System.out.println("\nTo log with UTC time, start command with 'java -Duser.timezone=GMT -jar QueueCleaner.jar....etc'");
        System.out.println("The keystore must contain at the very least a root-CA which QueueCleaner can trust when connecting to the broker");
        System.out.println("If the QueueCleaner fails, take a look at the jndi.properties file that it makes and examine the URL it uses");
    }

}