package no.statnett.ecp.qc;

import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

public class QueueNameFinder {
    public static String[] findQueueNamesFromECP414AndAbove(String logFilename, Map<String, String> config) throws NoSuchAlgorithmException, KeyManagementException, IOException {
        String[] queues;
        queues = findQueuesFromECP412AndBelow(logFilename);
        if (queues.length == 0) { // We could be on ECP 4.14 or later, thus no queues in the logs - we have to try to connect to the broker and get the queues
            // However, we cannot retrieve the queues from the AMQP-port, but we have to go to the HTTPS jolokia-backed console. Thus, we have to guess the
            // console-port. The default is 8161, so we simply try that one.
            String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
            queues = findQueueNamesFromECP414AndAbove(protocol, config.get("host"), "8161", config.get("user"), config.get("password"));
            if (queues.length == 0) {
                System.err.println("No queues specified with the -q options, nor was any queues found in the ecp.log (4.12 and below), nor from Artemis (4.14 and above)");
                System.exit(1);
            }
        }
        return queues;
    }

    static String[] findQueuesFromECP412AndBelow(String logFileName) {
        Set<String> queuesFromLogs = new HashSet<>();
        final AtomicInteger counter = new AtomicInteger(0);
        try (Stream<String> stream = Files.lines(Paths.get(logFileName), StandardCharsets.UTF_8)) {
            stream.forEach(line -> {
                counter.incrementAndGet();
                String searchStr = "queue://";
                if (line.contains(searchStr)) {
                    int startPos = line.indexOf(searchStr) + searchStr.length();
                    int endPos = line.indexOf(",", startPos);
                    if (endPos > -1) {
                        String qName = line.substring(startPos, endPos);
                        if (qName.endsWith(".")) {
                            qName = qName.substring(0, qName.length() - 1);
                        }
                        if (qName.matches("[a-zA-Z0-9-\\.]+"))
                            queuesFromLogs.add(qName);
                    }
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
        return queuesFromLogs.toArray(new String[]{});
    }

    public static boolean isUUID(String str) {
        try {
            UUID uuid = UUID.fromString(str);
            return true;
        } catch (IllegalArgumentException e) {
            return false;
        }
    }


    public static String[] findQueueNamesFromECP414AndAbove(String prot, String host, String port, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
        String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(prot, host, port, username, password);
        String json = ArtemisConsoleAPI.retrieveBrokerInfoJSON(prot, host, port, brokerNameEncoded, username, password);
        int queueNamesStartPos = json.indexOf("\"QueueNames\":");
        int queueNamesStartArrPos = json.indexOf("[", queueNamesStartPos) + 1;
        int queueNamesEndArrPos = json.indexOf("]", queueNamesStartArrPos);
        String queueNames = json.substring(queueNamesStartArrPos, queueNamesEndArrPos);
        Set<String> queueNamesSet = new HashSet<>();
        for (String queueName : queueNames.split(",")) {
            String qn = queueName.replaceAll("\"", "").replaceAll("\\\\", "");
            if (isUUID(qn)) {
                // Artemis occasionally returns UUIDs as queue-names, we do not want to do anything with these names, as soon as we browse/creates them - they will be auto-created
                // The hope is that by avoiding to return this strange queues, we will avoid to create them
                System.out.println(LocalDateTime.now().format(Const.localTmsSec) + " DEBUG Found the qn: " + qn + ", but it is a UUID, thus we skip it");
                continue;
            }
            queueNamesSet.add(qn);
        }
        return queueNamesSet.toArray(new String[]{});
    }
}
