/*
 * Copyright (C) Red Hat, Inc.
 * http://www.redhat.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package no.statnett.ecp.qp;

import jakarta.jms.*;
import no.statnett.ecp.utils.LogOut;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Restore {


    public static int restore(Session session, Queue queue, Map<String, Map<String, File>> restoreMessages, long startTime, File directory, long waitMs) throws JMSException, IOException {
        MessageProducer producer = session.createProducer(queue);
        int restoreCounter = 0;
        for (String name : restoreMessages.keySet()) {
            Map<String, File> filesPrExt = restoreMessages.get(name);
            Message message = Restore.buildMessage(session, filesPrExt, directory, name);
            if (message != null) {
                producer.send(message, DeliveryMode.PERSISTENT, Message.DEFAULT_PRIORITY, 1000000);
                for (File file : filesPrExt.values())
                    Files.delete(file.toPath());
                restoreCounter++;
            }
            if (System.currentTimeMillis() - startTime > 10000) {
                break; // Max 10 seconds - then we evaluate if the queue is looking good (not filling up again)
            }
            try {
                Thread.sleep(waitMs);
            } catch (InterruptedException e) {
                // do nothing
            }
        }
        producer.close();
        return restoreCounter;
    }


    public static Map<String, Map<String, File>> findMessagesToRestore(File directory) {
        Map<String, Map<String, File>> msgToRestore = new HashMap<>();
        File[] filesInDir = directory.listFiles();
        if (filesInDir != null) {
            for (File file : filesInDir) {
                // Example of filename we get from ecp.endpoint.download queue - other queues may have other formats, but follow the same pattern
                // QP-50V000000000115O_50V-SN-TEST--ATK_BATCH-25-FILE-80-1KB_fb497a59-121b-479a-9e4a-26e53c07eebc.headers.txt
                // QP-50V000000000115O_50V-SN-TEST--ATK_BATCH-25-FILE-80-1KB_fb497a59-121b-479a-9e4a-26e53c07eebc.metadata.xml
                // QP-50V000000000115O_50V-SN-TEST--ATK_BATCH-25-FILE-80-1KB_fb497a59-121b-479a-9e4a-26e53c07eebc.payload.bin

                // If file starts with "QP-" and is at least one minute old we'll restore it
                if (file.isFile() && file.getName().startsWith("QP-") && file.lastModified() < System.currentTimeMillis() - 60000) {
                    // For each file, determine the first part of the filename (before the first dot) and the second part (after the first dot)
                    int firstDot = file.getName().indexOf(".");
                    String name = file.getName().substring(0, firstDot);
                    String ext = file.getName().substring(firstDot + 1);
                    msgToRestore.computeIfAbsent(name, k -> new HashMap<>()).put(ext, file);
                }
            }
        }
        return msgToRestore;
    }

    public static Message buildMessage(Session session, Map<String, File> filesPrExt, File directory, String name) throws IOException, JMSException {
        if (filesPrExt.size() == 3) { // most likely the ecp.endpoint.download-queue
            return buildEcpEndpointDownloadMessage(session, filesPrExt);
        } else {
            System.out.println(LogOut.e() + "Did not find 3 files for message " + name + " as expected, in directory " + directory.getAbsolutePath());
            return null;
        }
    }

    public static Message buildEcpEndpointDownloadMessage(Session session, Map<String, File> filesPrExt) throws JMSException, IOException {
        StreamMessage message = session.createStreamMessage();
        File headersFile = filesPrExt.get("headers.txt");
        // read file and set properties on message
        List<String> headers = Files.readAllLines(headersFile.toPath());

        // Example of headers we get from ecp.endpoint.download queue
        //java.lang.String messageID=fb497a59-121b-479a-9e4a-26e53c07eebc
        //java.lang.String ecpMessageDirection=DOWN
        //java.lang.String receiverCode=50V-SN-TEST--ATK
        //java.lang.Integer messageMversion=2
        //java.lang.String internalType=STANDARD_MESSAGE
        //java.lang.String senderCode=50V000000000115O
        //java.lang.Long CamelMessageTimestamp=1706821214895
        //java.lang.String messageType=BATCH-25-FILE-80-1KB
        //java.lang.Long generated=1706821214718
        //java.lang.String receivedFromComponent=50V000000000112U
        //java.lang.String senderApplication=
        //java.lang.String baMessageID=BATCH-25-FILE-80
        //java.lang.String JMS_AMQP_MESSAGE_ANNOTATIONS={ "x-opt-jms-msg-type" : { "byte" : 4 }, "x-opt-jms-dest" : { "byte" : 0 } }

        for (String header : headers) {
            String type = header.substring(0, header.indexOf(" "));
            String name = header.substring(header.indexOf(" ") + 1, header.indexOf("="));
            String value = header.substring(header.indexOf("=") + 1);
            if (type.equals("java.lang.String")) {
                message.setStringProperty(name, value);
            } else if (type.equals("java.lang.Integer")) {
                message.setIntProperty(name, Integer.parseInt(value));
            } else if (type.equals("java.lang.Long")) {
                message.setLongProperty(name, Long.parseLong(value));
            } else if (type.equals("java.lang.Boolean")) {
                message.setBooleanProperty(name, Boolean.parseBoolean(value));
            } else {
                System.out.println(LogOut.w() + "Did not recognize header type " + type + " for header " + name);
            }
        }

        File metadataFile = filesPrExt.get("metadata.xml");
        String metadata = Files.readString(metadataFile.toPath());
        message.writeString(metadata);

        File payloadFile = filesPrExt.get("payload.bin");
        byte[] payload = Files.readAllBytes(payloadFile.toPath());
        message.writeBytes(payload);

        return message;
    }
}
