package no.statnett.ecp.utils;

import jakarta.jms.JMSException;
import jakarta.jms.Message;

import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;

public class EcpMsg {
    public static String findMessageType(Message msg) throws JMSException {
        return getHeader(msg, "businessType", "messageType", "EDX_MESSAGE_TYPE");
    }

    public static String getMessageID(Message msg) throws JMSException {
        return getHeader(msg, "messageID", "EDX_MESSAGE_ID");
    }

    public static String getReceiver(Message msg) throws JMSException {
        return getHeader(msg, "receiverCode", "EDX_RECEIVER");
    }

    public static String getSender(Message msg) throws JMSException {
        return getHeader(msg, "senderCode", "sender", "EDX_SENDER_BA");
    }

    public static String getSenderBA(Message msg) throws JMSException {
        return getHeader(msg, "senderApplication", "EDX_SENDER_BA");
    }


    public static String getInternalType(Message msg) throws JMSException {
        return getHeader(msg, "internalType", "EDX_MESSAGE_KIND");
    }

    private static String getHeader(Message msg, String... headerNames) throws JMSException {
        for (String headerName : headerNames) {
            String header = msg.getStringProperty(headerName);
            if (header != null) {
                return header;
            }
        }
        return null;
    }

    public static ZonedDateTime findZDT(Message msg) {
        try {
            if (msg == null)
                return null;
            // Choose oldest of JMSTimestamp and generated
            long tms = msg.getJMSTimestamp();
            String generated = msg.getStringProperty("generated");
            if (generated != null && Long.parseLong(generated) < tms) {
                tms = Long.parseLong(generated);
            }
            return Instant.ofEpochMilli(tms).atZone(ZoneOffset.UTC);
        } catch (JMSException e) {
            return null;
        }
    }

    public static long calculateAgeSeconds(Message msg) {
        ZonedDateTime zdt = findZDT(msg);
        long ageSeconds = 0;
        if (zdt != null) {
            ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC);
            ageSeconds = ChronoUnit.SECONDS.between(zdt, now);
        }
        return ageSeconds;
    }

}
