package no.statnett.ecp.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import java.util.zip.GZIPOutputStream;

import static java.time.ZoneOffset.UTC;

/*
 * Why write code to rotate logs? Why not use standard log rotation tools? The answer is
 * 1. We want to avoid property/xml-config, instead doing config through code -> difficult to understand how to do this
 * 2. We want as few libraries as possible (minimize dependencies/attack-vectors)
 * 3. We want to rotate based on the content/timestamp inside the log, not the clock-time of the system
 * 4. We want to absolutely sure that no rotation happens while writing (probably solved by a library anyway - but we want to be *absolutely* sure)
 * 5. We want to always keep N days of logs before rotating, to avoid sometimes having a 0-byte or small-sized log
 * 6. We want to split a long log (that is not rotated) into multiple files based on the timestamp in the log
 * 7. We want to handle both Local-timestamps and UTC-timestamps in logs (with and without 'Z' at the end)
 * 8. We want to rotate regardless of actual writing to the file (probably related to 4. - you can have 4, but then not 8)
 * 9. We want to handle different kinds of timestamps in the log file we're rotating
 */
public class LogRotator {

    public static final String FULLTMSREGEXP = "^\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}.*";

    private final File logFile;
    private final Predicate<String> lineFilter;
    private final Function<String, String> timestampExtractor;
    private final DateTimeFormatter timestampFormat;
    private final ZonedDateTime newest;
    private final int noOfDaysDelayRotation;
    private final DebugLog debugLog;
    private final boolean utc;

    // Constructor for CC (ConnectivityCheck) and QM (QueueMonitor)
    public LogRotator(File logFile, ZonedDateTime newest, int noOfDaysDelayRotation) {
        // A line from cc-log starts like this:
        // {"statustime": "2024-08-13 09:05:24"...,
        this.logFile = logFile;
        this.lineFilter = s -> s.length() > 35;  // ignorerer linjer som er garantert for korte
        this.timestampExtractor = s -> s.substring(16, 35);
        this.newest = newest;
        this.noOfDaysDelayRotation = noOfDaysDelayRotation;
        this.debugLog = new DebugLog(null, false);
        this.timestampFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        this.utc = false;
    }

    // Constructor used for EML (EcpMessageLogger)
    public LogRotator(File logFile, ZonedDateTime newest, int noOfDaysDelayRotation, DebugLog debugLog) {
        // A line from eml-log starts like this:
        // 2024-08-15T08:52:30.914Z....
        this.logFile = logFile;
        this.lineFilter = s -> s.length() > 23 && s.matches(FULLTMSREGEXP); // ecp.log kan inneholde stack-trace som ikke matcher
        this.timestampExtractor = s -> s.substring(0, 23);
        this.timestampFormat = null; // Will lead to default timestamp-format (ISO: 2024-05-09T07:01:57.960)
        this.newest = newest;
        this.noOfDaysDelayRotation = noOfDaysDelayRotation;
        this.debugLog = debugLog;
        this.utc = true;
    }

    public void deleteExcessLogs(int maxNoOfLogFiles) {
        File[] rotatedLogFiles = getRotatedLogFiles();
        int numberOfFilesToDelete = rotatedLogFiles.length - maxNoOfLogFiles;
        if (numberOfFilesToDelete > 0) {
            List<File> files = new java.util.ArrayList<>(List.of(rotatedLogFiles));
            files.sort(Comparator.comparingLong(File::lastModified));
            // files is now sorted by lastModified, so that the oldest files are first

            for (int i = 0; i < numberOfFilesToDelete; i++) {
                try {
                    // delete the oldest files first
                    Files.delete(files.get(i).toPath());
                    debug("Log-file " + files.get(i).getName() + " deleted since it's an excess log-file (limit is " + maxNoOfLogFiles + ")");
                } catch (IOException e) {
                    debug("Error occurred during deletion of log-file " + files.get(i).getName() + ": " + e.getMessage());
                }
            }
        }
    }

    public void compress() {
        File[] rotatedLogFiles = getRotatedLogFiles();
        // Compress logFile on the form
        // logFile-20240509.log to logFile-20240509.log.gz
        // logFile-2024W19.log to logFile-2024W19.log.gz
        // logFile-202405.log to logFile-202405.log.gz
        for (File rotatedLogFile : rotatedLogFiles) {
            if (!rotatedLogFile.getName().endsWith(".gz")) {
                gzip(rotatedLogFile);
            }
        }
    }

    private void gzip(File sourceFile) {
        File gzipFile = new File(sourceFile.getAbsolutePath() + ".gz");

        try (FileInputStream fis = new FileInputStream(sourceFile);
             FileOutputStream fos = new FileOutputStream(gzipFile);
             GZIPOutputStream gzipOS = new GZIPOutputStream(fos)) {

            byte[] buffer = new byte[1024];
            int len;
            while ((len = fis.read(buffer)) != -1) {
                gzipOS.write(buffer, 0, len);
            }

        } catch (IOException e) {
            debug("Error occurred during compression of log-file " + sourceFile.getName() + ": " + e.getMessage());
        }
        // delete source file
        try {
            Files.delete(sourceFile.toPath());
            debug("Log-file " + sourceFile.getName() + " compressed to " + gzipFile.getName() + " and then deleted");
        } catch (IOException e) {
            debug("Error occurred during deletion of log-file (excess log-file) " + sourceFile.getName() + ": " + e.getMessage());
        }


    }

    private File[] getRotatedLogFiles() {
        int dotPos = logFile.getName().indexOf(".");
        // All rotated log files have starts this way (logFile-20240509.log or logFile-2024W19.log or logFile-202405.log), this code is safe until year 3000
        String prefix = logFile.getName().substring(0, dotPos) + "-2";
        File directory = logFile.getParentFile();
        if (directory == null) {
            directory = new File(".");
        }
        if (directory.exists() && directory.isDirectory()) {
            return directory.listFiles((dir, name) -> name.startsWith(prefix));
        }
        return new File[]{};
    }


    private ZonedDateTime getZDT(String timestamp, DateTimeFormatter format) {
        if (format == null) // Use default format (ISO-standard: 2024-05-09T07:01:57.960)
            return LocalDateTime.parse(timestamp).atZone(utc ? UTC : ZoneId.systemDefault());
        else
            return LocalDateTime.parse(timestamp, format).atZone(utc ? UTC : ZoneId.systemDefault());

    }

    public void rotateLogs(String rotationType) {

        // Read from beginning of logFile and find first timestamp on the form 2024-05-09T07:01:57.960Z, return this timestamp as ZonedDateTime
        ZonedDateTime firstTmsZDT = findFirstTimestampInLogFile(logFile, lineFilter, timestampExtractor, timestampFormat);
        if (firstTmsZDT == null)
            return;
        // Find the last timestamp written to the log
        ZonedDateTime endLoggingMinuteZDT = newest.truncatedTo(ChronoUnit.MINUTES);
        // Subtract noOfDaysDelayRotation from endLoggingMinuteZDT
        endLoggingMinuteZDT = endLoggingMinuteZDT.minusDays(noOfDaysDelayRotation);
        if (endLoggingMinuteZDT.isBefore(firstTmsZDT)) {
            debug("No rotation needed for log-file " + logFile.getName() + " since endLoggingMinuteZDT - " + noOfDaysDelayRotation + " days is " + endLoggingMinuteZDT + " and is before firstTmsZDT (" + firstTmsZDT + ")");
            return;
        }

        // rotationType can be day, week or month
        String timeFormat = getTimeFormat(rotationType);
        String startTimeId = firstTmsZDT.format(DateTimeFormatter.ofPattern(timeFormat));
        String endTimeId = endLoggingMinuteZDT.format(DateTimeFormatter.ofPattern(timeFormat));
        if (!startTimeId.equals(endTimeId)) {
            // Read all lines in file and write each line to a new file with the date in the filename.
            // Discard lines that do not match timestamp-regexp in the beginng of the line
            // Each time a new currentTimeId is found, create a new file and write to this file until the next date is found
            // When endTimeId is found, write to this file until the endTimeId of the file, but keep the logname as it is
            try {
                // First rotation-file startTimeId with startTimeId, last will be the same filename as logFile with ".tmp" extension
                File rotationFile = createFile(logFile, startTimeId, endTimeId);
                String previousTimeId = null;
                Stream<String> lines = Files.lines(logFile.toPath());
                StringBuilder buffer = new StringBuilder(); // Will hold the lines to be written to the rotationFile, when buffer.length() > 1000000 (write line by line is slow)
                for (String line : (Iterable<String>) lines::iterator) {
                    if (lineFilter.test(line)) {
                        String currentTimeId = getZDT(timestampExtractor.apply(line), timestampFormat).format(DateTimeFormatter.ofPattern(timeFormat, new Locale("no", "NO")));
                        // If the rotation file is a tmp file, then we have reached the end of the ecp-logs (because endTimeId is reached)
                        // and we should write all the rest of the eml-file to the tmp-file (which will in the end be renamed to the original log-file)
                        // If it's not a tmp file, then we should create a new file if the currentTimeId is different from the previousTimeId
                        if (!rotationFile.getName().endsWith(".tmp") && previousTimeId != null && !currentTimeId.equals(previousTimeId)) {
                            if (buffer.length() > 0) {
                                Files.writeString(rotationFile.toPath(), buffer.toString(), StandardOpenOption.APPEND);
                                buffer.setLength(0);
                                debug("Rotation of  log-file " + rotationFile.getName() + " completed, since currentTimeId (" + currentTimeId + ") is different from previousTimeId (" + previousTimeId + ") based on timestamp " + timestampExtractor.apply(line));
                            }
                            rotationFile = createFile(logFile, currentTimeId, endTimeId);
                        }
                        buffer.append(line).append("\n");
                        if (buffer.length() > 1000000) {
                            Files.writeString(rotationFile.toPath(), buffer.toString(), StandardOpenOption.APPEND);
                            buffer.setLength(0);
                        }
                        previousTimeId = currentTimeId;
                    }
                }
                if (buffer.length() > 0) {
                    Files.writeString(rotationFile.toPath(), buffer.toString(), StandardOpenOption.APPEND);
                    buffer.setLength(0);
                    debug("Rotation of  log-file " + rotationFile.getName() + " completed");
                }
                try {
                    // When completed - delete logFile
                    Files.delete(logFile.toPath());
                    // If lastRotationFile has a ".tmp" extension, rename it to the original logFile, else create a new file with the original logFile name
                    if (rotationFile.getName().endsWith(".tmp")) {
                        Files.move(rotationFile.toPath(), logFile.toPath());
                    } else {
                        Files.createFile(logFile.toPath());
                    }
                } catch (Throwable t) {
                    debug("Error occurred during delete or renameing of log-file " + rotationFile.getName() + " to " + logFile.getName() + " - the process will run once more next minute: " + t.getMessage());
                }
            } catch (IOException e) {
                debug("Error occurred during rotation of log-file " + logFile.getName() + ", a new attempt will begun next minute: " + e.getMessage());
            }
        }
    }

    private ZonedDateTime findFirstTimestampInLogFile(File logFile, Predicate<String> lineFilter, Function<String, String> timestampExtractor, DateTimeFormatter format) {
        try {
            String firstLoglineWithTms = Files.lines(logFile.toPath(), StandardCharsets.UTF_8).filter(lineFilter).findFirst().orElse(null);
            if (firstLoglineWithTms != null) {
                // Use timestampExtractor to find the timestamp-part:
                return getZDT(timestampExtractor.apply(firstLoglineWithTms), format);
            } else {
                debug("No timestamp found in log-file " + logFile.getName() + " - will not rotate log-file");
            }
        } catch (IOException e) {
            debug("Error occurred during reading of log-file during findFirstTimestampInLogFile()" + logFile.getName() + " - will not rotate log-file: " + e.getMessage());
        }
        return null;
    }

    private String getTimeFormat(String rotationType) {
        String timeFormat = "yyyyMMdd";
        if (rotationType.equals("week")) {
            timeFormat = "yyyy'W'ww";
        } else if (rotationType.equals("month")) {
            timeFormat = "yyyyMM";
        }
        return timeFormat;
    }

    private File createFile(File mainLogFile, String currentTimeId, String endTimeId) throws IOException {
        int dotPos = mainLogFile.getName().indexOf("."); // Find the position of the '.' in the filename - we need this to create new filenames
        File rotationFile = new File((mainLogFile.getParent()  == null ? "./" : mainLogFile.getParent() + "/") + getNewFilename(mainLogFile, dotPos, currentTimeId, endTimeId));
        if (rotationFile.exists()) { // Delete file if it already exists, so we start with an empty file
            Files.delete(rotationFile.toPath());
        }
        rotationFile.createNewFile();
        return rotationFile;
    }

    private String getNewFilename(File logFile, int dotPos, String date, String endDate) {
        if (date.equals(endDate)) {
            return logFile.getName() + ".tmp";
        } else {
            return logFile.getName().substring(0, dotPos) + "-" + date + logFile.getName().substring(dotPos);
        }
    }

    private void debug(String message) {
        debugLog.debug(message);
    }

}
