package no.statnett.ecp.utils;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class URLParser {
    /**
     * This method will find the configuration for the AMQP-broker. It can take two types of input:
     * 1) A URL to the broker (ex: amqps://localhost:5672)
     * 2) A path to a directory containing ecp.properties or edx.properties
     * Based on the input, the method will return a map with the following data:
     * <p>
     * url: The URL to the broker (ex: amqps://localhost:5672) - the URL is without user/pass because that's how the library wants it
     * user: The user to connect with the AMQP-broker, I think it's safe to set it even if the broker does not require it
     * password: The password to connect with the AMQP-broker, I think it's safe to set it even if the broker does not require it
     * host: The host part of the URL - yes, it founds in the URL also, but we can skip some parsing by having it here
     * port: The port part of the URL - yes, it founds in the URL also, but we can skip some parsing by having it here
     * Only for scenario 1) - URL to the broker:
     * suggest: If the user has specified the URL directly, this key will be set to "tls" or "no-tls" based on the URL. This
     * is used to suggest the opposite URL if the first attempt fails (so it doesn't matter as much if the wrong scheme
     * is given in the URL)
     * Only for scenario 2) - Path to a directory containing ecp.properties or edx.properties:
     * ecp: If the path was to a directory containing ecp.properties, this key will be set to "true"
     * edx: If the path was to a directory containing edx.properties, this key will be set to "true"
     * yml: If the path was to a directory containing edx.properties, this key will be set to the path of the edx.yml file
     * logs: The path to the log file for the endpoint
     *
     * @param urlPathArg
     * @return
     * @throws IOException
     */
    public static Map<String, String> parse(String urlPathArg, boolean amqpRequired) throws IOException {
        Map<String, String> configMap = new HashMap<>();
        if ((urlPathArg.startsWith("amqp") && amqpRequired) || (urlPathArg.startsWith("http") && !amqpRequired)) { // The URL has been given directly - no need to parse ecp/edx.properties
            // The url may look like this: amqp://endpoint:password@host:5672 (or starting with amqps)
            // OR
            // The url may look like this: http://endpoint:password@host:8161 (or starting with https)
            // We must remove user and password from the URL and put them in the configMap
            int userStart = urlPathArg.indexOf("//") + 2;
            int passStart = urlPathArg.indexOf(":", userStart) + 1;
            int hostStart = urlPathArg.indexOf("@") + 1;
            int portStart = urlPathArg.indexOf(":", Math.max(userStart, hostStart)) + 1; // should always find a port number
            try {
                if (portStart > 0) {
                    Integer.parseInt(urlPathArg.substring(portStart));
                }
            } catch (NumberFormatException e) {
                System.out.println("This part '" + urlPathArg.substring(portStart) + "' was assumed to be the port number, but is not a valid port number");
                if (amqpRequired)
                    System.out.println("The URL must be like this 'amqp[s]://[user:pass@]host:port' (ex: amqp://localhost:5672) - exiting.");
                else
                    System.out.println("The URL must be like this 'http[s]://[user:pass@]host:port' (ex: https://endpoint:password@localhost:8161) - exiting.");
                System.exit(1);
            }

            configMap.put("protocol", urlPathArg.substring(0, userStart - 3));

            if (userStart > 6 && passStart > userStart && hostStart > passStart && portStart > hostStart) {
                configMap.put("user", urlPathArg.substring(userStart, passStart - 1));
                configMap.put("password", urlPathArg.substring(passStart, hostStart - 1));
                configMap.put("host", urlPathArg.substring(hostStart, portStart - 1));
                configMap.put("port", urlPathArg.substring(portStart));
                configMap.put("url", urlPathArg.substring(0, userStart) + urlPathArg.substring(hostStart)); // Remove user and pass from URL
                if (urlPathArg.startsWith("amqps") || urlPathArg.startsWith("https")) {
                    configMap.put("suggest", "tls");
                } else {
                    configMap.put("suggest", "no-tls");
                }
            } else if (userStart > 6 && portStart > userStart) {
                configMap.put("host", urlPathArg.substring(userStart, portStart - 1));
                configMap.put("port", urlPathArg.substring(portStart));
                configMap.put("url", urlPathArg); // Remove user and pass from URL
            } else {
                if (amqpRequired)
                    System.out.println("URL must be like this 'amqp[s]://[user:pass@]host:port' (ex: amqp://localhost:5672) - exiting.");
                else
                    System.out.println("URL must be like this 'http[s]://[user:pass@]host:port' (ex: https://endpoint:password@localhost:8161) - exiting.");
                System.exit(1);
            }
        } else {
            String url = "";
            File f = new File(urlPathArg);
            if (f.isDirectory()) {
                File ecpPropFile = new File(urlPathArg + "/" + "ecp.properties");
                if (ecpPropFile.exists()) { // ECP-config
                    Options.checkFile(ecpPropFile.getAbsolutePath(), false);
                    String host = findHost(urlPathArg);
                    configMap.put("host", host);
                    List<String> lines = Files.readAllLines(ecpPropFile.toPath());
                    Map<String, String> map = makeMap(lines);
                    String auth = getProp(map, "internalBroker.useAuthentication", "true", true);
                    if (auth.equals("true")) {
                        url += "amqps://" + host + ":";
                    } else {
                        url += "amqp://" + host + ":";
                    }
                    String port = getProp(map, "internalBroker.amqp.port", "5672");
                    configMap.put("port", port);
                    url += port;
                    configMap.put("url", url);
                    configMap.put("user", getProp(map, "internalBroker.auth.user", "endpoint"));
                    configMap.put("password", getProp(map, "internalBroker.auth.password", "password"));
                    configMap.put("ecp", "true");
                    String ecpLog = getProp(map, "logging.file.name", "/var/log/ecp-endpoint/ecp.log", false);
                    Options.checkFile(ecpLog, false);
                    configMap.put("log", ecpLog);
                } else {
                    File edxPropFile = new File(urlPathArg + "/" + "edx.properties");
                    if (edxPropFile.exists()) { // EDX-config
                        Options.checkFile(edxPropFile.getAbsolutePath(), false);
                        String host = findHost(urlPathArg);
                        configMap.put("host", host);
                        List<String> lines = Files.readAllLines(edxPropFile.toPath());
                        Map<String, String> map = makeMap(lines);
                        String auth = getProp(map, "internalBroker.useAuthentication", "true", true);
                        if (auth.equals("true")) {
                            url += "amqps://" + host + ":";
                        } else {
                            url += "amqp://" + host + ":";
                        }
                        String port = getProp(map, "internalBroker.amqp.port", "5672");
                        configMap.put("port", port);
                        url += port;
                        configMap.put("url", url);
                        configMap.put("user", getProp(map, "internalBroker.auth.user", "toolbox"));
                        configMap.put("password", getProp(map, "internalBroker.auth.password", "password"));
                        configMap.put("edx", "true");
                        configMap.put("yml", urlPathArg + "/" + "edx.yml");
                        String edxLogs = getProp(map, "logging.file.name", "/var/log/edx-toolbox/edx.log", false);
                        Options.checkFile(edxLogs, false);
                        configMap.put("log", edxLogs);

                    } else {
                        throw new RuntimeException("Did not find ecp.properties nor edx.properties in " + urlPathArg);
                    }
                }
            } else {
                throw new RuntimeException("Argument " + urlPathArg + " is not a directory, if you intend it to be a URL make sure it starts with 'amqp'");
            }
        }
        return configMap;
    }

    private static String findHost(String urlPathArg) {
        try {
            return Files.readString(Path.of(urlPathArg + "/host")).trim();
        } catch (Throwable t) {
            return "localhost";
        }
    }

    private static String getProp(Map<String, String> map, String name, String defaultValue) {
        return getProp(map, name, defaultValue, false);
    }

    private static String getProp(Map<String, String> map, String name, String defaultValue, boolean toLowerCase) {
        if (map.get(name) != null) {
            // We may receive a property that references another property through the syntax ${property-name} - we must resolve this
            // A simple way is to resolve the first reference we encounter, then replace the reference with the resolved value - and then repeat until no ref is found
            String value = map.get(name);
            int start = value.indexOf("${");
            while (start > -1) {
                int end = value.indexOf("}", start);
                String ref = value.substring(start + 2, end);
                String refValue = map.get(ref);
                if (refValue == null) {
                    throw new RuntimeException("Property " + name + " references a property " + ref + " which is not found in the same property-file!");
                }
                value = value.substring(0, start) + refValue + value.substring(end + 1);
                start = value.indexOf("${");
            }
            return toLowerCase ? value.trim().toLowerCase() : value.trim();
        } else {
            return defaultValue;
        }
    }

    private static Map<String, String> makeMap(List<String> lines) {
        Map map = new HashMap();
        for (String line : lines) {
            if (!line.trim().startsWith("#") && line.contains("=") && line.split("=").length == 2) {
                map.put(line.split("=")[0].trim(), line.split("=")[1].trim());
            }
        }
        return map;
    }
}
