package no.statnett.ecp.brs.actions;

import no.statnett.ecp.brs.Config;
import no.statnett.ecp.brs.Div;
import no.statnett.ecp.brs.state.*;
import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;
import org.json.JSONArray;
import org.json.JSONObject;

import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ListConsumers {

  public static class Print {
    public static void printCI(List<ConsumerInfo> consumers, boolean echo) {
      int i = 1;

      if (echo) {
        String hostHd = "Host (1)";  // Never null
        String queueHd = "Queue (2)"; // Never null
        String creationHd = "Creation (3)"; // Never null
        String lastMessageHd = "LastMessage (4)"; // Can be null
        String connIdHd = "Conn-ID (5)"; // Never null
        String inTransitHd = "Trans (6)"; // // Never null
        String ackHd = "Rcv (7)"; // // Never null
        String ipHd = "IP (8)"; // Never null
        String portHd = "Port (9)"; // Never null

        // Compute dynamic widths
        List<String> creationStrs = new ArrayList<>();
        for (ConsumerInfo c : consumers) {
          String creationStr = c.getCreationTime() > 0
              ? java.time.Instant.ofEpochMilli(c.getCreationTime()).atZone(java.time.ZoneId.systemDefault()).toLocalDateTime().format(Const.localTmsSec)
              : "";
          creationStrs.add(creationStr);
        }

        int hostWidth = Math.max(consumers.stream().mapToInt(c -> c.getHost() == null ? 0 : c.getHost().length()).max().orElse(0), hostHd.length());
        int queueWidth = Math.max(consumers.stream().mapToInt(c -> c.getQueueName() == null ? 0 : c.getQueueName().length()).max().orElse(0), queueHd.length());
        int creationWidth = 19; //tms
        int lastMsgWidth = 19; // tms
        int connIdWidth = Math.max(consumers.stream().mapToInt(c -> c.getConnectionID() == null ? 0 : c.getConnectionID().length()).max().orElse(0), connIdHd.length());
        int inTransitWidth = Math.max(consumers.stream().mapToInt(c -> ("" + c.getInTransitCount()).length()).max().orElse(0), inTransitHd.length());
        int ackWidth = Math.max(consumers.stream().mapToInt(c -> ("" + c.getAcknowledgedCount()).length()).max().orElse(0), ackHd.length());
        int ipWidth = Math.max(consumers.stream().mapToInt(c -> ("" + c.getIpAddress()).length()).max().orElse(0), ipHd.length());
        int portWidth = Math.max(consumers.stream().mapToInt(c -> ("" + c.getPort()).length()).max().orElse(0), portHd.length());

        int totalWidth = hostWidth + queueWidth + creationWidth + lastMsgWidth + connIdWidth + inTransitWidth + ackWidth + ipWidth + portWidth + 4 + 9 * 3;

        // Header
        System.out.println(String.format("%-4s | %-" + hostWidth + "s | %-" + queueWidth + "s | %-" + creationWidth + "s | %-" + lastMsgWidth + "s | %-" + connIdWidth + "s | %" + inTransitWidth + "s | %" + ackWidth + "s | %" + ipWidth + "s | %" + portWidth + "s",
            "#", hostHd, queueHd, creationHd, lastMessageHd, connIdHd, inTransitHd, ackHd, ipHd, portHd) + " (" + LocalDateTime.now().format(Const.localTmsSec) + ")");

        // Separator
        System.out.println("-".repeat(totalWidth));

        // Rows
        for (ConsumerInfo ci : consumers) {
          String creationStr = ci.getCreationTime() > 0
              ? java.time.Instant.ofEpochMilli(ci.getCreationTime()).atZone(java.time.ZoneId.systemDefault()).toLocalDateTime().format(Const.localTmsSec)
              : "";
          long lastMsgTms = Math.max(ci.getLastDeliveredTime(), ci.getLastAcknowledgedTime());
          String lastMsgStr = lastMsgTms > 0
              ? java.time.Instant.ofEpochMilli(lastMsgTms).atZone(java.time.ZoneId.systemDefault()).toLocalDateTime().format(Const.localTmsSec)
              : "";
          String sb = String.format("%-4d |", i) +
              String.format(" %-" + hostWidth + "s |", ci.getHost()) +
              String.format(" %-" + queueWidth + "s |", ci.getQueueName()) +
              String.format(" %-" + creationWidth + "s |", creationStr) +
              String.format(" %-" + lastMsgWidth + "s |", lastMsgStr) +
              String.format(" %-" + connIdWidth + "s |", ci.getConnectionID()) +
              String.format(" %" + inTransitWidth + "d |", ci.getInTransitCount()) +
              String.format(" %" + ackWidth + "d |", ci.getAcknowledgedCount()) +
              String.format(" %" + ipWidth + "s |", ci.getIpAddress()) +
              String.format(" %" + portWidth + "d", ci.getPort());
          System.out.println(sb);
          i++;
        }

      }
    }
  }

  public static class Retrieve {

    public static List<ConsumerInfo> retrieveAndFilter(List<QueueInfo> qiList, Config config, ConsumerFilter consumerFilter) {
      List<ConsumerInfo> CIList = retrieve(qiList, config);
      List<ConsumerInfo> filteredByScope = filterByScope(CIList, consumerFilter.getCscp().arg());
      List<ConsumerInfo> filteredBySearchString = filterBySearchString(filteredByScope, consumerFilter.getCsch().arg());
      List<ConsumerInfo> filteredByAge = filterByAge(filteredBySearchString, consumerFilter.getCage().number(), consumerFilter.getCage().op());
      List<ConsumerInfo> filteredByInTransit = filterByInTransit(filteredByAge, consumerFilter.getCtrs().number(), consumerFilter.getCtrs().op());
      List<ConsumerInfo> filteredByAck = filterByAck(filteredByInTransit, consumerFilter.getCack().number(), consumerFilter.getCack().op());
      List<ConsumerInfo> sortedList = sortConsumerInfo(filteredByAck, consumerFilter.getCsrt().number());
      List<ConsumerInfo> limitedList = consumerFilter.getClim().number() > 0 ? sortedList.subList(0, Math.min(consumerFilter.getClim().number(), sortedList.size())) : sortedList;
      return limitedList;
    }

    public static List<ConsumerInfo> retrieve(List<QueueInfo> qiList, Config config) {
      List<ConsumerInfo> result = new ArrayList<>();
      Map<String, String> broker2Encoded = new HashMap<>();
      for (QueueInfo qi : qiList) {
        try {
          if (broker2Encoded.get(qi.getHost()) == null) {
            String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(Div.prot(config, qi), qi.getHost(), config.get("port"), Div.user(config, qi), Div.pass(config, qi));
            broker2Encoded.put(qi.getHost(), brokerNameEncoded);
            // For every new broker/host we run listConsumers - retrieve all consumers on this particular broker
            ArtemisConsoleAPI.ACAPIResult consResult = ArtemisConsoleAPI.listConsumers(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi));
            List<ConsumerInfo> cList = null;
            if (consResult.getStatus() != null && consResult.getStatus().equals("200")) {
              cList = parseConsumers(qi, consResult.getJsonArray());
              result.addAll(cList);
            } else {
              System.out.println("Failed to list consumers for queue: " + qi.getQueueName() + " on host: " + qi.getHost() + " - " + consResult.getRawResponse());
            }
            if (cList != null) { // We have received data about consumers, now try to enrich with IP-address of that consumer
              ArtemisConsoleAPI.ACAPIResult connResult = ArtemisConsoleAPI.listConnections(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi));
              Map<String, String> connId2IPMap = getConnId2IPMap(connResult.getJsonArray());
              for (ConsumerInfo ci : cList) {
                if (connId2IPMap.containsKey(ci.getConnectionID())) {
                  String clientAddress = connId2IPMap.get(ci.getConnectionID());
                  String ip = clientAddress.split(":")[0];
                  int port = Integer.parseInt(clientAddress.split(":")[1]);
                  ci.setIpAddress(ip);
                  ci.setPort(port);
                }
              }
            }
          }
        } catch (Exception e) {
          System.out.println("Error occurred: During listConsumers-API for queue: " + qi.getQueueName() + " on host " + qi.getHost() + ": " + e + " " + e.getMessage());
        }
      }

      List<ConsumerInfo> filteredResult = new ArrayList<>();
      for (QueueInfo qi : qiList) {
        // Copy consumers that match the queue to filteredResult
        result.stream().filter(c -> c.getHost().equals(qi.getTrimmedHost()) && c.getQueueName().equals(qi.getQueueName())).forEach(filteredResult::add);
      }
      return filteredResult;
    }

    private static List<ConsumerInfo> parseConsumers(QueueInfo qi, JSONArray jsonArray) {
      List<ConsumerInfo> list = new ArrayList<>();

      for (int i = 0; i < jsonArray.length(); i++) {
        JSONObject elementJson = jsonArray.getJSONObject(i);
        ConsumerInfo ci = new ConsumerInfo();
        ci.setHost(qi.getTrimmedHost());
        ci.setQueueName(elementJson.getString("queueName"));
        ci.setConsumerID("" + elementJson.getInt("consumerID"));
        ci.setCreationTime(elementJson.getLong("creationTime"));
        ci.setConnectionID(elementJson.getString("connectionID"));
        ci.setBrowseOnly(elementJson.getBoolean("browseOnly"));
        ci.setDeliveredCount(elementJson.getInt("messagesDelivered"));
        ci.setInTransitCount(elementJson.getInt("messagesInTransit"));
        ci.setAcknowledgedCount(elementJson.getInt("messagesAcknowledged"));
        ci.setLastDeliveredTime(elementJson.getLong("lastDeliveredTime"));
        ci.setLastAcknowledgedTime(elementJson.getLong("lastAcknowledgedTime"));
        list.add(ci);
      }
      return list;
    }

    private static Map<String, String> getConnId2IPMap(JSONArray jsonArray) {
      Map<String, String> result = new HashMap<>();
      for (int i = 0; i < jsonArray.length(); i++) {
        JSONObject elementJson = jsonArray.getJSONObject(i);
        String connectionID = elementJson.getString("connectionID");
        String clientAddress = elementJson.getString("clientAddress");
        result.put(connectionID, clientAddress);
      }
      return result;
    }

    private static List<ConsumerInfo> filterByScope(List<ConsumerInfo> CIList, String scope) {
      if (scope == null || (!scope.equalsIgnoreCase("loc") && !scope.equalsIgnoreCase("rem")))
        return CIList;
      List<ConsumerInfo> filteredCIList = new ArrayList<>();
      for (ConsumerInfo ci : CIList) {
        boolean localhost = ci.getIpAddress().equals("localhost") || ci.getIpAddress().equals("127.0.0.1");
        if (scope.equalsIgnoreCase("loc") && localhost)
          filteredCIList.add(ci);
        if (scope.equalsIgnoreCase("rem") && !localhost)
          filteredCIList.add(ci);
      }
      return filteredCIList;
    }

    private static List<ConsumerInfo> filterBySearchString(List<ConsumerInfo> CIList, String matchStr) {
      if (matchStr == null || matchStr.trim().isEmpty())
        return CIList;
      List<ConsumerInfo> filteredCIList = new ArrayList<>();
      boolean positive = true;
      if (matchStr.startsWith("!")) {
        positive = false;
        matchStr = matchStr.substring(1);
      }
      Pattern pattern = Pattern.compile(matchStr.toLowerCase());
      for (ConsumerInfo ci : CIList) {
        Matcher matcher = pattern.matcher(ci.toString().toLowerCase());
        if (matcher.find()) {
          if (positive)
            filteredCIList.add(ci);
        } else {
          if (!positive)
            filteredCIList.add(ci);
        }
      }
      return filteredCIList;
    }

    private static List<ConsumerInfo> filterByAge(List<ConsumerInfo> CIList, int age, String op) {
      if (age < 0)
        return CIList;
      List<ConsumerInfo> filteredCIList = new ArrayList<>();
      for (ConsumerInfo ci : CIList) {
        if (compare(ci.getAge(), op, age))
          filteredCIList.add(ci);
      }
      return filteredCIList;
    }

    private static List<ConsumerInfo> filterByInTransit(List<ConsumerInfo> CIList, int number, String op) {
      if (number < 0)
        return CIList; // no filter
      List<ConsumerInfo> filteredCIList = new ArrayList<>();
      for (ConsumerInfo ci : CIList) {
        if (compare(ci.getInTransitCount(), op, number))
          filteredCIList.add(ci);
      }
      return filteredCIList;
    }

    private static List<ConsumerInfo> filterByAck(List<ConsumerInfo> CIList, int number, String op) {
      if (number < 0)
        return CIList; // no filter
      List<ConsumerInfo> filteredCIList = new ArrayList<>();
      for (ConsumerInfo ci : CIList) {
        if (compare(ci.getAcknowledgedCount(), op, number))
          filteredCIList.add(ci);
      }
      return filteredCIList;
    }


    private static boolean compare(int arg1, String op, int arg2) {
      if (op.equals("=")) {
        return arg1 == arg2;
      } else if (op.equals(">")) {
        return arg1 > arg2;
      } else {
        return arg1 < arg2;
      }
    }

    private static List<ConsumerInfo> sortConsumerInfo(List<ConsumerInfo> CIL, int sort) {
      if (sort > 0) {
        switch (sort) {
          case 1:
            CIL.sort(Comparator.comparing(ConsumerInfo::getHost));
            break;
          case 2:
            CIL.sort(Comparator.comparing(ConsumerInfo::getQueueName));
            break;
          case 3:
            CIL.sort(Comparator.comparingLong(ConsumerInfo::getCreationTime));
            break;
          case 4:
            CIL.sort(Comparator.comparingLong(ci -> Math.max(ci.getLastDeliveredTime(), ci.getLastAcknowledgedTime())));
            break;
          case 5:
            CIL.sort(Comparator.comparing(ConsumerInfo::getConnectionID, Comparator.nullsFirst(String::compareTo)));
            break;
          case 6:
            CIL.sort(Comparator.comparingInt(ConsumerInfo::getInTransitCount));
            break;
          case 7:
            CIL.sort(Comparator.comparingInt(ConsumerInfo::getAcknowledgedCount));
            break;
          case 8:
            CIL.sort(Comparator.comparing(ConsumerInfo::getIpAddress));
            break;
          case 9:
            CIL.sort(Comparator.comparingInt(ConsumerInfo::getPort));
            break;
        }
      } else if (sort < 0) {
        switch (sort) {
          case -1:
            CIL.sort((o1, o2) -> o2.getHost().compareTo(o1.getHost()));
            break;
          case -2:
            CIL.sort((o1, o2) -> o2.getQueueName().compareTo(o1.getQueueName()));
            break;
          case -3:
            CIL.sort((o1, o2) -> Long.compare(o2.getCreationTime(), o1.getCreationTime()));
            break;
          case -4:
            CIL.sort((o1, o2) -> Long.compare(Math.max(o2.getLastDeliveredTime(), o2.getLastAcknowledgedTime()), Math.max(o1.getLastDeliveredTime(), o1.getLastAcknowledgedTime())));
            break;
          case -5:
            CIL.sort((o1, o2) -> Comparator.nullsLast(String::compareTo).reversed().compare(o1.getConnectionID(), o2.getConnectionID()));
            break;
          case -6:
            CIL.sort((o1, o2) -> Integer.compare(o2.getInTransitCount(), o1.getInTransitCount()));
            break;
          case -7:
            CIL.sort((o1, o2) -> Integer.compare(o2.getAcknowledgedCount(), o1.getAcknowledgedCount()));
            break;
          case -8:
            CIL.sort((o1, o2) -> o2.getIpAddress().compareTo(o1.getIpAddress()));
            break;
          case -9:
            CIL.sort((o1, o2) -> Integer.compare(o2.getPort(), o1.getPort()));
            break;
        }
      }
      return CIL;
    }


  }
}
