package no.statnett.ecp.brs.actions;

import no.statnett.ecp.brs.Config;
import no.statnett.ecp.brs.Div;
import no.statnett.ecp.brs.Login;
import no.statnett.ecp.brs.state.QueueFilter;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.SimpleParser;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


public class ListQueues {

  public static class Print {
    public static void printToConsole(List<QueueInfo> sortedQIList, boolean echo) {
      if (echo) {
        String hostHd = "Host (1)";
        String queueHd = "Queue (2)";
        String sizeHd = "Size (3)";
        String consHd = "Cons (4)";
        String enqHd = "Enqueue (5)";
        String deqHd = "Dequeue (6)";

        int hostWidth = Math.max(sortedQIList.stream().mapToInt(qi -> qi.getHost().length()).max().orElse(0), hostHd.length());
        int queueWidth = Math.max(sortedQIList.stream().mapToInt(qi -> qi.getQueueName().length()).max().orElse(0), queueHd.length());
        int sizeWidth = Math.max(sortedQIList.stream().mapToInt(qi -> ("" + qi.getSize()).length()).max().orElse(0), sizeHd.length());
        int consWidth = Math.max(sortedQIList.stream().mapToInt(qi -> ("" + qi.getConsumers()).length()).max().orElse(0), consHd.length());
        int enqWidth = Math.max(sortedQIList.stream().mapToInt(qi -> ("" + qi.getEnqueueCount()).length()).max().orElse(0), enqHd.length());
        int deqWidth = Math.max(sortedQIList.stream().mapToInt(qi -> ("" + qi.getDequeueCount()).length()).max().orElse(0), deqHd.length());
        int totalWidth = hostWidth + queueWidth + sizeWidth + consWidth + enqWidth + deqWidth + 4 + 6 * 3;

        System.out.println(String.format("%-4s | %-" + hostWidth + "s | %-" + queueWidth + "s | %" + sizeWidth + "s | %" + consWidth + "s | %" + enqWidth + "s | %" + deqWidth + "s", "#", hostHd, queueHd, sizeHd, consHd, enqHd, deqHd) + " (" + LocalDateTime.now().format(Const.localTmsSec) + ")");
        System.out.println("-".repeat(totalWidth));

        int i = 1;
        for (QueueInfo qi : sortedQIList) {
          String sb = String.format("%-4d |", i) +
              String.format(" %-" + hostWidth + "s |", qi.getHost()) +
              String.format(" %-" + queueWidth + "s |", qi.getQueueName()) +
              String.format(" %" + sizeWidth + "d |", qi.getSize()) +
              String.format(" %" + consWidth + "d |", qi.getConsumers()) +
              String.format(" %" + enqWidth + "d |", qi.getEnqueueCount()) +
              String.format(" %" + deqWidth + "d", qi.getDequeueCount());
          System.out.println(sb);
          i++;
        }
      }

    }

    public static void printToJson(List<QueueInfo> sortedQIList, File jsonFile) throws IOException {
      for (QueueInfo qi : sortedQIList) {
        StringBuilder sb = new StringBuilder();
        sb.append("{ \"statustime\": \"").append(LocalDateTime.now().format(Const.localTmsSec)).append("\", ");
        sb.append("\"host\": ").append(format(40, qi.getHost()));
        sb.append("\"queue\": ").append(format(50, qi.getQueueName()));
        sb.append("\"size\": ").append(format(5, qi.getSize()));
        sb.append("\"consumers\": ").append(format(2, qi.getConsumers()));
        sb.append("\"enqueue\": ").append(format(6, qi.getEnqueueCount()));
        sb.append("\"dequeue\": ").append(formatNoComma(6, qi.getDequeueCount()));
        sb.append("}\n");
        Files.writeString(jsonFile.toPath(), sb.toString(), StandardOpenOption.APPEND);
      }
    }

    private static String format(int size, int number) {
      return String.format("%" + size + "d, ", number);
    }

    private static String formatNoComma(int size, int number) {
      return String.format("%" + size + "d ", number);
    }

    private static String format(int size, String str) {
      return String.format("%-" + size + "s, ", "\"" + str + "\"");
    }

  }

  public static class Retrieve {
    public static List<QueueInfo> retrieveAndFilter(Config config, QueueFilter queueFilter) throws NoSuchAlgorithmException, IOException, KeyManagementException {
      List<QueueInfo> QIList = retrieve(Div.prot(config, null), config.get("host"), config.get("port"), config.get("user"), config.get("password")); // Find the main host first
      // Retrieve extra logins in parallel for simplicity and speed
      List<QueueInfo> extraResults = config.getExtraLogins().values()
          .parallelStream()
          .flatMap(login -> {
            String user = login.getUser() == null ? config.get("user") : login.getUser();
            String password = login.getPassword() == null ? config.get("password") : login.getPassword();
            String protocol = login.getProtocol() == null ? Div.prot(config, null) : login.getProtocol();
            try {
              return retrieve(protocol, login.getHost(), config.get("port"), user, password).stream();
            } catch (Exception expcetion) {
              System.out.println("Error in communcation with " + login.getHost() + " - " + expcetion.getMessage());
              return java.util.stream.Stream.<QueueInfo>empty();
            }
          })
          .collect(Collectors.toList());
      QIList.addAll(extraResults);
      List<QueueInfo> trimmedHostnameList = trimHostNames(QIList);
      List<QueueInfo> filteredByQueueName = filterByQueueNames(trimmedHostnameList, queueFilter.getQque().arg());
      List<QueueInfo> filteredBySearchString = filterBySearchString(filteredByQueueName, queueFilter.getQsch().arg());
      List<QueueInfo> filteredByConsumerCount = filterByConsumerCount(filteredBySearchString, queueFilter.getQcon().number(), queueFilter.getQcon().op());
      List<QueueInfo> filteredBySize = filterByQueueSize(filteredByConsumerCount, queueFilter.getQsiz().number(), queueFilter.getQsiz().op());
      List<QueueInfo> sortedQIList = sortQueueInfo(filteredBySize, queueFilter.getQsrt().number());
      List<QueueInfo> limitedQIList = queueFilter.getQlim().number() > 0 ? sortedQIList.subList(0, Math.min(queueFilter.getQlim().number(), sortedQIList.size())) : sortedQIList;
      return limitedQIList;
    }


    /* SUB-METHODS SUPPORTING THE ABOVE MAIN METHOD */
    public static List<QueueInfo> retrieve(String prot, String host, String port, String username, String password) throws NoSuchAlgorithmException, IOException, KeyManagementException {
      String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(prot, host, port, username, password);
      String json = ArtemisConsoleAPI.retrieveQueueInfoJSON(prot, host, port, brokerNameEncoded, username, password);
      int dataArrPosStart = json.indexOf("data\\\":[") + 7;
      int dataArrPosEnd = json.indexOf("]", dataArrPosStart);

      String dataArrJson = json.substring(dataArrPosStart, dataArrPosEnd);
      dataArrJson = dataArrJson.replaceAll("\\\\", "");

      List<QueueInfo> queueInfoList = new ArrayList<>();
      for (String elementJson : dataArrJson.split("},\\{")) {
        // Each element looks like this (except first is prefixed with "[{" and last is postfixed with "}".
        // "id":"7","name":"ExpiryQueue","address":"ExpiryQueue","filter":"","durable":"true","paused":"false","temporary":"false","purgeOnNoConsumers":"false","consumerCount":"0","maxConsumers":"-1","autoCreated":"false","user":"","routingType":"ANYCAST","messagesAdded":"0","messageCount":"0","messagesAcked":"0","messagesExpired":"0","deliveringCount":"0","messagesKilled":"0","directDeliver":"false","exclusive":"false","lastValue":"false","lastValueKey":"","scheduledCount":"0","groupRebalance":"false","groupRebalancePauseDispatch":"false","groupBuckets":"-1","groupFirstKey":"","enabled":"true","ringSize":"-1","consumersBeforeDispatch":"0","delayBeforeDispatch":"-1","autoDelete":"false"
        // All attributes we care about are on this form:
        // "key":"value",
        // The values we're interested in, and how they relate to QueueInfo: name, consumerCount, messageCount, messagesAcked, messagesAdded
        QueueInfo queueInfo = new QueueInfo();
        queueInfo.setHost(host);
        queueInfo.setTrimmedHost(host);
        queueInfo.setConsumers(Integer.parseInt(SimpleParser.findStringInElement(elementJson, "consumerCount")));
        queueInfo.setDequeueCount(Integer.parseInt(SimpleParser.findStringInElement(elementJson, "messagesAcked")));
        queueInfo.setEnqueueCount(Integer.parseInt(SimpleParser.findStringInElement(elementJson, "messagesAdded")));
        queueInfo.setQueueName(SimpleParser.findStringInElement(elementJson, "name"));
        queueInfo.setSize(Integer.parseInt(SimpleParser.findStringInElement(elementJson, "messageCount")));
        queueInfoList.add(queueInfo);
      }
      return queueInfoList;
    }

    private static List<QueueInfo> trimHostNames(List<QueueInfo> list) {
      // Assume that most hostname have the same suffixes
      // Start from the end of the hostname and examine last character of every hostname
      // If all hostnames have the same last character, remove that character from all hostnames
      // and save that character as suffix
      // Repeat process until a character differs on any hostname or there are no dots in the hostname
      Set<String> noDots;
      Set<String> lastChars;
      do {
        // Check last character of all hostnames
        noDots = list.stream().map(QueueInfo::getTrimmedHost).filter(s -> !s.contains(".")).collect(Collectors.toSet()); // All hostnames without dots
        lastChars = list.stream().map(QueueInfo::getTrimmedHost).map(s -> s.substring(s.length() - 1)).collect(Collectors.toSet());
        if (lastChars.size() == 1 && noDots.isEmpty()) {
          // If lastChar is not "" and
          for (QueueInfo qi : list) {
            String trimmedHost = qi.getTrimmedHost();
            if (!trimmedHost.isEmpty()) {
              qi.setTrimmedHost(trimmedHost.substring(0, trimmedHost.length() - 1));
            }
          }
        }
        // If lastChar is not "" and
      } while (lastChars.size() == 1 && noDots.isEmpty());
      return list;
    }

    private static List<QueueInfo> filterByQueueNames(List<QueueInfo> QIList, String queueNames) {
      if (queueNames == null || queueNames.trim().isEmpty())
        return QIList;
      List<QueueInfo> filteredQIList = new ArrayList<>();
      for (QueueInfo qi : QIList) {
        for (String matchQueueName : queueNames.split(",")) {
          if (qi.getQueueName().equalsIgnoreCase(matchQueueName))
            filteredQIList.add(qi);
        }
      }
      return filteredQIList;
    }

    private static List<QueueInfo> filterBySearchString(List<QueueInfo> QIList, String matchStr) {
      if (matchStr == null || matchStr.trim().isEmpty())
        return QIList;
      List<QueueInfo> filteredQIList = new ArrayList<>();
      boolean positive = true;
      if (matchStr.startsWith("!")) {
        positive = false;
        matchStr = matchStr.substring(1);
      }
      Pattern pattern = Pattern.compile(matchStr.toLowerCase());
      for (QueueInfo qi : QIList) {
        Matcher matcher = pattern.matcher(qi.toString().toLowerCase());
        if (matcher.find()) {
          if (positive)
            filteredQIList.add(qi);
        } else {
          if (!positive)
            filteredQIList.add(qi);
        }
      }
      return filteredQIList;
    }

    private static List<QueueInfo> filterByConsumerCount(List<QueueInfo> QIList, int consumerCount, String op) {
      if (consumerCount < 0)
        return QIList;
      List<QueueInfo> filteredQIList = new ArrayList<>();
      for (QueueInfo qi : QIList) {
        if (compare(qi.getConsumers(), op, consumerCount))
          filteredQIList.add(qi);
      }
      return filteredQIList;
    }

    private static List<QueueInfo> filterByQueueSize(List<QueueInfo> QIList, int queueSize, String op) {
      if (queueSize < 0)
        return QIList; // no filter
      List<QueueInfo> filteredQIList = new ArrayList<>();
      for (QueueInfo qi : QIList) {
        if (compare(qi.getSize(), op, queueSize))
          filteredQIList.add(qi);
      }
      return filteredQIList;
    }

    private static boolean compare(int arg1, String op, int arg2) {
      if (op.equals("=")) {
        return arg1 == arg2;
      } else if (op.equals(">")) {
        return arg1 > arg2;
      } else {
        return arg1 < arg2;
      }
    }


    private static List<QueueInfo> sortQueueInfo(List<QueueInfo> QIL, int sort) {
      if (sort > 0) {
        switch (sort) {
          case 1:
            QIL.sort(Comparator.comparing(QueueInfo::getHost));
            break;
          case 2:
            QIL.sort(Comparator.comparing(QueueInfo::getQueueName));
            break;
          case 3:
            QIL.sort(Comparator.comparingInt(QueueInfo::getSize));
            break;
          case 4:
            QIL.sort(Comparator.comparingInt(QueueInfo::getConsumers));
            break;
          case 5:
            QIL.sort(Comparator.comparingInt(QueueInfo::getEnqueueCount));
            break;
          case 6:
            QIL.sort(Comparator.comparingInt(QueueInfo::getDequeueCount));
            break;
        }
      } else if (sort < 0) {
        switch (sort) {
          case -1:
            QIL.sort((QueueInfo o1, QueueInfo o2) -> o2.getHost().compareTo(o1.getHost()));
            break;
          case -2:
            QIL.sort((QueueInfo o1, QueueInfo o2) -> o2.getQueueName().compareTo(o1.getQueueName()));
            break;
          case -3:
            QIL.sort((QueueInfo o1, QueueInfo o2) -> Integer.compare(o2.getSize(), o1.getSize()));
            break;
          case -4:
            QIL.sort((QueueInfo o1, QueueInfo o2) -> Integer.compare(o2.getConsumers(), o1.getConsumers()));
            break;
          case -5:
            QIL.sort((QueueInfo o1, QueueInfo o2) -> Integer.compare(o2.getEnqueueCount(), o1.getEnqueueCount()));
            break;
          case -6:
            QIL.sort((QueueInfo o1, QueueInfo o2) -> Integer.compare(o2.getDequeueCount(), o1.getDequeueCount()));
            break;
        }
      }
      return QIL;
    }

  }
}
