package no.statnett.ecp.cc;

import no.statnett.ecp.utils.*;

import java.io.File;
import java.io.IOException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

public class ConnectivityCheck {

  public static final String VERSION = "v1.3.1";
  public static final String NONE = "NONE";
  private static File logFile;
  private static String rotationType = null;
  private static int noOfLogFiles = 0;
  private static int noOfDaysDelayRotation = 0;
  private static boolean compress = false;
  private static String user;
  private static String pw;
  private static URL url;
  private static Map<String, String> brokerMap;
  private static List<String> codes;
  // default is a maximum run of 5 minutes, since this job is started every 5 min in most cron-jobs
  // this is to avoid that multiple CC-jobs run at the same time because one or endpoints do not
  // respond.
  private static int maxSecRun = 300;
  private static final long startTms = System.currentTimeMillis();

  public ConnectivityCheck(String user, String pw, URL url) {
    ConnectivityCheck.user = user;
    ConnectivityCheck.pw = pw;
    ConnectivityCheck.url = url;
  }

  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String brokerMapStr = Options.parseString(initArgsList, "-b");
    if (brokerMapStr != null) {
      brokerMap = new HashMap<>();
      String[] brokerMapArr = brokerMapStr.split(",");
      for (String entry : brokerMapArr) {
        String[] arr = entry.split(":");
        String countryCode = arr[0];
        String brokerIP = arr.length > 1 ? arr[1] : null;
        // countryCode must match this regex: \d\dV
        // brokerIP must match this regex: \d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}
        if (countryCode.matches("\\d\\dV")) {
          if (brokerIP != null && brokerIP.matches("\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}")) {
            brokerMap.put(countryCode, brokerIP);
          } else {
            System.out.println("The -b option presented an invalid broker-IP: " + brokerIP + "(whole option: " + entry + ")");
            System.exit(1);
          }
        } else {
          System.out.println("The -b option presented an invalid code-prefix: " + countryCode + "(whole option: " + entry + ")");
          System.exit(1);
        }
      }
    }
    String logFilename = Options.parseString(initArgsList, "-o");
    if (logFilename != null) {
      logFile = Options.checkFile(logFilename, true);
    }
    rotationType = Options.parseString(initArgsList, "-r");
    if (rotationType != null && logFile == null) {
      System.err.println("Rotate option (-r) requires output file option (-o) - exiting");
      System.exit(1);
    }
    if (rotationType != null && !rotationType.matches("^(day|week|month)$")) {
      System.err.println("Rotate option (-r) must be either 'day', 'week' or 'month' - exiting");
      System.exit(1);
    }
    noOfLogFiles = Options.parseInt(initArgsList, "-m", 1, 1000, 10);
    noOfDaysDelayRotation = Options.parseInt(initArgsList, "-n", 0, 1000, 0);
    compress = Options.parseBoolean(initArgsList, "-z");
    maxSecRun = Options.parseInt(initArgsList, "-t", 0, 3600, 300);
    return initArgsList;
  }


  private static void parseMandatoryArg(List<String> mandatoryArgs) throws MalformedURLException {
    if (mandatoryArgs.isEmpty()) {
      usage();
      System.exit(1);
    }
    codes = Arrays.asList(mandatoryArgs.get(0).split(","));
    if (brokerMap != null) { // optional, but if they exist - must match the mandatory codes
      for (String code : codes) {
        if (code.equals(NONE))
          continue;
        if (code.length() < 4) {
          System.out.println("The code " + code + " is too short");
          System.exit(1);
        }
        if (brokerMap.get(code.substring(0, 3)) == null) {
          System.out.println("The code " + code + " is missing a broker-IP in the broker-map");
          System.exit(1);
        }
      }
    }
    if (mandatoryArgs.size() == 2) {
      URL initialURL = new URL(mandatoryArgs.get(1));
      url = new URL(initialURL.getProtocol(), initialURL.getHost(), initialURL.getPort(), "/ECP_MODULE/settings/connectivityCheck");
      if (initialURL.getUserInfo() != null) {
        user = initialURL.getUserInfo().split(":")[0];
        pw = initialURL.getUserInfo().split(":")[1];
      } else {
        System.out.println("The URL must contain user:password");
        System.exit(1);
      }
    } else if (!mandatoryArgs.get(0).equals(NONE)) {
      usage();
      System.exit(1);
    }
  }

  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    List<String> mandatoryArgs = parseOptions(args);
    parseMandatoryArg(mandatoryArgs);

    // Populate the brokerConnectionMap by running netstat for each broker-IP and check number of connections to port 5671
    Map<String, Integer> brokerConnectionsMap = brokerMap == null ? null : brokerMap.entrySet().stream().map(e -> Map.entry(e.getKey(), netstat(e.getValue()))).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

    // Specify the ECP-endpoints to check connectivity from
    ConnectivityCheck cc = new ConnectivityCheck(user, pw, url);

    String hostName = InetAddress.getLocalHost().getHostName();
    if (hostName.contains(".")) {
      hostName = hostName.substring(0, hostName.indexOf("."));
    }
    ConnectionStatus prevStatus = ConnectionStatus.INIT;
    // Run connectivity check for each code
    for (String code : codes) {
      // Exit without logging anything. 5 different cases where this can happen
      if (prevStatus == ConnectionStatus.CONNECT_ERROR ||
          prevStatus == ConnectionStatus.JSON_ERROR ||
          code.equals(NONE) ||
          noneDetected() ||
          (startTms + maxSecRun * 1000L < System.currentTimeMillis())) {
        System.exit(0);
      }
      int braceStartPos = code.indexOf("(");
      String messageType = null;
      if (braceStartPos > -1) {
        int braceEndPos = code.indexOf(")", braceStartPos);
        if (braceEndPos > -1) {
          messageType = code.substring(braceStartPos + 1, braceEndPos);
        }
        code = code.substring(0, braceStartPos);
      }
      ConnectionStatus status = cc.check(code, messageType);

      // The report printed from ConnectivityCheck can be like this:
      // {"statustime": "2024-08-09 21:40:08", "timestamp": 1723232408, "connectivity_check_status": "OK", "hostname": "h1-a-ecp-apt07", "endpoint_code_checked": "48V0000000000132", "broker_connections": 0, "area_code": "48V", "broker_ip": "62.190.148.249"}
      // Or like this (if broker-connection-check is omitted):
      // {"statustime": "2024-08-09 21:40:08", "timestamp": 1723232408, "connectivity_check_status": "OK", "hostname": "h1-a-ecp-apt07", "endpoint_code_checked": "48V0000000000132"}
      LocalDateTime now = LocalDateTime.now();
      String nowAsTimestamp = now.format(Const.localTmsSecNotISO);
      long currentTimeSec = System.currentTimeMillis() / 1000;
      String output = "{\"statustime\": \"" + nowAsTimestamp + "\", \"timestamp\": " + currentTimeSec + ", \"connectivity_check_status\": \"" + status + "\", \"hostname\": \"" + hostName + "\", \"endpoint_code_checked\": \"" + code + "\"";
      if (brokerMap == null) {
        output += "}";
      } else {
        String codePrefix = code.substring(0, 3);
        String brokerIp = brokerMap.get(codePrefix);
        int brokerConnections = brokerConnectionsMap.get(codePrefix);
        output += ", \"broker_connections\": " + brokerConnections + ", \"area_code\": \"" + codePrefix + "\", \"broker_ip\": \"" + brokerIp + "\"}";
      }
      if (logFile == null) {
        System.out.println(output);
      } else {
        Files.writeString(logFile.toPath(), output + "\n", StandardOpenOption.APPEND);
        if (rotationType != null && Long.parseLong(now.format(Const.tmsNumberMillisec)) > 0) {
          LogRotator logRotator = new LogRotator(logFile, now.atZone(ZoneId.systemDefault()), noOfDaysDelayRotation);
          logRotator.rotateLogs(rotationType);
          logRotator.deleteExcessLogs(noOfLogFiles);
          if (compress)
            logRotator.compress();
        }
      }
      prevStatus = status;
    }
  }

  private static boolean noneDetected() {
    // Check if a NONE-file is present
    File f = new File("cc.none");
    return f.exists();
  }

  public ConnectionStatus check(String code, String messageType) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    if (messageType == null)
      messageType = "TEST";
    String body = "{\"receiver\":\"" + code + "\",\"messageType\":\"" + messageType + "\"}";
    try {
      String json = EcpHTTP.getECPHTTResponse("PUT", url, user, pw, body);
      if (json != null && json.contains("\"OK\"")) {
        return ConnectionStatus.OK;
      } else if (json.contains("ECP4 | Error")) {
        return ConnectionStatus.JSON_ERROR; // Maybe login to ECP-API failed?
      }
      return ConnectionStatus.NOT_OK;
    } catch (ConnectException ce) {
      return ConnectionStatus.CONNECT_ERROR;
    }
  }

  public static int netstat(String ip) {
    // Run "netstat -anp" command in a Process and print the output
    int connections = 0;
    try {
      Process p = Runtime.getRuntime().exec("netstat -a -n");
      java.io.BufferedReader stdInput = new java.io.BufferedReader(new java.io.InputStreamReader(p.getInputStream()));
      String line;
      while ((line = stdInput.readLine()) != null) {
        if (line.contains(":5671") && line.contains(ip) && line.contains("ESTABLISHED")) {
          connections++;
        }
      }
    } catch (IOException ioe) {
      System.err.println("Warning: Could not run netstat command ('netstat -a -n') " + ioe.getMessage() + " option -b cannot be used");
    }
    return connections;
  }


  private static void usage() {
    System.out.println("ConnectivityCheck (CC) " + VERSION + " offers a simple way to check connectivity to ECP-endpoints. The tool must");
    System.out.println("be executed on the same host as the ECP-endpoint if run with -b option, otherwise the result is not useful.");
    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar CC [-b <BROKER-MAP>] <CODE-LIST> <ECP-URL>  \n");
    System.out.println("\n\nThe arguments:");
    System.out.println(" OPTIONS      ");
    System.out.println("     -b <BROKER-MAP>   : Broker-IP mapped to 3 first characters of ECP code, separated by commas (no spaces)");
    System.out.println("                         on the form ddV:IP[,ddV:IP]* where d is a digit and IP is an IP-address. The map must");
    System.out.println("                         cover all CODE-prefixes used in CODE-LIST. The benefit of this, is that the result");
    System.out.println("                         will say if it is *your* broker-connections which is at fault, or it is the remote");
    System.out.println("                         endpoint");
    System.out.println("     -o <OUTPUT-FILE>  : Write output to file instead of stdout");
    System.out.println("     -t<sec>           : Terminate run after t seconds, to avoid running for too long. Default is 300 sec.");
    System.out.println("     -r day|week|month : Rotate log-file every day, week or month. Requires -o option");
    System.out.println("     -m<noLogFiles>    : Allow maximum number of log-files. Default is 10. Requires -r option");
    System.out.println("     -n<noDays>        : Do not rotate data younger than noDays. Default is 0. If set higher, the output log will");
    System.out.println("                         always contain the last noDays of data, even if the log-file is rotated. This is useful");
    System.out.println("                         for analysis of the last days of data (GapMonitor). Requires -r option");
    System.out.println("     -z                : Compress rotated logs. Requires -r option");
    System.out.println("MANDATORY ARGUMENTS");
    System.out.println("     <CODE-LIST>       : List of ECP codes to check connectivity to, separated by comma (no spaces)");
    System.out.println("     <ECP-URL>         : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("\nSpecial cases to avoid running the check:");
    System.out.println("\t1. Abort it by setting a CODE in the CODE-LIST to 'NONE'. The CC will abort when processing that code");
    System.out.println("\t2. Abort by having a file called 'cc.none' in the working directory. It's content is not important");
    System.out.println("\t3. Abort after a number of seconds by setting -t<sec> option. Default is 300. Abort immediately by setting to 0.");
    System.out.println();
    System.out.println("\nThe output of the CC can provide 4 types of 'connectivity_check_status' - which is useful to monitor:");
    System.out.println("\t1. OK             Able to connect to the remote endpoint");
    System.out.println("\t2. NOT_OK         Not able to connect to the remote endpoint");
    System.out.println("\t3. CONNECT_ERROR  Not able to connect to the local endpoint");
    System.out.println("\t4  JSON_ERROR     Able to connect to local endpoint, but login or something else failed");
    System.out.println("One suggestion is to use CC both to monitor your local endpoint (through the 3. status, and the remote through 1. and 2.");
    System.out.println("status. The 4. status would usually only occur during first time testing or configuration change on the endpoint.");
    System.out.println();
    System.out.println("\nExample 1:  Run connectivity check towards two endpoint from your localhost-endoint");
    System.out.println("\tjava -jar ekit.jar CC 50V000000000115O,50V-SN-TEST2-AT7 https://admin:password@remote-host:8443");
    System.out.println("\nExample 2:  Run connectivity check towards two endpoint from your localhost-endoint, with broker-IP mapping");
    System.out.println("\tjava -jar ekit.jar CC -b 50V:195.204.145.157 50V000000000115O,50V-SN-TEST2-AT7 https://admin:password@localhost:8443");
    System.out.println("\nExample 3:  Run connectivity check, append output to a file which is rotated monthly, keep 12 months of data");
    System.out.println("\tjava -jar ekit.jar CC -o cc.out -r month -m12 -z 50V000000000115O https://admin:password@remote-host:8443");
    System.out.println("\nExample 4:  Run connectivity check, but abort it after maximum 60 sec:");
    System.out.println("\tjava -jar ekit.jar CC -o cc.out -t60 50V000000000115O https://admin:password@remote-host:8443");
  }

}
