package no.statnett.ecp.edx;

import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.EcpEdxLog;
import no.statnett.ecp.utils.Options;

import java.io.*;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Stream;

import static java.time.ZoneOffset.UTC;


public class EDXMonitor {


  public static final String VERSION = "v1.8.0";

  protected static final int TOLERANCE_BAND_MINUTE = 2;

  private static final Flow flowSnd = new Flow(true);
  private static final Flow flowRcv = new Flow(false);

  private static long oldestEcpTms = Long.MAX_VALUE;
  private static long oldestEdxTms = Long.MAX_VALUE;
  private static ZonedDateTime oldestEcpZDT = null;
  private static ZonedDateTime oldestEdxZDT = null;
  private static File debugFile;
  private static final Set<String> excludeBusinessTypes = new HashSet<>();

  static {
    excludeBusinessTypes.add("EDX-INTERNAL"); // we will always exclude this one
  }


  private static void usage() {
    System.out.println("EDXMonitor (EDX) " + VERSION + " can monitor the flow of messages between ECP and EDX.");
    System.out.println("Usage  : java -jar ekit.jar EDX [OPTION] <ecp-log> <edx-log>\n");
    System.out.println("\nThe tool is compatible with ECP 4.14+ - it depends upon certain log events. ");
    System.out.println("The tool will monitor the traffic between ECP and EDX, and will print alert if");
    System.out.println("there is a discrepancy between the traffic going through ECP and EDX in the last measurement");
    System.out.println("minute. The minute which is measured is the whole/full minute that you find when calculating");
    System.out.println("now() - 3 minutes. If ecp- or edx-log is shorter than 5 minutes, the tool will not calculate");
    System.out.println("diff (instead show N/A). The reason behind this is to avoid false positives when logs are rotated");
    System.out.println("or if some traffic passes through ECP and EDX in different minutes");
    System.out.println("examined. The interval examined will always be");
    System.out.println("The expected way to run the tool is by using a crontab to run every minute");
    System.out.println("\nUsage  : java -jar ekit.jar EDX [OPTION] <ecp-log> <edx-log>\n");
    System.out.println(" OPTION:");
    System.out.println("     -d <debug-file>   : Debug mode, will print log lines if there is a diff < 0, so to quickly compare");
    System.out.println("     -x <list>         : Exclude businesstypes in this list from counting. The list is comma-separated");
    System.out.println("\nExample - run this every minute, will monitor if there is a discrepancy in the last minute:");
    System.out.println("java -jar ekit.jar EDX /var/log/ecp-endpoint/ecp.log /var/log/edx-toolbox/edx.log");
    System.out.println("\nExample - same as above, but excludes a businessType that never goes through both ECP and EDX");
    System.out.println("java -jar ekit.jar EDX -x ENTSOE-OPDM-INTERNAL-Echo-NOT /var/log/ecp-endpoint/ecp.log /var/log/edx-toolbox/edx.log");
    System.out.println("\nExample - same as first example, but prints the loglines to file if diff < 0 is detected");
    System.out.println("java -jar ekit.jar EDX -d /tmp/debug.log /var/log/ecp-endpoint/ecp.log /var/log/edx-toolbox/edx.log");
  }

  private static List<String> parseOptions(String[] initialArgs) {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String debugFileStr = Options.parseString(initArgsList, "-d");
    if (debugFileStr != null) {
      debugFile = Options.checkFile(debugFileStr, true);
    }
    String excludeBusinesstypesStr = Options.parseString(initArgsList, "-x");
    if (excludeBusinesstypesStr != null) {
      excludeBusinessTypes.addAll(Arrays.asList(excludeBusinesstypesStr.split(",")));
    }
    return initArgsList;
  }

  public static void main(String[] args) throws UnknownHostException {
    // Initialize some key variables
    ZonedDateTime nowZDT = Instant.now().atZone(UTC);
    oldestEcpZDT = nowZDT;
    oldestEdxZDT = nowZDT;

    // Read the command line arguments
    List<String> mandatoryArgs = parseOptions(args);
    if (mandatoryArgs.size() != 2) {
      usage();
      System.err.println("\n\nExpects two files to be specified (found " + mandatoryArgs.size() + " files): the ECP-log file and the EDX-log file");
      System.exit(1);
    }
    File ecpLog = Options.checkFile(mandatoryArgs.get(0), false);
    File edxLog = Options.checkFile(mandatoryArgs.get(1), false);

    // Process the log files
    processPlainText(ecpLog, true);
    processPlainText(edxLog, false);

    String hostName = InetAddress.getLocalHost().getHostName();
    if (hostName.contains(".")) {
      hostName = hostName.substring(0, hostName.indexOf("."));
    }

    // if we calculate now()-6m, we should have 5 full/whole minutes of logs (2+1+2)
    int completeIntervalMinutes = TOLERANCE_BAND_MINUTE + 1 + TOLERANCE_BAND_MINUTE;
    ZonedDateTime logsMustIncludeZDT = nowZDT.minusMinutes(1 + completeIntervalMinutes);
    if (oldestEcpZDT.isAfter(logsMustIncludeZDT) || oldestEdxZDT.isAfter(logsMustIncludeZDT)) {
      // This happens when logs rotate, we will not have enough data, but we will rather avoid a
      // false positive than missing a real one.
      System.out.println(makeEmptyLogLine(nowZDT, "INFO ", completeIntervalMinutes, hostName));
      System.exit(0);
    }

    // We have logs that cover the entire measurment periode (usually 3 minutes) - and can perform the measurement
    flowSnd.findBestDiff(nowZDT, TOLERANCE_BAND_MINUTE);
    flowRcv.findBestDiff(nowZDT, TOLERANCE_BAND_MINUTE);

    // Retrieve the data from the measurements
    int diffSnd = flowSnd.diff();
    Measurement fmSnd = flowSnd.getMain();
    boolean noNegativeACK = fmSnd.getNegativeACK().isEmpty();
    int diffRcv = flowRcv.diff();
    Measurement fmRcv = flowRcv.getMain();

    // Print the logline
    String logLevel = (diffSnd >= 0 && diffRcv >= 0 && noNegativeACK) ? "INFO " : (diffSnd < -1 || diffRcv < -1) ? "ERROR" : "WARN ";
    ZonedDateTime measurementZDT = nowZDT.minusMinutes(TOLERANCE_BAND_MINUTE + 1).truncatedTo(ChronoUnit.MINUTES);
    System.out.println(makeLogline(measurementZDT, logLevel, completeIntervalMinutes, fmSnd, fmRcv, hostName));

    if (debugFile != null) {
      if (diffRcv < 0) {
        flowRcv.debug(debugFile);
      }
      if (diffSnd < 0) {
        flowSnd.debug(debugFile);
      }
    }
  }

  private static void processPlainText(File log, boolean ecp) {
    try (Stream<String> stream = Files.lines(log.toPath(), StandardCharsets.UTF_8)) {
      stream.forEach(s -> processLogLine(s, ecp));
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  private static boolean allow(String businessType) {
    boolean allow = true;
    for (String excludeBusinessType : excludeBusinessTypes) {
      if (businessType.contains(excludeBusinessType)) {
        allow = false;
        break;
      }
    }
    return allow;
  }

  private static void processLogLine(String s, boolean ecp) {
    try {
      if (s != null && s.length() > 20 && (s.startsWith("202"))) {
        long yyyyMMddHHmmSS = Long.parseLong(s.substring(0, 19).replaceAll("[^\\d]", ""));
        long yyyyMMddHHmm = Long.parseLong(s.substring(0, 16).replaceAll("[^\\d]", ""));
        if (ecp) {
          if (s.contains("ID=MSG-08") && s.contains("STANDARD_MESSAGE")) { // Receive from BROKER into ECP, on its way to EDX
            if (allow(EcpEdxLog.findBusinessTypeInLogLine(s))) {
              flowRcv.addEcpId(yyyyMMddHHmm, EcpEdxLog.findMessageIDInLogLine(s), s);
            }
          } else if (s.contains("ID=MSG-11") && s.contains("STANDARD_MESSAGE")) { // Send from ECP to BROKER (upload to broker)
            if (allow(EcpEdxLog.findBusinessTypeInLogLine(s))) {
              flowSnd.addEcpId(yyyyMMddHHmm, EcpEdxLog.findMessageIDInLogLine(s), s);
            }
          } else if (s.contains("Message expiration error")) {
            flowRcv.addEcpExpired(yyyyMMddHHmm, s); // Store all expiration on rcv-flow (most likely correct flow, we don't know direction for sure)
          }
        } else { // edx
          if (s.contains("routed to endpoint") && !s.contains("toolbox-gateway") && !s.contains("EDX-INTERNAL")) { // EDX -> BA (RECEIVE)
            if (allow(EcpEdxLog.findBusinessTypeInLogLine(s))) {
              flowRcv.addEdxId(yyyyMMddHHmm, EcpEdxLog.findEdxMessageIDInLogLine(s), s);
            }
          } else if (s.contains("ID=MSG-01")) { // EDX -> ECP (4.14+)
            if (allow(EcpEdxLog.findBusinessTypeInLogLine(s))) {
              // if you send to yourself, we exclude this logline, as the message will not go to ECP anyway

              flowSnd.addEdxId(yyyyMMddHHmm, EcpEdxLog.findEdxMessageIDInLogLine(s), s);
            }
          } else if (s.contains("positive ACK")) { // This is where we can find the corresponding ECP-MessageID
            String ecpMessageId = EcpEdxLog.findEcpMessageIDInLogLine(s);
            String edxMessageId = EcpEdxLog.findEdxMessageIdInLogLine(s); // notice use of "Id" not "ID" for this particular log-line
            if (ecpMessageId == null || ecpMessageId.length() < 10) {
              // This will happen if you send a message from EDX, but receiver is specified to yourself. Therefore - this kind of
              // traffic must be excluded from the EDX-ECP-monitor:
              flowSnd.removeEdxId(edxMessageId);
            }
            else { // it's not "<null>" or empty...we assume a real ecpMessageId
              flowSnd.correlate(yyyyMMddHHmm, edxMessageId, ecpMessageId);
            }
          } else if (s.contains("negative ACK")) { // This indicates that EDX was not able to process the message
            flowSnd.addNegativeACK(yyyyMMddHHmm, s);
          }
        }
        if (ecp) {
          if (yyyyMMddHHmmSS < oldestEcpTms) {
            oldestEcpZDT = LocalDateTime.parse("" + yyyyMMddHHmmSS, Const.tmsNumberSec).atZone(UTC);
            oldestEcpTms = yyyyMMddHHmmSS;
          }
        } else {
          if (yyyyMMddHHmmSS < oldestEdxTms) {
            oldestEdxZDT = LocalDateTime.parse("" + yyyyMMddHHmmSS, Const.tmsNumberSec).atZone(UTC);
            oldestEdxTms = yyyyMMddHHmmSS;
          }
        }
      }
    } catch (NumberFormatException |
             DateTimeParseException ex) {
      System.err.println("Error occurred: " + ex + " with input text: " + s);
    }
  }

  public static String makeEmptyLogLine(ZonedDateTime tmsMsg, String loglevel, int completeInterval, String hostname) {
    return tmsMsg.format(Const.utcTmsMillisec) + " " + loglevel + " " +
        "interval=" + String.format("%-3s", completeInterval + ",") + " " + // room for up to 2 digit minutes
        "hostname=" + String.format("%-20s", hostname + ",") + " " +
        "ecpSnd=" + String.format("%-6s", "N/A,") + " " +
        "edxSnd=" + String.format("%-6s", "N/A,") + " " +
        "edxFail=" + String.format("%-6s", "N/A,") + " " +
        "corrSz=" + String.format("%-6s", "N/A,") + " " +
        "diffSnd=" + String.format("%-6s", "0,") + " " +
        "ecpRcv=" + String.format("%-6s", "N/A,") + " " +
        "edxRcv=" + String.format("%-6s", "N/A,") + " " +
        "expRcv=" + String.format("%-6s", "N/A,") + " " +
        "diffRcv=" + String.format("%-6s", "0");
  }


  public static String makeLogline(ZonedDateTime tmsMsg, String loglevel, int completeInterval, Measurement fmSnd, Measurement fmRcv, String hostname) {
    return tmsMsg.format(Const.utcTmsMillisec) + " " + loglevel + " " +
        "interval=" + String.format("%-3s", completeInterval + ",") + " " + // room for up to 2 digit minutes
        "hostname=" + String.format("%-20s", hostname + ",") + " " +
        "ecpSnd=" + String.format("%-6s", fmSnd.getEcpTraffic() + ",") + " " + // room for up to 5 digit traffic
        "edxSnd=" + String.format("%-6s", fmSnd.getEdxTraffic() + ",") + " " + // room for up to 5 digit traffic
        "edxFail="+ String.format("%-6s", fmSnd.getNegativeACK().size() + ",") + " " + // room for up to 5 digit traffic
        "corrSz=" + String.format("%-6s", fmSnd.getCorrelationSize() + ",") + " " +
        "diffSnd=" + String.format("%-6s", fmSnd.diff() + ",") + " " + // room for up to 5 digit diff
        "ecpRcv=" + String.format("%-6s", fmRcv.getEcpTraffic() + ",") + " " + // room for up to 5 digit traffic
        "edxRcv=" + String.format("%-6s", fmRcv.getEdxTraffic() + ",") + " " + // room for up to 5 digit traffic
        "expRcv=" + String.format("%-6s", fmRcv.getEcpExpired().size() + ",") + " " + // room for up to 5 digit traffic
        "diffRcv=" + String.format("%-6s", fmRcv.diff());
  }
}

