package no.statnett.ecp.gm;

import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.Div;
import no.statnett.ecp.utils.EcpEdxLog;
import no.statnett.ecp.utils.Options;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Stream;

public class GapMonitor {
    public static final String VERSION = "v1.2.3";
    public static final int WEEK_MS = 7 * 24 * 60 * 60 * 1000;

    private static boolean alarm = false;
    private static boolean report = false;


    private static void usage() {
        System.out.println("GapMonitor (GM) " + VERSION);
        System.out.println("\nGapMonitor is a tool for monitoring gaps in the ECP-message traffic. Typical scenarios are when a message");
        System.out.println("must be sent every quarter-hour, and if that is not the case, it is important to know. GapMonitor depends");
        System.out.println("upon ECPMsgLogger (EML) to work, and EML must be running and logging to a file. GapMonitor will read the EML");
        System.out.println("log and check if there are gaps in the quarter-hour (QH) traffic based on an input-file provided to GP. GP should");
        System.out.println("be run every quarter-hour (QH) to check for gaps in the previous quarter-hour, a cron-job is a good way to do this.");
        System.out.println("Once every monday a report can be made for previous week (also through a cron-job).");

        System.out.println("\nUsage  : java -jar ekit.jar GM <OPTION> <eml-log> <traffic-file>\n");
        System.out.println(" OPTION:");
        System.out.println("     -a    : Alarm mode. Check for gaps beginning from the previous quarter-hour and search backwords in time.");
        System.out.println("     -r    : Report mode. Check for gaps beginning from the previous 7 days up until start of this day. Default is -a");

        System.out.println("\nThe traffic-file is a simple text-file with the following content:");
        System.out.println("# Ex: NBM-CIM-PTZ12-MTZ35 must be sent every QH from 50V000A to 50V000B. Max 4 in a row can be missing. Min 664 in a week must be found.");
        System.out.println("# Max-missing-QH-in-a-row Min-required-QH-pr-week MessageType RemoteEP LocalEP");
        System.out.println("4 664 NBM-CIM-PTZ12-MTZ35 50V000B 50V000A");
    }

    private static List<String> parseOptions(String[] initialArgs) {
        List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
        alarm = Options.parseBoolean(initArgsList, "-a");
        if (!alarm) {
            report = Options.parseBoolean(initArgsList, "-r");
            if (!report) {
                alarm = true;
            }
        }
        return initArgsList;
    }


    public static void main(String[] args) throws IOException {
        List<String> mandatoryArgs = parseOptions(args);
        if (mandatoryArgs.size() != 2) {
            usage();
            System.err.println("\n\nApart from the options, exactly two filenames must be specified (found " + mandatoryArgs.size() + " filenames), first the EML-log and second the GAP-settings-file");
            System.exit(1);

        }
        File logFile = Options.checkFile(mandatoryArgs.get(0), false);
        File trafficFile = Options.checkFile(mandatoryArgs.get(1), false);

        for (String trafficLine : Files.readAllLines(Path.of(trafficFile.getAbsolutePath()))) {
            if (trafficLine.startsWith("#") || trafficLine.trim().isEmpty()) {
                continue;
            }
            String[] settings = trafficLine.split(" ");
            if (settings.length != 5) {
                System.err.println("Error in settings-file: " + trafficFile + ". Expected 5 fields, found " + settings.length + " fields");
                System.exit(1);
            }
            Traffic traffic = new Traffic(Integer.parseInt(settings[0]), Integer.parseInt(settings[1]), settings[2], settings[3], settings[4]);
            int firstQH;
            int lastQH;
            if (report) {
                // Find start of day in ZonedDateTime
                long startOfPreviousDayMs = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.DAYS).toInstant().toEpochMilli();
                // Find 7 days before start of this day in milliseconds
                long startOf7Days = startOfPreviousDayMs - WEEK_MS;

                firstQH = Div.calcQH(startOf7Days);
                lastQH = Div.calcQH(startOfPreviousDayMs - 1); // siste ms i 7 dagers perioden
            } else {
                firstQH = Div.calcQH(System.currentTimeMillis()) - 1;
                lastQH = firstQH;
            }
            Set<Integer> qhSetWithTraffic = findQhSetWithTraffic(logFile, firstQH, lastQH, traffic);
            if (alarm) {
                StringBuilder sb = new StringBuilder();
                sb.append("{ \"statustime\": \"").append(LocalDateTime.now().format(Const.localTmsMillisec)).append("\", ");
                sb.append("\"status\": ").append(String.format("%-8s", (qhSetWithTraffic.isEmpty() ? "\"ERROR\"," : "\"INFO\","))).append(" ");
                sb.append("\"qh\": ").append(firstQH).append(", ");
                sb.append("\"qhStartUTC\": \"").append(Div.calcZDTFromQH(firstQH)).append("\", ");
                sb.append("\"qhWeek\": \"").append(Div.calcWeek(Div.calcZDTFromQH(firstQH))).append("\", ");
                sb.append("\"treshold\": ").append("0").append(", "); // For alarm the default treshold is 0, therefore a simple case to monitor
                sb.append("\"qhCount\": ").append(qhSetWithTraffic.isEmpty() ? 0 : 1).append(", ");
                sb.append("\"traffic\": \"").append(traffic).append("\" ");
                sb.append("}\n");
                System.out.print(sb);
            } else { // report-mode
                Map<Integer, Integer> foundInRowQH = new HashMap<>();
                Map<Integer, Integer> missingInRowQH = new HashMap<>();
                int statusQh = firstQH;
                boolean found = true;
                for (int qh = firstQH; qh <= lastQH; qh++) {
                    if (qhSetWithTraffic.contains(qh)) {
                        if (!found) {
                            statusQh = qh;
                        }
                        foundInRowQH.merge(statusQh, 1, Integer::sum);
                        found = true;
                    } else {
                        if (found) {
                            statusQh = qh;
                        }
                        missingInRowQH.merge(statusQh, 1, Integer::sum);
                        found = false;
                    }
                }
                int totalFound = foundInRowQH.values().stream().mapToInt(Integer::intValue).sum();
                int totalMissingEvents = (int) missingInRowQH.values().stream().filter(v -> v > traffic.getMaxMissingQHInARow()).count();
                boolean error = totalFound < traffic.getMinRequiredQHInAWeek() || totalMissingEvents > 0;
                ZonedDateTime zdt = Div.calcZDTFromQH(firstQH);
                StringBuilder sb = new StringBuilder();
                sb.append("{ \"statustime\": \"").append(LocalDateTime.now().format(Const.localTmsMillisec)).append("\", ");
                sb.append("\"status\": ").append(String.format("%-8s", (error ? "\"ERROR\"," : "\"INFO\","))).append(" ");
                sb.append("\"startUTC\": \"").append(zdt).append("\", ");
                sb.append("\"startWeekNo\": \"").append(Div.calcWeek(zdt)).append("\", ");
                sb.append("\"qh-total-required\": ").append(traffic.getMinRequiredQHInAWeek()).append(", ");
                sb.append("\"qh-total-found\": ").append(totalFound).append(", ");
                sb.append("\"qh-max-missing-in-row\": ").append(traffic.getMaxMissingQHInARow()).append(", ");
                sb.append("\"qh-missing-in-row-events\": ").append(totalMissingEvents).append(", ");
                sb.append("\"traffic\": \"").append(traffic).append("\" ");
                sb.append("}\n");
                System.out.print(sb);
            }
        }
    }

    private static Set<Integer> findQhSetWithTraffic(File logFile, int firstQH, int lastQH, Traffic traffic) throws IOException {
        Set<Integer> qhSet = new HashSet<>();
        Stream<String> lines = Files.lines(logFile.toPath());

        for (String s : (Iterable<String>) lines::iterator) {
            int qh = findQHInLogLine(s);
            if (qh >= firstQH && qh <= lastQH) {
                String localEp = findLocalEpInLogLine(s);
                String remoteEp = findRemoteEpInLogLine(s);
                String messageType = findMessageTypeInLogLine(s);
                String stat = findStatInLogLine(s);
                boolean okOrSmallDelay = stat.equals("OK") || stat.equals("DL"); // Skal jeg ta hensyn til dette??
                if (localEp.equals(traffic.getLocalEP()) && remoteEp.equals(traffic.getRemoteEP()) && messageType.equals(traffic.getMessageType()) /*&& okOrSmallDelay*/) {
                    qhSet.add(qh);
                }
            }
        }
        return qhSet;
    }

    private static String findStatInLogLine(String logLine) {
        return EcpEdxLog.getString(logLine, "stat=", ",");
    }

    private static int findQHInLogLine(String logline) {
        try {
            String qhNo = EcpEdxLog.getString(logline, "qh=", ",");
            if (qhNo == null) { // TODO Fjernes om en stund - når v1.5.0 av EML har vært i bruk i minst 10 uker alle steder
                qhNo = EcpEdxLog.getString(logline, "qhNo=", ",");
            }
            return Integer.parseInt(qhNo);
        } catch (NumberFormatException e) {
            return -1;
        }
    }

    private static String findLocalEpInLogLine(String logLine) {
        String local = EcpEdxLog.getString(logLine, "loc=", ",");
        if (local == null) { // TODO Fjernes om en stund - når v1.5.0 av EML har vært i bruk i minst 10 uker alle steder
            local = EcpEdxLog.getString(logLine, "local=", ",");
        }
        return local;
    }

    private static String findRemoteEpInLogLine(String logLine) {
        String remote = EcpEdxLog.getString(logLine, "rem=", ",");
        if (remote == null) { // TODO Fjernes om en stund - når v1.5.0 av EML har vært i bruk i minst 10 uker alle steder
            remote = EcpEdxLog.getString(logLine, "remote=", ",");
        }
        return remote;
    }

    private static String findMessageTypeInLogLine(String logLine) {
        String messageType = EcpEdxLog.getString(logLine, "type=", ",");
        if (messageType == null) { // TODO Fjernes om en stund - når v1.5.0 av EML har vært i bruk i minst 10 uker alle steder
            messageType = EcpEdxLog.getString(logLine, "msgType=", ",");
        }
        return messageType;
    }

}
