package no.statnett.ecp.pf;

import jakarta.jms.JMSException;
import no.statnett.ecp.brs.actions.ListQueues;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.utils.LogOut;
import no.statnett.ecp.utils.Options;
import no.statnett.ecp.utils.URLParser;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;

public class Performance {

    public static final String VERSION = "v1.0.5";
    public static final DateTimeFormatter HHMMSS = DateTimeFormatter.ofPattern("HH:mm:ss");
    public static final int DEFAULT_MAX_QUEUE_SIZE_MB = 2 * 1024; // 2GB
    public static final int DEFAULT_MAX_NOT_CONSUMED_SIZE_MB = 512;


    public static int REGULATE_LOOP_MS = 50;
    public static int PRINT_LOOP_MS = 5000;
    public static int INITAL_CONSUME_LOOP_MS = 1000;

    // Can also be controlled by command-line arguments
    public static String queueName;
    public static long bufferLimitB = 0; // 1 GB
    public static int noOfConsumers = 1;
    public static long maxQueueSizeB = 2L * 1024 * 1024 * 1024;
    public static String options = null;
    public static int noOfProducers = 1;
    public static int sizeB = 1024;
    public static int ttl = 60;
    public static int maxMBNotConsumed = DEFAULT_MAX_NOT_CONSUMED_SIZE_MB;
    public static boolean persistence = false;
    public static boolean regulateProduceRate = false;

    private static final List<Consumer> consumerGroup = new ArrayList<>();
    private static final List<Producer> producerGroup = new ArrayList<>();

    private static List<String> parseOptions(String[] initialArgs) {
        List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
        long maxHeapMB = Runtime.getRuntime().maxMemory() / (1024 * 1024);
        // The buffer-limit is default 512MB, but can range from 1MB to 50% of maxHeapSize
        int maxBufferSizeMB = (int) maxHeapMB / 2;
        int defaultBufferSizeMB = Math.min(512, maxBufferSizeMB);
        maxMBNotConsumed = Options.parseInt(initArgsList, "-a", 0, 10 * DEFAULT_MAX_QUEUE_SIZE_MB, DEFAULT_MAX_NOT_CONSUMED_SIZE_MB);
        bufferLimitB = 1024L * 1024L * Options.parseInt(initArgsList, "-b", 1, maxBufferSizeMB, defaultBufferSizeMB);
        noOfConsumers = Options.parseInt(initArgsList, "-c", 0, 64, 1);
        maxQueueSizeB = 1024L * 1024L * Options.parseInt(initArgsList, "-m", 1, 10 * DEFAULT_MAX_QUEUE_SIZE_MB, DEFAULT_MAX_QUEUE_SIZE_MB);
        options = Options.parseString(initArgsList, "-o");
        noOfProducers = Options.parseInt(initArgsList, "-p", 0, 64, 1);
        sizeB = 1024 * Options.parseInt(initArgsList, "-s", 1, 1024 * 1024, 1);
        ttl = 1000 * Options.parseInt(initArgsList, "-t", 1, 3600, 60);
        persistence = Options.parseBoolean(initArgsList, "-x");
        return initArgsList;
    }

    private static void usage() {
        System.out.println("Performance (PF) " + VERSION + " is a tool for testing the performance of an Artemis-server. The goal");
        System.out.println("is to find the limits of the Artemis-broker and to see how it must be tuned and how latency affects");
        System.out.println("the throughput. The tool offers a number of options to vary the load/run");
        System.out.println();
        System.out.println("Usage  : java -jar ekit.jar PF [OPTIONS] QUEUE AMQP-URL HTTP-URL\n");
        System.out.println("\n\nThe options and arguments:");
        System.out.println(" OPTIONS   : ");
        System.out.println("             -a<size>   : size of max non-consumed messages (in MB). Default 512MB");
        System.out.println("             -b<size>   : size of the produce-buffer (in MB). Default 512MB");
        System.out.println("             -c<number> : number of consumers. Default 1");
        System.out.println("             -m<size>   : maximum size of queue (in MB). Default 2GB. Tries to prevent a run-away test");
        System.out.println("             -o<string> : options to be passed to the connection.");
        System.out.println("             -p<number> : number of producers. Default 1");
        System.out.println("             -s<size>   : size of the message (in KB). Default 1024KB");
        System.out.println("             -t<sec>    : time-to-live (in seconds). Default 60s");
        System.out.println("             -x         : enable persistence. Default false");

        System.out.println(" QUEUE     : The name of the queue to be used for this run");
        System.out.println(" AMQP-URL  : The URL used to transmit messages (Artemis-broker):    amqp[s]://user:pass@host:port");
        System.out.println(" HTTP-URL  : The URL used to retrieve queue-info (Artemis-console): http[s]://user:pass@host:port");
        System.out.println();
        System.out.println("\nExample 1:  Run PF with all default options set, queue name is 'PFQ'");
        System.out.println("\tjava -jar ekit.jar PF PFQ amqp://admin:pw@localhost:5672 http://admin:pw@localhost:8161");
        System.out.println("\nExample 2:  Run PF with other options");
        System.out.println("\tjava -jar ekit.jar PF -c4 -p2 -o \"jms.prefetchPolicy.all=10\" -s1024 -x -z amqp://admin:pw@localhost:5672 http://admin:pw@localhost:8161");
    }


    public static void main(String[] args) throws JMSException, IOException, NoSuchAlgorithmException, KeyManagementException, InterruptedException {

        List<String> mandatoryArgs = parseOptions(args);
        if (mandatoryArgs.size() < 3) {
            if (args.length == 0) {
                usage();
            } else {
                System.out.println(LogOut.e() + " Missing mandatory arguments: QUEUE, AMQP-URL and/or HTTP-URL");
            }
            System.exit(1);
        }
        // Read mandatory args:
        queueName = mandatoryArgs.get(0);

        Map<String, String> amqpConfig = URLParser.parse(mandatoryArgs.get(1), true);
        amqpConfig.put("options", options);

        Map<String, String> httpConfig = URLParser.parse(mandatoryArgs.get(2), false);

        long maxHeapMB = Runtime.getRuntime().maxMemory() / (1024 * 1024);
        long maxBufferMB = bufferLimitB / (1024 * 1024);
        long maxQueueSizeMB = maxQueueSizeB / (1024 * 1024);

        System.out.println("Performance " + VERSION + " starts. The configuration is listed below.\n");
        System.out.println("============================================================================================================================================================");
        System.out.println("Limits         : max-heap: " + maxHeapMB + " MB, max-not-consumed-limit (-a): " + maxMBNotConsumed + "MB, produce-buffer (-b): " + maxBufferMB + " MB");
        System.out.println("Artemis-broker : host:" + amqpConfig.get("host") + ", amqpPort: " + amqpConfig.get("port") + ", user/pw: " + amqpConfig.get("user") + "/" + amqpConfig.get("password"));
        System.out.println("Artemis-console: host:" + httpConfig.get("host") + ", amqpPort: " + httpConfig.get("port") + ", user/pw: " + httpConfig.get("user") + "/" + httpConfig.get("password"));
        System.out.println("AMQP-config    : options (-o): " + (options == null ? "NONE" : options));
        System.out.println("Queue          : name: " + queueName + ", maxSizeAllowed (-m): " + maxQueueSizeMB + "MB");
        System.out.println("Message        : msgSize (-s): " + sizeB / 1024 + "KB, time-to-live (-t): " + ttl / 1000 + "s, persistence (-x): " + persistence);
        System.out.println("Load           : #producers (-p): " + noOfProducers + ", #consumers (-c): " + noOfConsumers);
        System.out.println("============================================================================================================================================================");

        // Start consumers
        cleanQueueBeforeStart(amqpConfig, httpConfig);


        // Now the performance test can start
        long startTms = System.currentTimeMillis();
        long printTms = startTms;

        // Start producers
        ProduceCounter produceCounter = new ProduceCounter(sizeB);
        for (int i = 0; i < noOfProducers; i++) {
            Producer producer = new Producer(amqpConfig, queueName, sizeB, ttl, persistence, produceCounter, maxMBNotConsumed);
            Thread massProduceThread = new Thread(producer, "MP-" + i);
            massProduceThread.start();
            producerGroup.add(producer);
        }

        // Performance-goal: Keep the size of the queue below max-size of queue and run produce as fast as possible
        while (true) {
            Thread.sleep(REGULATE_LOOP_MS);

            int consumeMsgCounterTotal = consumerGroup.stream().mapToInt(Consumer::getMsgConsumedCounterTotal).sum();
            int producedMsgCounterTotal = produceCounter.getMsgProducedCounterTotal();
            int queueSize = queueSize(httpConfig, queueName);
            // Some input useful to regulate production
            producerGroup.stream().forEach(mc -> mc.setRegulationInfo(queueSize, producedMsgCounterTotal - consumeMsgCounterTotal));


            // Print result
            long sinceLastPrintMs = System.currentTimeMillis() - printTms;
            if (sinceLastPrintMs > PRINT_LOOP_MS) {
                long totalMs = System.currentTimeMillis() - startTms;
                int producedMsgCounter = produceCounter.resetMsgProducedCounter();
                long throughputProducedNow = ((long) producedMsgCounter * sizeB) / sinceLastPrintMs;
                long throughputProducedTotal = ((long) producedMsgCounterTotal * sizeB) / totalMs;
                int skip = produceCounter.resetSkip();
                String prodStr = String.format(
                        "Production: [Skip: %6d ms, Count: %7d, Throughput KB/s (now/total): %7d/%7d]",
                        skip,
                        producedMsgCounter,
                        throughputProducedNow,
                        throughputProducedTotal
                );


                int consumeMesageCount = consumerGroup.stream().mapToInt(Consumer::resetMsgConsumedCounter).sum();
                long throughputConsumeNow = ((long) consumeMesageCount * sizeB) / sinceLastPrintMs;
                long throughputConsumeTotal = ((long) consumeMsgCounterTotal * sizeB) / totalMs;
                String consStr = String.format(
                        "Consumption: [Count: %7d, Throughput KB/s (now/total): %7d/%7d]",
                        consumeMesageCount,
                        throughputConsumeNow,
                        throughputConsumeTotal
                );
                System.out.println(HHMMSS.format(Instant.now().atZone(ZoneId.systemDefault())) + "   Size: " + String.format("%6d", queueSize) + "      " + prodStr + "       " + consStr);
                printTms = System.currentTimeMillis();
            }
        }
    }

    private static void cleanQueueBeforeStart(Map<String, String> amqpConfig, Map<String, String> httpConfig) throws JMSException, InterruptedException, NoSuchAlgorithmException, IOException, KeyManagementException {
        for (int i = 0; i < noOfConsumers; i++) {
            Consumer consumer = new Consumer(amqpConfig, queueName);
            Thread massConsumerThread = new Thread(consumer, "MC-" + i);
            massConsumerThread.start();
            consumerGroup.add(consumer);
        }

        // Empties the queue before start
        if (noOfConsumers > 0) {
            long initialSize;
            boolean first = true;
            do {
                Thread.sleep(INITAL_CONSUME_LOOP_MS);
                initialSize = queueSize(httpConfig, queueName);
                if (initialSize > 0) {
                    if (first) {
                        first = false;
                        System.out.print("Emptying queue before start, we have " + initialSize + " messages");
                    } else {
                        System.out.print(".");
                    }
                }
            } while (initialSize > 0);
            if (!first) {
                System.out.println("OK");
            }
        }

        consumerGroup.stream().forEach(mc -> {
            mc.resetMsgConsumedCounter();
            mc.resetMsgConsumedCounterTotal();
        });
    }

    private static int queueSize(Map<String, String> httpConfig, String queueName) throws NoSuchAlgorithmException, IOException, KeyManagementException {
        String protocol = httpConfig.get("protocol");
        String host = httpConfig.get("host");
        String port = httpConfig.get("port");
        String user = httpConfig.get("user");
        String pw = httpConfig.get("password");
        List<QueueInfo> qiList = ListQueues.Retrieve.retrieve(protocol, host, port, user, pw);
        return qiList.stream().filter(qi -> qi.getQueueName().equals(queueName)).findFirst().get().getSize();
    }
}


