package no.statnett.ecp.qm;

import no.statnett.ecp.utils.*;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.*;

/**
 * QueueMonitoring (QM) will monitor queues in an Artemis broker (ECP 4.14+). The tool will output a log line for each monitoring cycle.
 * Each line is a JSON-object containing the status of the queues at that time.
 * The output will be either INFO (everything is ok) or ERROR (some queue is not OK). The output will be appended to a file every minute.
 * There are two cases where QM will report ERROR:
 * 1) If a queue is not being dequeued at all - this report will be given after 1 minute of no dequeuing
 * 2) If a queue is being processed to slow - this report will be given after 5 minutes of increasing queue-size
 * <p>
 * The QM will run in a forever loop, checking the queues every minute.
 * <p>
 * To achieve this we need to create storage for the queue-information
 */

public class QueueMonitor {

  public static final String VERSION = "v1.2.0";
  public static final int MAX_MINUTES_IN_QMI_HISTORY = 70;
  private static final long DELAY_MINUTES = 60;
  private static Map<String, QMIHistory> qmiHistoryMap = new HashMap<>();

  private static File logFile;
  private static String rotationType = null;
  private static int noOfLogFiles = 0;
  private static int noOfDaysDelayRotation = 0;
  private static boolean compress = false;
  private static String user;
  private static String pw;
  private static URL url;

  private static List<String> ignoreQs = new ArrayList<>();
  private static List<String> delayQs = new ArrayList<>();
  private static int stopMinutes = 1;

  private static boolean debug = false;

  private static List<String> parseOptions(String[] initialArgs) {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));

    rotationType = Options.parseString(initArgsList, "-r");
    if (rotationType != null && !rotationType.matches("^(day|week|month)$")) {
      System.err.println("Rotate option (-r) must be either 'day', 'week' or 'month' - exiting");
      System.exit(1);
    }
    noOfLogFiles = Options.parseInt(initArgsList, "-m", 1, 1000, 10);
    noOfDaysDelayRotation = Options.parseInt(initArgsList, "-n", 0, 1000, 0);
    compress = Options.parseBoolean(initArgsList, "-z");

    String ignoreStr = Options.parseString(initArgsList, "-i");
    if (ignoreStr != null) {
      ignoreQs = List.of(ignoreStr.split(","));
    }
    String delayStr = Options.parseString(initArgsList, "-d");
    if (delayStr != null) {
      delayQs = List.of(delayStr.split(","));
    }

    stopMinutes = Options.parseInt(initArgsList, "-s", 1, 60, 1);

    debug = Options.parseBoolean(initArgsList, "-x");

    return initArgsList;
  }

  private static void parseMandatoryArg(List<String> mandatoryArgs) throws MalformedURLException {
    if (mandatoryArgs.size() < 2) {
      usage();
      System.exit(1);
    }
    logFile = new File(mandatoryArgs.get(0));
    if (!logFile.exists()) {
      try {
        Files.writeString(logFile.toPath(), "");
      } catch (IOException e) {
        System.out.println(LogOut.e() + "Cannot write to '" + mandatoryArgs.get(0) + "' (exception: " + e.getMessage() + ")");
        System.exit(1);
      }
    }

    URL initialURL = new URL(mandatoryArgs.get(1));
    url = new URL(initialURL.getProtocol(), initialURL.getHost(), initialURL.getPort(), "/metrics");
    if (initialURL.getUserInfo() != null) {
      user = initialURL.getUserInfo().split(":")[0];
      pw = initialURL.getUserInfo().split(":")[1];
    }
  }


  private static void sleep(LocalDateTime monitorLDT) {
    LocalDateTime nextRun = monitorLDT.plusMinutes(1);
    long diff = nextRun.toInstant(ZoneOffset.UTC).toEpochMilli() - LocalDateTime.now().toInstant(ZoneOffset.UTC).toEpochMilli();
    if (diff > 0) {
      try {
        Thread.sleep(diff);
      } catch (InterruptedException e) {
        e.printStackTrace();
      }
    }
  }

  public static void main(String[] args) throws InterruptedException, IOException {
    List<String> mandatoryArgs = parseOptions(args);
    parseMandatoryArg(mandatoryArgs);
    LocalDateTime monitorLDT = LocalDateTime.now();
    while (true) {
      try {
        monitorLDT = LocalDateTime.now();
        // Retrieve data from Artemis, and build the initial QMI-map, next step is to compare with QMIHistory
        Map<String, QMI> qmiNowMap = buildQMINowMap(monitorLDT);
        // These queues have size > 0 and *could* be in an ERROR situation:
        boolean allQueuesAreOK = true; // Assume everything is OK until proven otherwise
        List<String> queueOutput = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        // Remove all queues in QMIHistoryMap that are not in qmiNowMap's keyset - because those queues are OK now
        qmiHistoryMap.keySet().removeIf(queueName -> !qmiNowMap.containsKey(queueName));
        for (String queueName : qmiNowMap.keySet()) {
          QMI qmiNow = qmiNowMap.get(queueName);
          QMIHistory qmiHistory = calculateState(queueName, qmiNow, monitorLDT, queueOutput);
          if (!qmiHistory.isOk()) {
            allQueuesAreOK = false; // At least one queue is not OK (ERROR)
            String reason = qmiHistory.isStop() ? "stop" : "slow";
            sb.append(monitorLDT.format(Const.localTmsSecNotISO) + " ERROR [" + reason + "] =>" + qmiHistory + "\n");
          } else {
            sb.append(monitorLDT.format(Const.localTmsSecNotISO) + " OK           =>" + qmiHistory + "\n");
          }
        }
        if (debug) {
          if (qmiNowMap.isEmpty()) {
            sb.append(monitorLDT.format(Const.localTmsSecNotISO) + " OK           => No queues found with size > 0\n");
          }
          System.out.print(sb);
        }
        // Write output to log-file
        printOutput(monitorLDT, allQueuesAreOK, queueOutput);
        sleep(monitorLDT); // sleep until 1 minute has passed since monitorLDT
      } catch (Exception e) {
        System.err.println("An error occurred in the main loop: " + e.getMessage() + " - will continue. Stacktrace:" + Div.onelinerStacktrace(e.getStackTrace(), "no.statnett"));
        sleep(monitorLDT); // sleep until 1 minute has passed since monitorLDT
      }
    }
  }

  // This map contains the NOW/CURRENT status of the queues (QMI). We remove those queues/QMI which are of no interest (ignore-queues and size == 0)
  private static Map<String, QMI> buildQMINowMap(LocalDateTime monitorLDT) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    // retrieve all queue-metrics, parse it into QMI-objects representing "now" (monitorTms)
    Map<String, QMI> qmiNowMap = buildQMINowMap(EcpHTTP.getECPHTTResponse("GET", url, user, pw, null), monitorLDT);
    // remove all queues that should be ignored
    qmiNowMap.keySet().removeIf(queueName -> ignoreQs.stream().anyMatch(queueName::contains));
    // remove all queue that has size 0
    qmiNowMap.keySet().removeIf(queueName -> qmiNowMap.get(queueName).getSize() == 0);
    // mark all noConsumerQs with a delayedMonitoring-flag
    qmiNowMap.keySet().forEach(queueName -> {
      if (delayQs.stream().anyMatch(queueName::contains)) {
        qmiNowMap.get(queueName).setDelayedMonitoring(true);
      }
    });
    return qmiNowMap;
  }

  private static QMIHistory calculateState(String queueName, QMI qmiNow, LocalDateTime monitorLDT, List<String> queueOutput) {
    QMIHistory qmiHistory = qmiHistoryMap.get(queueName);
    if (qmiHistory == null) {
      qmiHistory = new QMIHistory(queueName, qmiNow.isDelayedMonitoring());
      qmiHistoryMap.put(queueName, qmiHistory);
    } else {
      qmiHistory.setOk(true);
      qmiHistory.setStop(false);
      qmiHistory.setSlow(false);
    }
    qmiHistory.removeWithHigherDequeuCount(qmiNow); // Remove QMIs that comes from before restart of broker
    qmiHistory.removeOlderThan(monitorLDT.minusMinutes(MAX_MINUTES_IN_QMI_HISTORY)); // After a while, history of QMIs can be removed
    qmiHistory.addQMI(qmiNow);

    // THE logic of QM:
    if (qmiHistory.hasStopped(qmiNow.isDelayedMonitoring() ? DELAY_MINUTES : stopMinutes)) {
      qmiHistory.setOk(false);
      qmiHistory.setStop(true);
      // full stop
      queueOutput.add("{\"name\": \"" + queueName + "\", \"size\": " + qmiNow.getSize() + ", \"cons\": " + qmiNow.getConsumerCount() + ", \"reason\": " + "\"stopped\"}");
    } else {
      if (qmiHistory.isSlow(qmiNow.isDelayedMonitoring() ? DELAY_MINUTES : 10)) {
        qmiHistory.setOk(false);
        qmiHistory.setSlow(true);
        // too slow
        queueOutput.add("{\"name\": \"" + queueName + "\", \"size\": " + qmiNow.getSize() + ", \"cons\": " + qmiNow.getConsumerCount() + ", \"reason\": " + "\"slow\"}");
      }
    }

    return qmiHistory;
  }

  private static void printOutput(LocalDateTime monitorLDT, boolean ok, List<String> queueOutput) throws IOException {
    String nowAsTimestamp = monitorLDT.format(Const.localTmsSecNotISO);
    long currentTimeSec = System.currentTimeMillis() / 1000;
    String output = "{\"statustime\": \"" + nowAsTimestamp + "\", \"timestamp\": " + currentTimeSec + ", \"status\": \"" + (ok ? "OK" : "ERROR") + "\"";
    if (!ok) {
      output += ", \"queues\": [" + String.join(",", queueOutput) + "]";
    }
    output += "}";
    Files.writeString(logFile.toPath(), output + "\n", StandardOpenOption.APPEND);

    if (rotationType != null) {
      LogRotator logRotator = new LogRotator(logFile, monitorLDT.atZone(ZoneId.systemDefault()), noOfDaysDelayRotation);
      logRotator.rotateLogs(rotationType);
      logRotator.deleteExcessLogs(noOfLogFiles);
      if (compress)
        logRotator.compress();
    }
  }

  private static Map<String, QMI> buildQMINowMap(String response, LocalDateTime nowLDT) {
    Map<String, QMI> qmiNowMap = new HashMap<>();
    List<String> lines = List.of(response.split("\n"));

    for (String line : lines) {
      if (line.startsWith("artemis_messages_acknowledged") || line.startsWith("artemis_message_count") || line.startsWith("artemis_consumer_count")) {
        String queueName = parseQueueName(line);
        QMI qmi = qmiNowMap.get(queueName);
        if (qmi == null) {
          qmi = new QMI(queueName, nowLDT.toInstant(ZoneOffset.UTC).toEpochMilli());
          qmiNowMap.put(queueName, qmi);
        }
        if (line.startsWith("artemis_messages_acknowledged")) {
          qmi.setDequeueCount(parseMetric(line));
        } else if (line.startsWith("artemis_consumer_count")) {
          qmi.setConsumerCount(parseMetric(line));
        } else {
          qmi.setSize(parseMetric(line));
        }
      }
    }
    return qmiNowMap;
  }


  private static int parseMetric(String line) {
    // line examples:
    // artemis_messages_acknowledged{address="edx.internal.delivery.ecp-endpoint",broker="ECCo SP Artemis",queue="edx.internal.delivery.ecp-endpoint",} 0.0
    // artemis_message_count{address="edx.internal.delivery.ecp-endpoint",broker="ECCo SP Artemis",queue="edx.internal.delivery.ecp-endpoint",} 0.0
    return (int) Float.parseFloat(line.substring(line.lastIndexOf(" ") + 1));
  }

  private static String parseQueueName(String line) {
    // line examples:
    // artemis_messages_acknowledged{address="edx.internal.delivery.ecp-endpoint",broker="ECCo SP Artemis",queue="edx.internal.delivery.ecp-endpoint",} 0.0
    // artemis_message_count{address="edx.internal.delivery.ecp-endpoint",broker="ECCo SP Artemis",queue="edx.internal.delivery.ecp-endpoint",} 0.0
    int beginIndex = line.indexOf("queue=") + 7;
    int endIndex = line.indexOf("\"", beginIndex);
    return line.substring(beginIndex, endIndex);
  }

  private static void usage() {
    System.out.println("QueueMonitor (QM) " + VERSION + " will monitor queues in an Artemis broker (ECP 4.14+). The tool will output");
    System.out.println("an ERROR message if one or more queues are not dequeued for 1 minute (default or use -s option) or if the ");
    System.out.println("queue-size is not decreasing within the last 10 minutes.\n");
    System.out.println("The tool will run in a forever loop, checking the queues every minute and print the status to a log-file every minute.");
    System.out.println("To allow the tool to be triggered by cron-job, the tool will only run if it detects that no other instance is running");
    System.out.println("A running instance is detected by checking that the modified timestamp of the log-file is younger than 70 sec.");
    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar QM <OPTIONS> <LOG-FILE> <ECP-URL> \n");
    System.out.println("\n\nThe arguments:");
    System.out.println(" OPTIONS      ");
    System.out.println("    -i <IGNORE-Q>      : List of strings, commaseparated. If a queue-name contains any of these strings, the queue will be ignored");
    System.out.println("    -d <DELAY-Q>       : List of strings, commaseparated. If a queue-name contains any of these strings the monitoring will be");
    System.out.println("                         delayed by 60 min. Useful for queues that are purged by BRS or similar tools, to prevent false alarms");
    System.out.println("    -s<noMinutes>      : signal stop situation if lasted for <noMinutes>. Default is 1 minute. Set to higher in test-environments");
    System.out.println("    -r day|week|month  : Rotate log-file every day, week or month. Requires -o option");
    System.out.println("    -m<noLogFiles>     : Allow maximum number of log-files. Default is 10. Requires -r option");
    System.out.println("    -n<noDays>         : Do not rotate data younger than noDays. Default is 0. If set higher, the output log will");
    System.out.println("                         always contain the last noDays of data, even if the log-file is rotated. This is useful");
    System.out.println("                         for analysis of the last days of data (GapMonitor). Requires -r option");
    System.out.println("    -z                 : Compress rotated logs. Requires -r option");
    System.out.println("    -x                 : Will print the internal state of QM to stdout, useful for debugging");
    System.out.println("MANDATORY ARGUMENTS");
    System.out.println("     <LOG-FILE>        : File to log output to - new data will be appended to the file every minute");
    System.out.println("     <ECP-URL>         : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("\nExample 1:  Run QueueMonitor with default options");
    System.out.println("\tjava -jar ekit.jar QM https://localhost:8161");
    System.out.println("\nExample 2:  Run QueueMonitor with ignore and delay options. Ignore DLQ/ECP-Expiry queues and delay monitoring of reply-queues");
    System.out.println("\tjava -jar ekit.jar QM -i DLQ,Expiry -c reply https://localhost:8161");
    System.out.println("\nExample 3: Run QueueMonitor with log-rotation every day and max 10 log-files. Compress rotated logs");
    System.out.println("\tjava -jar ekit.jar QM -r day -m10 -z https://localhost:8161");
  }
}
