/*
 * Copyright (C) Red Hat, Inc.
 * http://www.redhat.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package no.statnett.ecp.qp;

import jakarta.jms.*;
import jakarta.jms.Queue;
import no.statnett.ecp.utils.*;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

public class QueueProtector {


    public static final String VERSION = "v1.1.9";

    // OPTIONS
    private static int maxPrSender = 0; // default
    private static int maxAgeSeconds = 30; // default
    private static int maxMessagesInQueue = 100; // default
    private static File summaryFile;
    private static File directory;
    private static boolean restore = false; // default
    private static boolean continuous = false; // default

    // MANDATORY QUEUENAME
    private static String queueName = null;

    public static void main(String[] args) throws IOException {
        List<String> mandatoryArgs = parseOptions(args);
        if (mandatoryArgs.size() < 2) {
            if (args.length == 0) {
                usage();
            } else {
                System.out.println(LogOut.e() + " Missing mandatory arguments: QUEUE and/or BROKER");
            }
            System.exit(1);
        }
        queueName = mandatoryArgs.get(0);
        Map<String, String> config = URLParser.parse(mandatoryArgs.get(1), true);

        Connection connection = null;
        Session session = null;
        LoopState loopState = new LoopState(continuous, 500); // start with 500ms waitMs
        do {
            try {
                loopState.sleep();

                connection = Broker.createAndStartConnection(config); // Connect to the broker and queue
                session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE); // transacted = false, acknowledgeMode = AUTO_ACKNOWLEDGE (only a consumer care about this)
                Queue queue = session.createQueue(queueName);

                Counters counters = new Counters();
                Browse.updateCountersAndWaitMs(session, queue, counters, loopState);

                if (protectMode(counters.getBrowserCounter(), counters)) {
                    long startTime = System.currentTimeMillis();
                    Set<String> senders = null;
                    if (maxPrSender > 0) {
                        // Find list of keys with more than maxPrSender messages
                        senders = counters.getNoMsgPrSender().entrySet().stream().filter(e -> e.getValue() >= maxPrSender).map(Map.Entry::getKey).collect(Collectors.toSet());
                    } else {
                        senders = new HashSet<>();
                        senders.add("ALL"); // will make sure one loop is run in the Protect.consume()-method - a little bit of a hack
                    }
                    int noConsumed = Protect.consume(session, queue, directory, senders, maxAgeSeconds);
                    long consumeTime = System.currentTimeMillis() - startTime;
                    long msPrConsume = noConsumed == 0 ? 0 : consumeTime / noConsumed;
                    System.out.println(LogOut.i() + "Consumed " + noConsumed + " msg in " + consumeTime + " ms (" + msPrConsume + " ms/msg)");
                    logProtectStateToJson(counters, counters.getBrowserCounter(), noConsumed, msPrConsume);
                    loopState.setLastMode(Mode.PROTECT);
                } else {
                    if (restore && loopState.getLastMode() != Mode.PROTECT) { // Only restore if previous mode was not PROTECT
                        Map<String, Map<String, File>> restoreMessages = Restore.findMessagesToRestore(directory);
                        if (!restoreMessages.isEmpty()) {
                            if (counters.getBrowserCounter() < 10) { // Do not restore if
                                System.out.println(LogOut.i() + "Found " + restoreMessages.size() + " msg to restore, will start restoring...");
                                long startTime = System.currentTimeMillis();
                                int restoreCounter = Restore.restore(session, queue, restoreMessages, startTime, directory, loopState.getWaitMs());
                                loopState.setMsgRestored(restoreCounter);
                                long restoreTime = System.currentTimeMillis() - startTime;
                                long msPrRestore = restoreTime / restoreCounter;
                                System.out.println(LogOut.i() + "Restored " + restoreCounter + " of " + restoreMessages.size() + " msg in " + restoreTime + " ms (" + msPrRestore + " ms/msg) (waitMs=" + loopState.getWaitMs() + ")");
                                logRestoreStateToJson(counters, counters.getBrowserCounter(), restoreMessages.size(), restoreCounter, msPrRestore, loopState.getWaitMs());
                                loopState.setLastMode(Mode.RESTORE);
                            } else {
                                System.out.println(LogOut.i() + "Found " + restoreMessages.size() + " msg to restore, but still " + counters.getBrowserCounter() + " in queue. Restore is postponed");
                                logRestorePostponedStateToJson(counters, counters.getBrowserCounter(), restoreMessages.size(), loopState.getWaitMs());
                                loopState.setLastMode(Mode.RSTWAIT);
                            }
                        } else {
                            logMonitorStateToJson(loopState, counters, counters.getBrowserCounter());
                            loopState.setLastMode(Mode.MONITOR);
                        }
                    } else {
                        logMonitorStateToJson(loopState, counters, counters.getBrowserCounter());
                        loopState.setLastMode(Mode.MONITOR);

                    }
                }
                if (!loopState.isContinuous()) {
                    break;
                }
            } catch (Exception t) {
                System.out.println(LogOut.e() + "Error occurred: " + t + ", Stacktrace: " + Div.onelinerStacktrace(t.getStackTrace(), "no.statnett.ecp"));
            } finally {
                try {
                    if (session != null)
                        session.close();
                    if (connection != null)
                        connection.close();
                } catch (JMSException e) {
                    System.out.println(LogOut.e() + "Error occurred while close JMS-connection:" + e + ", Stacktrace: " + Div.onelinerStacktrace(e.getStackTrace(), "no.statnett.ecp"));
                }
            }
        } while (continuous);
    }


    private static List<String> parseOptions(String[] initialArgs) {
        List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
        maxMessagesInQueue = Options.parseInt(initArgsList, "-q", 0, 400, 100);
        maxPrSender = Options.parseInt(initArgsList, "-s", 0, 300, 0);
        if (maxMessagesInQueue < maxPrSender) {
            System.out.println(LogOut.e() + "Option -q (queue) is less than -s (sender) - and will therefore not have any effect");
            System.exit(1);
        }
        maxAgeSeconds = Options.parseInt(initArgsList, "-t", 0, 84600, 30);
        String summaryFilename = Options.parseString(initArgsList, "-o");
        if (summaryFilename != null)
            summaryFile = Options.checkFile(summaryFilename, true);
        String directoryName = Options.parseString(initArgsList, "-m");
        if (directoryName != null)
            directory = Options.checkDirectory(directoryName, true);
        restore = Options.parseBoolean(initArgsList, "-e");
        if (restore && (maxAgeSeconds == 0 && (maxMessagesInQueue == 0 || maxPrSender == 0))) {
            System.out.println(LogOut.w() + "Option -e (restore) was chosen but rejected, because -t is 0 and -s or -q is also 0 - which could create a forever loop");
            System.exit(1);
        }
        if (restore && directory == null) {
            System.out.println(LogOut.e() + "Option -e (restore) was chosen but rejected, because -m <directory> is missing - no files could be restored to queue");
            System.exit(1);
        }
        continuous = Options.parseBoolean(initArgsList, "-f");
        return initArgsList;
    }


    private static boolean protectMode(int browserCounter, Counters counters) {

        // Based on the counters found above, we retrieve som key numbers
        int senderWithHighestCount = counters.getNoMsgPrSender().values().stream().max(Integer::compareTo).orElse(0);
        int oldestMsgSec = counters.getNoMsgPrAgeSec().keySet().stream().max(Long::compareTo).orElse(0L).intValue();


        boolean protect = false;
        if (browserCounter > 0) { // No need to protect if queue is empty
            if (maxMessagesInQueue == 0 && maxPrSender == 0 && maxAgeSeconds == 0) {
                protect = true;
            } else {
                // ALL criterias must be fulfilled, unless there will be no protection
                protect = browserCounter >= maxMessagesInQueue && senderWithHighestCount >= maxPrSender && oldestMsgSec >= maxAgeSeconds;
            }
        }

        StringBuilder sb = new StringBuilder();
        sb.append(LogOut.i());
        sb.append("Monitoring ").append(queueName).append(" based on criterias: (REQ/FOUND): ");
        sb.append("messages ").append(metrics(maxMessagesInQueue, browserCounter, 4, "#")).append(", ");
        sb.append("senders ").append(metrics(maxPrSender, senderWithHighestCount, 4, "#")).append(", ");
        sb.append("age ").append(metrics(maxAgeSeconds, oldestMsgSec, 5, "s"));
        sb.append(" -> Protection: ").append(protect);
        System.out.println(sb);
        return protect;
    }

    private static String metrics(int requirement, int measurement, int size, String unit) {
        return String.format("(%" + size + "s/%-" + size + "s)", requirement + unit, measurement + unit);
    }

    private static void logMonitorStateToJson(LoopState loopState, Counters counters, int browserCounter) {
        boolean forceLog = false;
        if (loopState.isContinuous()) {
            // Two reasons to force log: 1) Exited from a non-trivial mode 2) New hour
            if (loopState.getLastMode() != Mode.MONITOR) {
                forceLog = true;
            }
            long hourSinceEpochStart = System.currentTimeMillis() / 3600000L;
            if (loopState.getLastLoggedTms() == -1 || loopState.getLastLoggedTms() != hourSinceEpochStart) {
                loopState.setLastLoggedTms(hourSinceEpochStart);
                forceLog = true;
            }
        }
        if (!loopState.isContinuous() || forceLog) {
            logStateToJson(counters, "MONITOR", browserCounter, 0, 0, 0, 0, 0, 0);
        }
    }

    private static void logProtectStateToJson(Counters counters, int browserCounter, int consumeCounter, long msPrConsume) {
        logStateToJson(counters, "PROTECT", browserCounter, consumeCounter, msPrConsume, 0, 0, 0, 0);
    }

    private static void logRestoreStateToJson(Counters counters, int browserCounter, int msgToRestore, int restoreCounter, long msPrRestore, int waitMs) {
        logStateToJson(counters, "RESTORE", browserCounter, 0, 0, msgToRestore, restoreCounter, msPrRestore, waitMs);
    }

    private static void logRestorePostponedStateToJson(Counters counters, int browserCounter, int msgToRestore, int waitMs) {
        logStateToJson(counters, "RSTWAIT", browserCounter, 0, 0, msgToRestore, 0, 0, waitMs);
    }


    private static void logStateToJson(Counters counters, String status, int browserCounter, int consumeCounter, long msPrConsume, int msgToRestore, int restoreCounter, long msPrRestore, int waitMs) {
        if (summaryFile == null)
            return;
        try {
            // Based on the counters found above, we retrieve som key numbers
            int senderWithHighestCount = counters.getNoMsgPrSender().values().stream().max(Integer::compareTo).orElse(0);
            long oldestMsgSec = counters.getNoMsgPrAgeSec().keySet().stream().max(Long::compareTo).orElse(0L);
            int noMsgTooOld = counters.getNoMsgPrAgeSec().entrySet().stream().filter(e -> e.getKey() > maxAgeSeconds).mapToInt(Map.Entry::getValue).sum();

            StringBuilder sb = new StringBuilder();
            sb.append("{ \"statustime\": \"").append(LocalDateTime.now().format(Const.localTmsSec)).append("\", ");
            sb.append("\"status\": ").append(String.format("\"%-7s\", ", status));
            sb.append("\"queueSize\": ").append(format(browserCounter, 3));
            sb.append("\"senderHighestCnt\": ").append(format(senderWithHighestCount, 3));
            sb.append("\"tooOldCnt\": ").append(format(noMsgTooOld, 3));
            sb.append("\"oldestMsgSec\": ").append(format((int) oldestMsgSec, 6));
            sb.append("\"consumeCnt\": ").append(format(consumeCounter, 4));
            sb.append("\"msPrConsume\": ").append(format((int) msPrConsume, 4));
            sb.append("\"msgToRestore\": ").append(format(msgToRestore, 5));
            sb.append("\"restoreCnt\": ").append(format(restoreCounter, 4));
            sb.append("\"msPrRestore\": ").append(format((int) msPrRestore, 3));
            sb.append("\"waitMs\": ").append(format(waitMs, 4));
            Set<String> senders = counters.getSenders(maxPrSender);
            if (senders.isEmpty()) {
                sb.append("\"sendersAboveLimit\": []");
            } else {
                String sendersCommaSeparated = senders.stream().map(s -> "\"" + s + "\"").collect(Collectors.joining(","));
                sb.append("\"senders\": [").append(sendersCommaSeparated).append("]");
            }
            sb.append("}\n");
            Files.writeString(summaryFile.toPath(), sb.toString(), StandardOpenOption.APPEND);
        } catch (Exception e) {
            System.out.println(LogOut.e() + "Cannot write to file " + summaryFile.getPath() + " (exception: " + e.getMessage() + ")" + ", Stacktrace: " + Div.onelinerStacktrace(e.getStackTrace(), "no.statnett.ecp"));
        }
    }

    private static String format(int browserCounter, int size) {
        return String.format("%" + size + "d, ", browserCounter);
    }

    private static String formatNoComma(int browserCounter, int size) {
        return String.format("%" + size + "d", browserCounter);
    }


    private static void usage() {
        System.out.println("QueueProtector (QP) " + VERSION + " is made to protect an ECP/EDX queue from getting too many messages.");
        System.out.println("There are three scenarios where QP is useful:");
        System.out.println(" 1) Trashcan: The queue has no consumers, and QP can consume messages so queue never runs full");
        System.out.println(" 2) Outdated removal: Some consumers are not working, but QP can consume old-no-longer-useful messages");
        System.out.println(" 3) Burst protection: QP can consume 'excess' messages, and *may* restore them later");
        System.out.println();
        System.out.println("The natural flow of messages in ECP/EDX ends in certain queues. Below follows a list of queues and their");
        System.out.println("purpose - listed in 'flow order'. (N) denotes the most likely scenario where QP is useful:");
        System.out.println();
        System.out.println("RECEIVING MESSAGES (INCOMING FROM THE NETWORK):");
        System.out.println(" - ecp.endpoint.download (3)                    (fills up if ECP is receiving too many from central broker)");
        System.out.println(" - ecp.endpoint.inbox (2)                       (fills up if EDX does not consume messages from ECP");
        System.out.println(" - ecp.endpoint.send.event (2)                  (fills up if EDX does not consume events/Acks from ECP");
        System.out.println(" - edx.endpoint.inbox (ex of an inbox) (2)      (inboxes may fill up if BA is not consuming)");
        System.out.println("SENDING MESSAGES (OUTGOING TO THE NETWORK):");
        System.out.println(" - edx.internal.sendProcess-delivery-send (3)  (fills up if EDX is receiving too many from BA system)");
        System.out.println(" - ecp.endpoint.upload.<brokerECPCode> (2)     (fills up if connection to broker is down");
        System.out.println("TRASH/OLD MESSAGES:");
        System.out.println(" - ActiveMQ.DLQ (1)                            (fills up if in both ECP and EDX if messages are never consumed)");
        System.out.println();
        System.out.println("IMPORTANT: ActiveMQ defaults to prefetch 1000 messages, this is not ideal and must be changed for burst");
        System.out.println("protection to work properly! Reduce prefetch limit by adding these properties (and restart endpoint):");
        System.out.println(" - ECP: In ecp.properties add 'internalBroker.parameters=jms.prefetchPolicy.queuePrefetch=10'");
        System.out.println(" - EDX: In edx.properties add 'edx.amqp.client.prefetch=10'");
        System.out.println();
        System.out.println("The protection will happen if these conditions are fulfilled:");
        System.out.println(" - The queue contains more than -q messages. Default is 100. Allowed range is 0-400");
        System.out.println(" - One sender has more than -s messages in the queue. Default is 0. Allowed range is 0-300");
        System.out.println(" - Some messages are older than -t seconds. Default is 30. Allowed range is 1-86400 (up till 24h)");
        System.out.println();
        System.out.println("The protection will kick inn once all these criteria have been met, and will consume message older than -t seconds.");
        System.out.println("If -s is above 0, only messages from the sender matching the criteria will be consumed. In addition are messages");
        System.out.println("with internalType = DELIVERY_ACKNOWLEDGEMENT consumed, because these ACKs are not essential.");
        System.out.println();
        System.out.println();
        System.out.println("Usage  : java -jar ekit.jar QP [OPTIONS] QUEUE BROKER\n");
        System.out.println("\n\nThe options and arguments:");
        System.out.println(" OPTIONS       : ");
        System.out.println("                 -e                 : restore messages from directory to queue. Requires -m option");
        System.out.println("                 -f                 : continuous operation - will not exit, but repeat scan/consume every 10 seconds.");
        System.out.println("                 -m <directory>     : messages will be stored here.");
        System.out.println("                 -o <summary-file>  : append a one-liner json-summary to <summary-file> for monitoring by Splunk/etc.");
        System.out.println("                 -q<number>         : consume if queue contains more than <number> messages. Default 200, allowed 0-400.");
        System.out.println("                 -s<number>         : consume if sender has more than <number> messages in the queue. Default 0, allowed 0-300");
        System.out.println("                 -t<number>         : consume if message is older than <number> seconds. Default 30, allowed 1-86400 (24h)");
        System.out.println(" QUEUE         : The queue under protection. See list above for suggestions");
        System.out.println(" BROKER        : <URL|PATH> - either specify a URL to a broker or a path for the ECP- or EDX-configuration");
        System.out.println("                 URL                : Specify url on this form: amqp[s]://user:pass@host:port");
        System.out.println("                 PATH               : Specify either the conf-path for ECP or EDX (ex: /etc/ecp-endpoint or /etc/edx-toolbox)");

        System.out.println("\nExample 1:  One-time protection of ecp.endpoint.download, default limits - no restore, no summary");
        System.out.println("\tjava -jar ekit.jar QP ecp.endpoint.download amqp://endpoint:password@localhost:5672");
        System.out.println("\nExample 2:  Same as Example 1, but now continuous mode + store msg to tmp-folder + summary to qp.json");
        System.out.println("\tjava -jar ekit.jar QP -f -m tmp -o qp.json ecp.endpoint.download amqp://endpoint:password@localhost:5672");
        System.out.println("\nExample 3:  Same as Example 2, but restore is active. Protection limits are quite low (queue-limit:20, age-limit:10 sec)");
        System.out.println("\tjava -jar ekit.jar QP -e -f -m tmp -o qp.json -q20 -t10 ecp.endpoint.download amqp://endpoint:password@localhost:5672");
        System.out.println("\nExample 4:  Same as Example 3, but protects only against senders with more than 15 msg - useful scenario");
        System.out.println("\tjava -jar ekit.jar QP -e -f -m tmp -o qp.json -q20 -s15 -t10 ecp.endpoint.download amqp://endpoint:password@localhost:5672");

        System.out.println("\nTo log with UTC time, start command with 'java -Duser.timezone=GMT -jar ekit.jar QP....etc'");
    }
}