package no.statnett.ecp.utils;

import jakarta.jms.*;
import org.apache.qpid.jms.JmsConnectionFactory;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Broker {

    /**
     * This method will create a connection to the AMQP-broker and start it. It requires a config-map with the following keys:
     * <p>
     * url: The URL to the broker (ex: amqps://localhost:5672) - the URL is without user/pass because that's how the library wants it
     * user: The user to connect with the AMQP-broker, I think it's safe to set it even if the broker does not require it
     * password: The password to connect with the AMQP-broker, I think it's safe to set it even if the broker does not require it
     * host: The host part of the URL - yes, it founds in the URL also, but we can skip some parsing by having it here
     * port: The port part of the URL - yes, it founds in the URL also, but we can skip some parsing by having it here
     * <p>
     * The method will try to connect to the broker with TLS if that is what the url suggests (starting with amqps) otherwise
     * it will try without TLS. However, if the user has specified the URL directly, the method will try the opposite suggestion
     * also, if the first suggestion fails. For the TLS-case, the method trust ALL certificates from the server, therefore requires no truststore,
     * nor does it require a verification of the host name. This makes it easy to use. However, if a client needs to know as
     * certain which server it is connected to, then this method is not advisable to use. However, users of this method are not
     * expected to be in such a situation, the risk is entirely on the server side to allow connection.
     *
     * @param config
     * @return
     */
    public static Connection createAndStartConnection(Map<String, String> config) {
        String suggest = config.get("suggest");
        String failed = config.get("failed");
        if (suggest == null) { // User is not suggestion any URL, but referring to ecp/edx.properties - so no reason to try anything else than the url found in the properties
            if (config.get("url").startsWith("amqps")) {
                return createConnectionImpl(config, true, true);
            } else {
                return createConnectionImpl(config, false, true);
            }
        } else { // User has suggested a URL, but it can be wrong, so we must try the opposite suggestion if the first fails
            boolean tls = suggest.equals("tls"); // can be either "tls" or "no-tls" (the user have specified amqp:// or amqps://)
            if (failed == null) { // First time run, or the last time it ran it was not necessary to switch to opposite suggestion
                Connection c = createConnectionImpl(config, tls, false);
                if (c == null) {
                    config.put("failed", "true");
                    return createConnectionImpl(config, !tls, true);
                } else {
                    return c;
                }
            } else { // This can only happen if this method has been run before and returned successfully - since failed is set if the first attempt failed
                return createConnectionImpl(config, true, true);
            }
        }
    }

    private static Connection createConnectionImpl(Map<String, String> config, boolean tls, boolean exitOnFail) {
        String url = config.get("url");
        String user = config.get("user");
        String password = config.get("password");
        boolean extraOptions = config.get("options") != null;
        String options = "";
        if (extraOptions || tls) {
            options += "?";
        }
        if (extraOptions) {
            options += config.get("options");
        }
        if (tls) {
            options += (extraOptions ? "&" : "") + "transport.verifyHost=false";
        }

        String modifiedUrl = url + options;
        modifiedUrl = tls ? modifiedUrl.replaceAll("amqp:", "amqps:") : modifiedUrl.replaceAll("amqps:", "amqp:");

        JmsConnectionFactory factory = new JmsConnectionFactory(modifiedUrl);
        factory.setUsername(user);
        factory.setPassword(password);

        try {
            if (tls) {
                // Section which trust all certificates from the server
                SSLContext sslContext = SSLContext.getInstance("TLS");
                TrustManager[] trustManagers = new TrustManager[]{new X509TrustManager() {
                    public void checkClientTrusted(X509Certificate[] chain, String authType) {
                    }

                    public void checkServerTrusted(X509Certificate[] chain, String authType) {
                    }

                    public X509Certificate[] getAcceptedIssuers() {
                        return new X509Certificate[]{};
                    }
                }};
                sslContext.init(null, trustManagers, null);
                factory.setSslContext(sslContext);
                //  We could, if we wanted, require trustStore and specify it, to trust certain certificates only. For most
                //  users connecting remotely to an AMQP-server it is just a hassle to retrieve/copy the server's keystores.
                //        System.setProperty("transport.trustStoreLocation", config.get("keystore-file"));
                //        System.setProperty("transport.trustStorePassword", config.get("keystore-pw"));
            }
            Connection connection = factory.createConnection();
            connection.start();
            return connection;
        } catch (JMSException | NoSuchAlgorithmException | KeyManagementException e) {
            if (exitOnFail) {
                System.out.println("Connection to AMQP-server on " + config.get("host") + ":" + config.get("port") + " failed (" + e.getMessage() + ") - exiting.");
                System.exit(1);
                return null; // Only to satisfy the compiler - this line will never be reached
            } else {
                return null;
            }
        }
    }

    public static String buildSelector(String messageType, String TTL, String senderCode, String receiverCode, String internalType) {
        String selector = null;
        if (messageType != null) {
            selector = "(messageType = '" + messageType + "' OR businessType = '" + messageType + "' OR EDX_MESSAGE_TYPE = '" + messageType + "')";
        }
        if (TTL != null && Long.parseLong(TTL) > 0) {
            long tms = System.currentTimeMillis() - (Long.parseLong(TTL) * 1000L);
            String newSelector = "(JMSTimestamp < " + tms + " OR generated < " + tms + ")";
//            String newSelector = "convert_string_expressions:(JMSTimestamp < " + tms + " OR generated < " + tms + ")";
//            String newSelector = "convert_string_expressions:generated < " + tms + "";
            selector = appendSelector(selector, newSelector);
        }
        if (senderCode != null && !senderCode.equals("ALL")) {
            String newSelector = "(senderCode = '" + senderCode + "' OR sender = '" + senderCode + "' OR EDX_SENDER = '" + senderCode + "')";
            selector = appendSelector(selector, newSelector);
        }
        if (receiverCode != null) {
            String newSelector = "(receiverCode = '" + receiverCode + "' OR receiver = '" + receiverCode + "' OR EDX_RECEIVER_CODE = '" + receiverCode + "')";
            selector = appendSelector(selector, newSelector);
        }
        if (internalType != null) {
            String newSelector = "(internalType = '" + internalType + "' OR EDX_MESSAGE_KIND = '" + internalType + "')";
            selector = appendSelector(selector, newSelector);
        }

        return selector;
    }

    private static String appendSelector(String selector, String TTLSelect) {
        selector = (selector == null ? TTLSelect : selector + " AND " + TTLSelect);
        return selector;
    }

}
