package no.statnett.ecp.bck;

import no.statnett.ecp.utils.EcpHTTP;
import no.statnett.ecp.utils.LogOut;
import no.statnett.ecp.utils.Options;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class Backup {
  public static final String VERSION = "v1.1.0";
  private static String user;
  private static String pw;
  private static URL url;

  private static String exportFilename;
  private static String importFilename;
  private static File importFile;
  private static boolean edx = false;

  public Backup(String user, String pw, URL url) {
    Backup.user = user;
    Backup.pw = pw;
    Backup.url = url;
  }

  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));

    edx = Options.parseBoolean(initArgsList, "-x");

    exportFilename = Options.parseString(initArgsList, "-e");
    importFilename = Options.parseString(initArgsList, "-i");
    if (importFilename != null) {
      importFile = Options.checkFile(importFilename, false);
    }

    if (exportFilename == null && importFilename == null) {
      exportFilename = edx ? "edx-export.zip" : "ecp-export.zip";
    } else if (exportFilename != null && importFilename != null) {
      importFilename = null;
      importFile = null;
    }

    return initArgsList;
  }

  private static void parseMandatoryArg(List<String> mandatoryArgs) throws MalformedURLException {
    if (mandatoryArgs.isEmpty()) {
      usage();
      System.exit(1);
    }
    URL initialURL = new URL(mandatoryArgs.get(0));
    String path = edx ? "api/rest" : "ECP_MODULE";
    String endpoint = exportFilename == null ? "restore" : "backup";
    url = new URL(initialURL.getProtocol(), initialURL.getHost(), initialURL.getPort(), "/" + path + "/settings/database/" + endpoint);
    if (initialURL.getUserInfo() != null) {
      user = initialURL.getUserInfo().split(":")[0];
      pw = initialURL.getUserInfo().split(":")[1];
    } else {
      System.out.println(LogOut.e() + "The URL must contain user:password");
      System.exit(1);
    }
  }

  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    List<String> mandatoryArgs = parseOptions(args);
    parseMandatoryArg(mandatoryArgs);
    Backup bck = new Backup(user, pw, url);
    if (exportFilename == null) {
      bck.importDB();
    } else {
      bck.exportDB();
    }
  }

  private void exportDB() throws NoSuchAlgorithmException, IOException, KeyManagementException {
    byte[] export = EcpHTTP.getECPHTTResponseAsBytes("POST", url, user, pw, null, 0);
    try (FileOutputStream fos = new FileOutputStream(exportFilename)) {
      fos.write(export);
    }
    String system = edx ? "EDX" : "ECP";
    System.out.println(LogOut.i() + "Export from " + system + " written to " + exportFilename + " with success");
  }

  private void importDB() throws IOException, NoSuchAlgorithmException, KeyManagementException {
    byte[] bytes = Files.readAllBytes(importFile.toPath());
    EcpHTTP.put(url, user, pw, bytes);
    String system = edx ? "EDX" : "ECP";
    System.out.println(LogOut.i() + "Import of " + importFilename + " to " + system + " completed succcessfully");
  }

  private static void usage() {
    System.out.println("Backup (BCK) " + VERSION + " can make export & import a backup from ECP/EDX during runtime.");
    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar BCK <OPTIONS> <URL>  \n");
    System.out.println("\n\nThe arguments:");
    System.out.println(" OPTIONS      ");
    System.out.println("     -e <file>     : Export - default option, default filename is edx-export.zip or ecp-export.zip");
    System.out.println("     -i <file>     : Import, -e will override -i.");
    System.out.println("     -x            : Use this parameter to indicate EDX, otherwise ECP is assumed");
    System.out.println("MANDATORY ARGUMENTS");
    System.out.println("     <URL>         : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("\nExample 1:  Export from ECP");
    System.out.println("\tjava -jar ekit.jar BCK https://admin:password@remote-host:8443");
    System.out.println("\nExample 2:  Import into ECP");
    System.out.println("\tjava -jar ekit.jar BCK -i ecp-export.zip https://admin:password@remote-host:8443");
    System.out.println("\nExample 3:  Export from EDX");
    System.out.println("\tjava -jar ekit.jar BCK -x https://admin:password@remote-host:9443");
    System.out.println("\nExample 4:  Import into EDX");
    System.out.println("\tjava -jar ekit.jar BCK -x -i edx-export.zip https://admin:password@remote-host:9443");
  }
}
