package no.statnett.ecp.brs.actions;

import no.statnett.ecp.brs.Config;
import no.statnett.ecp.brs.Div;
import no.statnett.ecp.brs.state.MessageFilter;
import no.statnett.ecp.brs.state.MessageInfo;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.SimpleParser;
import org.json.JSONArray;
import org.json.JSONObject;

import java.time.LocalDateTime;
import java.util.*;

public class ListMessages {

  public static class Print {
    public static void printMI(List<MessageInfo> messages, boolean echo) {
      int i = 1;


      if (echo) {
        String hostHd = "Host (1)";
        String queueHd = "Queue (2)";
        String ageHd = "Age(m) (3)";
        String receiverHd = "Receiver (4)";
        String senderHd = "Sender (5)";
        String senderAppHd = "SenderApp (6)";
        String msgTypeHd = "MessageType (7)";
        String baMsgIdHd = "BaMessageID (8)";
        String intTypeHd = "InternalType (9)";

        int hostWidth = Math.max(messages.stream().mapToInt(m -> m.getHost().length()).max().orElse(0), hostHd.length());
        int queueWidth = Math.max(messages.stream().mapToInt(m -> m.getQueueName().length()).max().orElse(0), queueHd.length());
        int ageWidth = Math.max(messages.stream().mapToInt(m -> ("" + m.getAge()).length()).max().orElse(0), ageHd.length());
        int receiverWidth = Math.max(messages.stream().mapToInt(m -> ("" + m.getReceiverCode()).length()).max().orElse(0), receiverHd.length());
        int senderWidth = Math.max(messages.stream().mapToInt(m -> ("" + m.getSenderCode()).length()).max().orElse(0), senderHd.length());
        int senderAppWidth = Math.max(messages.stream().mapToInt(m -> ("" + m.getSenderApp()).length()).max().orElse(0), senderAppHd.length());
        int msgTypeWidth = Math.max(messages.stream().mapToInt(m -> m.getMessageType().length()).max().orElse(0), msgTypeHd.length());
        int baMsgIdWidth = Math.max(messages.stream().mapToInt(m -> ("" + m.getBaMessageID()).length()).max().orElse(0), baMsgIdHd.length());
        int intTypeWidth = Math.max(messages.stream().mapToInt(m -> ("" + m.getInternalType()).length()).max().orElse(0), intTypeHd.length());
        int totalWidth = hostWidth + queueWidth + ageWidth + receiverWidth + senderWidth + +senderAppWidth + msgTypeWidth + baMsgIdWidth + intTypeWidth + 4 + 9 * 3;


        System.out.println(String.format("%-4s | %-" + hostWidth + "s | %-" + queueWidth + "s | %" + ageWidth + "s | %-" + receiverWidth + "s | %-" + senderWidth + "s | %-" + senderAppWidth + "s | %-" + msgTypeWidth + "s | %-" + baMsgIdWidth + "s | %" + intTypeWidth + "s", "#", hostHd, queueHd, ageHd, receiverHd, senderHd, senderAppHd, msgTypeHd, baMsgIdHd, intTypeHd) + " (" + LocalDateTime.now().format(Const.localTmsSec) + ")");

        System.out.println("-".repeat(totalWidth));
        for (MessageInfo mi : messages) {
          String sb = String.format("%-4d |", i) +
              String.format(" %-" + hostWidth + "s |", mi.getHost()) +
              String.format(" %-" + queueWidth + "s |", mi.getQueueName()) +
              String.format(" %" + ageWidth + "s |", mi.getAge()) +
              String.format(" %-" + receiverWidth + "s |", mi.getReceiverCode()) +
              String.format(" %-" + senderWidth + "s |", mi.getSenderCode()) +
              String.format(" %-" + senderAppWidth + "s |", mi.getSenderApp()) +
              String.format(" %-" + msgTypeWidth + "s |", mi.getMessageType()) +
              String.format(" %-" + baMsgIdWidth + "s |", mi.getBaMessageID()) +
              String.format(" %-" + intTypeWidth + "s", mi.getInternalType());
          System.out.println(sb);
          i++;
        }
      }
    }
  }

  public static class Retrieve {
    public static List<MessageInfo> retrieveAndFilter(List<QueueInfo> qiList, Config config, MessageFilter messageFilter) {
      List<MessageInfo> MIList = new ArrayList<>();
      Map<String, String> broker2Encoded = new HashMap<>();
      for (QueueInfo qi : qiList) {
        try {
          if (broker2Encoded.get(qi.getHost()) == null) { // The same broker on the same host, no need to look up again
            String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(Div.prot(config, qi), qi.getHost(), config.get("port"), Div.user(config, qi), Div.pass(config, qi));
            broker2Encoded.put(qi.getHost(), brokerNameEncoded);
          }
          ArtemisConsoleAPI.ACAPIResult result = ArtemisConsoleAPI.listMessages(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi), qi.getQueueName(), messageFilter.getFilterStr());
//          ArtemisConsoleAPI.ACAPIResult result2 = ArtemisConsoleAPI.listDeliveringMessages(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi), qi.getQueueName());
          if (result.getStatus().equals("200")) {
            MIList.addAll(parseMessages(qi, result.getJsonObject().getJSONArray("value")));
//            MIList.addAll(parseDeliveringMessages(qi, result2.getJsonObject().getJSONObject("value")));
          } else {
            System.out.println("Failed to run listMessages-API for queue: " + qi.getQueueName() + " on host: " + qi.getHost() + " - " + result.getRawResponse());
          }
        } catch (Exception e) {
          System.out.println("Error occurred: During listMessages-API for queue: " + qi.getQueueName() + " on host " + qi.getHost() + ": " + e + " " + e.getMessage());
        }
      }
      // Sort and limit the MIList according to messageFilter-sort/limits
      List<MessageInfo> sorted = sortMessageInfo(MIList, messageFilter.getMsrt().number());
      List<MessageInfo> limitedMIList = messageFilter.getMlim().number() > 0 ? sorted.subList(0, Math.min(messageFilter.getMlim().number(), sorted.size())) : sorted;
      return limitedMIList;
    }

    private static List<MessageInfo> parseMessages(QueueInfo qi, JSONArray jsonArray) {
      List<MessageInfo> mInfoList = new ArrayList<>();

      for (int i = 0; i < jsonArray.length(); i++) {
        JSONObject jsonObject = jsonArray.getJSONObject(i);
        MessageInfo mInfo = parseJsonObjectIntoMessageInfo(qi, jsonObject);
        mInfoList.add(mInfo);
      }
      return mInfoList;
    }

    private static List<MessageInfo> parseDeliveringMessages(QueueInfo qi, JSONObject valueObject) {
      List<MessageInfo> mInfoList = new ArrayList<>();
      Iterator<String> keys = valueObject.keys();
      while (keys.hasNext()) {
        String consumerKey = keys.next();

        JSONArray messages = valueObject.getJSONArray(consumerKey);
        for (int i = 0; i < messages.length(); i++) {
          JSONObject jsonObject = messages.getJSONObject(i);
          MessageInfo mInfo = parseJsonObjectIntoMessageInfo(qi, jsonObject);
          mInfoList.add(mInfo);
        }
      }
      return mInfoList;
    }

    private static MessageInfo parseJsonObjectIntoMessageInfo(QueueInfo qi, JSONObject jsonObject) {
      // Example of raw JSON: {"x-opt-jms-dest":0,"creationTime":1721641647472,"x-opt-jms-msg-type":4,"subject":"HEARTBEAT","messageID":161930336,"CamelJmsDeliveryMode":2,"receiverCode":"50V-EIDSIVAN-ATI","messageMversion":2,"userID":"ID:ID:2f070daf-ecd0-4177-b4f2-2e0f1c1dad3c:4:56126:1-1","durable":true,"messageType":"HEARTBEAT","generated":1721641647221,"x-opt-ORIG-QUEUE":"ecp.endpoint.receive.50V-EIDSIVAN-ATI","baMessageID":"","_AMQ_ACTUAL_EXPIRY":1721641707729,"_AMQ_ORIG_ADDRESS":"ecp.endpoint.receive.50V-EIDSIVAN-ATI","timestamp":1721641647472,"_AMQ_AD":"ExpiryQueue","x-opt-ACTUAL-EXPIRY":1721641707729,"address":"ExpiryQueue","_AMQ_ORIG_ROUTING_TYPE":1,"internalType":"TRACING_MESSAGE","senderCode":"45V000000000059N","x-opt-ORIG-ROUTING-TYPE":1,"priority":4,"x-opt-ORIG-ADDRESS":"ecp.endpoint.receive.50V-EIDSIVAN-ATI","_AMQ_ORIG_MESSAGE_ID":161920626,"x-opt-ORIG-MESSAGE-ID":161920626,"_AMQ_ORIG_QUEUE":"ecp.endpoint.receive.50V-EIDSIVAN-ATI","expiration":0,"senderApplication":"","to":"ExpiryQueue"}
      // The values we're interested in, and how they relate to QueueInfo: baMessageID, generated, internalType, messageType, receiverCode, senderApplication, senderCode
      MessageInfo mInfo = new MessageInfo();
      mInfo.setHost(qi.getTrimmedHost());
      mInfo.setQueueName(qi.getQueueName());

      mInfo.setBaMessageID(getStringFromJson(jsonObject, "baMessageID", null));

      if (jsonObject.has("generated")) {
        mInfo.setGenerated(jsonObject.getLong("generated"));
      } else if (jsonObject.has("timestamp")) {
        mInfo.setGenerated(jsonObject.getLong("timestamp"));
      }


      mInfo.setInternalType(getStringFromJson(jsonObject, "internalType", "EDX_MESSAGE_KIND"));
      mInfo.setMessageType(getStringFromJson(jsonObject, "messageType", "businessType"));
      mInfo.setReceiverCode(getStringFromJson(jsonObject, "receiverCode", "EDX_RECEIVER"));
      if (mInfo.getReceiverCode() == null && jsonObject.has("service") && !jsonObject.getString("service").isEmpty()) {
        mInfo.setReceiverCode("THIS@SERVICE-" + jsonObject.getString("service"));
      }
      mInfo.setSenderCode(getStringFromJson(jsonObject, "senderCode", "sender", "EDX_SENDER"));
      mInfo.setSenderApp(getStringFromJson(jsonObject, "senderApplication", "EDX_SENDER_BA"));
      if (jsonObject.has("messageID"))
        mInfo.setMessageID(jsonObject.getLong("messageID"));
      return mInfo;
    }

    public static String getStringFromJson(JSONObject jsonObject, String... keys) {
      for (String key : keys) {
        if (key != null && jsonObject.has(key) && !jsonObject.getString(key).isEmpty()) {
          return jsonObject.getString(key);
        }
      }
      return null;
    }

    private static List<MessageInfo> sortMessageInfo(List<MessageInfo> MIL, int sort) {
      if (sort > 0) {
        switch (sort) {
          case 1:
            MIL.sort(Comparator.comparing(MessageInfo::getHost));
            break;
          case 2:
            MIL.sort(Comparator.comparing(MessageInfo::getQueueName));
            break;
          case 3:
            MIL.sort(Comparator.comparingInt(MessageInfo::getAge));
            break;
          case 4:
            MIL.sort(Comparator.comparing(MessageInfo::getReceiverCode));
            break;
          case 5:
            MIL.sort(Comparator.comparing(MessageInfo::getSenderCode));
            break;
          case 6:
            MIL.sort(Comparator.comparing(MessageInfo::getSenderApp));
            break;
          case 7:
            MIL.sort(Comparator.comparing(MessageInfo::getMessageType));
            break;
          case 8:
            MIL.sort(Comparator.comparing(MessageInfo::getBaMessageID));
            break;
          case 9:
            MIL.sort(Comparator.comparing(MessageInfo::getInternalType));
            break;
        }
      } else if (sort < 0) {
        switch (sort) {
          case -1:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getHost().compareTo(o1.getHost()));
            break;
          case -2:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getQueueName().compareTo(o1.getQueueName()));
            break;
          case -3:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> Integer.compare(o2.getAge(), o1.getAge()));
            break;
          case -4:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getReceiverCode().compareTo(o1.getReceiverCode()));
            break;
          case -5:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getSenderCode().compareTo(o1.getSenderCode()));
            break;
          case -6:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getSenderApp().compareTo(o1.getSenderApp()));
            break;
          case -7:
          MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getMessageType().compareTo(o1.getMessageType()));
            break;
          case -8:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getBaMessageID().compareTo(o1.getBaMessageID()));
            break;
          case -9:
            MIL.sort((MessageInfo o1, MessageInfo o2) -> o2.getInternalType().compareTo(o1.getInternalType()));
            break;
        }
      }
      return MIL;
    }

  }
}
