package no.statnett.ecp.brs.actions;

import no.statnett.ecp.brs.Config;
import no.statnett.ecp.brs.Div;
import no.statnett.ecp.brs.state.MessageFilter;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.utils.ArtemisConsoleAPI;
import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.SimpleParser;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class RemoveMessages {
    public static void remove(List<QueueInfo> qiList, Config config, String input, boolean echo, MessageFilter messageFilter) {
        Integer tmp = SimpleParser.getInt(input);
        if (tmp == null)
            return;
        if (tmp < 0) {
            int dangerousCount = dangerousPurge(qiList);
            if (tmp == -1 && dangerousCount > 0) {
                System.out.println("This remove is aborted, you're attempting to purge at least " + dangerousCount + " messages which might not be intended. If you want to persist, run 'r-2' to confirm");
                return;
            }
            if (echo)
                System.out.println("Removing all messages that match the message filter from the queue list (v), since number is < 0");
            removeImpl(qiList, config, echo, messageFilter);
        } else if (tmp > 0 && tmp <= qiList.size()) {
            if (echo)
                System.out.println("Remove message that match the message filter from queue number " + tmp);
            List<QueueInfo> selectedQIList = new ArrayList<>();
            selectedQIList.add(qiList.get(tmp - 1));
            removeImpl(selectedQIList, config, echo, messageFilter);
        } else {
            System.out.println("Invalid queue number: " + tmp + ", command ignored");
        }
    }

    private static void removeImpl(List<QueueInfo> limitedQIList, Config config, boolean echo, MessageFilter messageFilter) {
        int messagesTotal = 0;
        int messagesRemoved = 0;
        int executionsOk = 0;
        int failed = 0;
        if (echo)
            System.out.print("Starting to remove.");
        String echoOffOneLinerSummary = "";
        Map<String, String> broker2Encoded = new HashMap<>();
        for (QueueInfo qi : limitedQIList) {
            messagesTotal += qi.getSize();
            try {
                if (broker2Encoded.get(qi.getHost()) == null) { // The same broker on the same host, no need to look up again
                    String brokerNameEncoded = ArtemisConsoleAPI.retrieveBrokerNameURLEncoded(Div.prot(config, qi), qi.getHost(), config.get("port"), Div.user(config, qi), Div.pass(config, qi));
                    broker2Encoded.put(qi.getHost(), brokerNameEncoded);
                }
                ArtemisConsoleAPI.ACAPIResult result = ArtemisConsoleAPI.removeMessages(Div.prot(config, qi), qi.getHost(), config.get("port"), broker2Encoded.get(qi.getHost()), Div.user(config, qi), Div.pass(config, qi), qi.getQueueName(), messageFilter.getFilterStr());
                if (result.getStatus() != null && result.getStatus().equals("200")) {
                    executionsOk++;
                    int removedOnThisQ = Integer.parseInt(result.getValue());
                    messagesRemoved += removedOnThisQ;
                    echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getQueueName() + ":" + qi.getSize() + ":" + removedOnThisQ + "] ";
                } else {
                    if (echo)
                      System.out.println("Failed to remove messages from queue: " + qi.getQueueName() + " on host: " + qi.getHost() + " - " + result.getRawResponse());
                    failed++;
                    echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getSize() + ":" + qi.getQueueName() + ":failed] ";
                }
                if (echo)
                    System.out.print(".");
            } catch (Exception e) {
                if (echo)
                  System.out.println("Error occurred: During removeMessages of queue: " + qi.getQueueName() + " on host " + qi.getHost() + ": " + e.getMessage());
                failed++;
                echoOffOneLinerSummary += "[" + qi.getHost() + ":" + qi.getSize() + ":" + qi.getQueueName() + ":failed] ";
            }
        }
        String failedStr = failed > 0 ? "Failed to remove messages from " + failed + " queues." : "";
        if (echo) {
            System.out.println("\nRemoval completed: Removed " + messagesRemoved + " out of " + messagesTotal + " messages from " + executionsOk + " out of " + limitedQIList.size() + " queues listed. " + failedStr);
        } else {
            String timestamp = LocalDateTime.now().format(Const.localTmsSec);
            if (messagesRemoved > 0)
              System.out.println(timestamp + " INFO  Removed " + messagesRemoved + " messages from " + limitedQIList.size() + " queues. " + echoOffOneLinerSummary);
            else
              System.out.println(timestamp + " DEBUG Nothing was removed. " + echoOffOneLinerSummary);
            if (failed > 0)
              System.out.println(timestamp + " WARN  Remove failed on " + failed + " queues");
        }
    }

    private static int dangerousPurge(List<QueueInfo> qiList) {
        int dangerousCount = 0;
        for (QueueInfo qi : qiList) {
            if (qi.getSize() > 0 && qi.getConsumers() > 0)
                dangerousCount += qi.getSize();
            else if (qi.getSize() > 0 && !qi.getQueueName().contains("DLQ") && !qi.getQueueName().contains("Expiry"))
                dangerousCount += qi.getSize();
        }
        return dangerousCount;
    }
}
