package no.statnett.ecp.brs.state;


public record FilterResult(String op, Integer number, String arg) {

  // Sort commands are in the form "[q|m|c]srt\s*<number>". The absolute value of the number cannot exceed max-arg. If number
  // is set to 0, then the sort is reset. Negative numbers are allowed. If the number is omitted, the sort is reset.
  public static FilterResult parseSortCmd(String input, int max, boolean echo) {
    String resetMessage = "Sort is turned off";
    if (input.trim().length() == 4) {
      print(echo, resetMessage);
      return new FilterResult("=", 0, null);
    }
    try {
      int number = Integer.parseInt(input.substring(4).trim());
      if (number != 0) {
        if (number > 0 && number > max) {
          print(echo, "Command ignored - max column is " + max + ", but " + number + " was attempted");
          return new FilterResult(null, null, null);
        } else if (number < 0 && number < -max) {
          print(echo, "Command ignored - min column is " + -max + ", but " + number + " was attempted");
          return new FilterResult(null, null, null);
        }
        String setMessage = "Sort is specified. Use the 'fl' command to see all filter/sort-settings";
        print(echo, setMessage);
        return new FilterResult("=", number, null);
      } else {
        print(echo, resetMessage);
        return new FilterResult("=", number, null); // number is 0
      }
    } catch (NumberFormatException e) {
      print(echo, "Command ignored - expected a number, but got '" + input + "'");
      return new FilterResult(null, null, null);
    }
  }

  // An attempt at a generic filter-parser. The filter command can be like this: <command>\s*[<op>]\s*<args>
  // and the <op> can be >,=,< and args can be number of string.
  public static FilterResult parseFilter(String input, int commandLength, boolean op, Integer reset, Integer max, boolean echo, FilterResult rollback) {

    String resetMessage = "Filter is turned off";
    if (input.length() == commandLength) {
      print(echo, resetMessage);
      return new FilterResult("=", reset, null);
    }
    input = input.substring(commandLength).trim();
    if (input.isEmpty()) {
      print(echo, resetMessage);
      return new FilterResult("=", reset, null);
    }

    String setMessage = "Filter is set. Use the 'fl' command to see all filter/sort-settings";
    String opStr = null;
    if (op) {
      if (input.startsWith("<") || input.startsWith(">") || input.startsWith("=")) {
        opStr = input.substring(0, 1);
        if (input.length() == 1 || input.substring(1).trim().isEmpty()) {
          print(echo, "Command ignored - expected a number after operator '" + opStr + "'");
          return rollback;
        }
        input = input.substring(1).trim();
      } else {
        print(echo, "Command ignored - expected operator (>,=,<) as first character after command, but found '" + input.substring(0, 1) + "'");
        return rollback;
      }
    }

    if (reset != null) {
      // Means we expect a number
      try {
        int number = Integer.parseInt(input);
        if (number == reset && (opStr == null || opStr.equals("=") || opStr.equals("<"))) {
          print(echo, resetMessage);
          return new FilterResult("=", reset, null);
        }
        if (number - 1 == reset && opStr != null && opStr.equals("<")) {
          print(echo, resetMessage);
          return new FilterResult("=", reset, null);
        }
        if (max != null && number > max) {
          print(echo, "Command ignored - max is " + max + ", but " + number + " was attempted");
          return rollback;
        }
        print(echo, setMessage);
        return new FilterResult(opStr, number, null);
      } catch (NumberFormatException e) {
        print(echo, "Command ignored - expected a number, but got '" + input + "'");
        return rollback;
      }
    } else {
      // Means we expect a string, and since we never combine that with <op>, we already know we have *something* as an argument
      print(echo, setMessage);
      return new FilterResult(null, null, input);
    }
  }

  private static void print(boolean echo, String message) {
    if (echo)
      System.out.println(message);
  }
}
