package no.statnett.ecp.cds;

import no.statnett.ecp.brs.actions.ListMessages;
import no.statnett.ecp.brs.state.MessageFilter;
import no.statnett.ecp.brs.state.MessageInfo;
import no.statnett.ecp.brs.state.QueueInfo;
import no.statnett.ecp.cds.actions.ListCertificates;
import no.statnett.ecp.cds.actions.ListComponents;
import no.statnett.ecp.cds.actions.ListMessagePaths;
import no.statnett.ecp.cds.state.*;
import no.statnett.ecp.utils.Options;
import no.statnett.ecp.utils.SimpleParser;
import no.statnett.ecp.utils.URLParser;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

public class CDShell {

  public static final String VERSION = "v1.4.2";

  public static final String versionPattern = "^\\d+[.\\d+]*$";

  // Options from input-args
  private static List<String> commands = null;

  // Useful global variables, set based on commands
  private static boolean echo = true;
  private static final ComponentFilter componentFilter = new ComponentFilter();
  private static final MessagePathFilter messagePathFilter = new MessagePathFilter();
  private static final CertFilter certFilter = new CertFilter();

  private static final List<String> filterHistory = new ArrayList<>();
  private static boolean filterChangedViewNotUpdated = true;

  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String commandsFromFile = Options.parseString(initArgsList, "-f");
    String commandsFromArg = Options.parseString(initArgsList, "-c");
    if (commandsFromFile != null) {
      File commandFile = Options.checkFile(commandsFromFile, false);
      commands = Files.readAllLines(commandFile.toPath());
    } else if (commandsFromArg != null) {
      commands = Arrays.asList(commandsFromArg.split(";"));
    }
    if (commands != null) { // Some commands have been specified, we'll filter out empty lines and comments
      commands = commands.stream().filter(c -> !c.trim().isEmpty()).filter(c -> !c.trim().startsWith("#")).collect(Collectors.toList());
      if (commands.isEmpty()) {
        System.out.println("Error: No commands found in file (-f) or in argument, exiting");
        System.exit(1);
      }
    }
    return initArgsList;
  }


  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    // READ OPTIONS
    List<String> mandatoryArgs = parseOptions(args);
    if (mandatoryArgs.size() != 1) {
      if (!mandatoryArgs.isEmpty()) {
        System.out.println("\nError: Expected exactly 1 (one) mandatory argument: URL to the CD, but received " + mandatoryArgs.size() + "\n");
      }
      if (echo)
        usage();
      System.exit(1);
    }

    // LOAD CONFIG - PREPARE INITIAL STATE
    Map<String, String> cdConfig = URLParser.parse(mandatoryArgs.get(0), false);
    Config config = new Config(cdConfig);
    State state = new State();

    // Establish which ComponentDirectory this is
    state.setDashboard(retrieveDashboard(config));

    // RUN
    if (commands != null) {
      script(state, config, commands);
    } else {
      interactive(state, config);
    }
  }

  public static Dashboard retrieveDashboard(Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
    Dashboard dashboard = new Dashboard();

    URL url = new URL(protocol, config.get("host"), Integer.parseInt(config.get("port")), "/ECP_MODULE/dashboard");
    String response = getECPHTTResponse("GET", url, config.get("user"), config.get("password"), null);
    if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
      return null;
    }
    JSONObject jsonObject = new JSONObject(response);
        /*
             {
                "activeComponentsCount": 79,
                "certificateExpiration": "2026-06-13",
                "certificateType": "AUTHENTICATION",
                "dateNow": "2025-06-20T15:37:08.311084367",
                "description": "",
                "directoryRestart": null,
                "id": "50V000000000111W",
                "jobs": {
                    "completedJobsCount": 7,
                    "failedJobsCount": 0,
                    "waitingToRunJobsCount": 1
                },
                "requestCount": 0,
                "synchronization": "OK",
                "version": "4.16.0.2186",
                "versionWarning": null

         */

    dashboard.setCdCode(jsonObject.getString("id"));
    dashboard.setVersion(jsonObject.getString("version"));
    dashboard.setActiveComponents(jsonObject.getInt("activeComponentsCount"));
    return dashboard;
  }


  private static void executeCommand(State state, Config config, String input) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    if (input.startsWith("e"))
      echo = !input.substring(1).trim().equalsIgnoreCase("off");
    else if (echo && input.equalsIgnoreCase("h"))
      help();
    else if (echo && input.startsWith("fl"))
      showSettings(input);
    else if (echo && input.equalsIgnoreCase("fh"))
      filterHistory();
    else if (input.startsWith("fr")) {
      reset(echo, input);
    } else if (input.equalsIgnoreCase("v") || input.equalsIgnoreCase("c"))
      state.setCList(refreshComponentList(config, state));

    else if (input.startsWith("c")) { // Component filter commands:
      filterChangedViewNotUpdated = true; // not perfect, because if you write a faulty m-command (mdummy), you will set this flag unnecessary
      if (input.startsWith("csrt"))
        componentFilter.setCsrt(FilterResult.parseSortCmd(input, 8, echo));
      else if (input.startsWith("csch")) {
        componentFilter.setCsch(FilterResult.parseStrFilter(input, 4, echo, componentFilter.getCsch()));
        filterHistory.add(input);
      } else if (input.startsWith("cver")) {
        componentFilter.setCver(FilterResult.parseFilter(input, 4, true, null, null, echo, componentFilter.getCver(), versionPattern));
        filterHistory.add(input);
      } else if (input.startsWith("cval")) {
        componentFilter.setCval(FilterResult.parseFilter(input, 4, true, null, null, echo, componentFilter.getCval(), null));
        filterHistory.add(input);
      } else if (input.startsWith("cscp"))
        componentFilter.setCscp(FilterResult.parseStrFilter(input, 4, echo, componentFilter.getCscp(), "loc|rem"));
      else if (input.startsWith("cmpc"))
        componentFilter.setCmpc(FilterResult.parseFilter(input, 4, true, -1, null, echo, componentFilter.getCmpc(), null));
      else if (input.startsWith("cacl"))
        componentFilter.setCacl(FilterResult.parseStrFilter(input, 4, echo, componentFilter.getCacl(), "on|off"));
      else if (input.startsWith("clim"))
        componentFilter.setClim(FilterResult.parseFilter(input, 4, false, 0, null, echo, componentFilter.getClim(), null));
    }

    else if (input.startsWith("m") && input.length() >= 4) { // MessagePath filter commands:
      if (input.startsWith("msrt")) {
        messagePathFilter.setMsrt(FilterResult.parseSortCmd(input, 8, echo));
      } else if (input.startsWith("msta")) {
        messagePathFilter.setMsta(FilterResult.parseStrFilter(input, 4, echo, messagePathFilter.getMsta(), "[ayn?]{4}"));
        filterHistory.add(input);
        filterChangedViewNotUpdated = true; // not perfect, because if you write a faulty m-command (mdummy), you will set this flag unnecessary
      } else if (input.startsWith("mcod")) {
        messagePathFilter.setMcod(FilterResult.parseStrFilter(input, 4, echo, messagePathFilter.getMcod()));
        filterHistory.add(input);
        filterChangedViewNotUpdated = true; // not perfect, because if you write a faulty m-command (mdummy), you will set this flag unnecessary
      } else if (input.startsWith("mbro")) {
        messagePathFilter.setMbro(FilterResult.parseStrFilter(input, 4, echo, messagePathFilter.getMbro()));
        filterHistory.add(input);
        filterChangedViewNotUpdated = true; // not perfect, because if you write a faulty m-command (mdummy), you will set this flag unnecessary
      } else if (input.startsWith("mtyp")) {
        messagePathFilter.setMtyp(FilterResult.parseStrFilter(input, 4, echo, messagePathFilter.getMtyp()));
        filterHistory.add(input);
        filterChangedViewNotUpdated = true; // not perfect, because if you write a faulty m-command (mdummy), you will set this flag unnecessary
      } else if (input.startsWith("mlim"))
        messagePathFilter.setMlim(FilterResult.parseFilter(input, 4, false, 0, null, echo, messagePathFilter.getMlim(), null));
    }

    else if (input.startsWith("r") && input.length() >= 4) {
      if (input.startsWith("rsrt"))
        certFilter.setRsrt(FilterResult.parseSortCmd(input, 6, echo));
      else if (input.startsWith("rtyp"))
        certFilter.setRtyp(FilterResult.parseStrFilter(input, 4, echo, certFilter.getRtyp(), "^[aA].*|^[eE].*|^[sS].*|^[gG].*|^[iI].*|^[rR].*"));
      else if (input.startsWith("rsta"))
        certFilter.setRsta(FilterResult.parseStrFilter(input, 4, echo, certFilter.getRsta(), "^[aA].*|^[eE].*"));
      else if (input.startsWith("rpre"))
        certFilter.setRpre(FilterResult.parseStrFilter(input, 4, echo, certFilter.getRpre(), "^[tT].*|^[fF].*"));
      else if (input.startsWith("ract")) {
        certFilter.setRact(FilterResult.parseFilter(input, 4, true, null, null, echo, certFilter.getRact(), null));
        filterHistory.add(input);
      } else if (input.startsWith("rval")) {
        certFilter.setRval(FilterResult.parseFilter(input, 4, true, null, null, echo, certFilter.getRval(), null));
        filterHistory.add(input);
      } else if (input.startsWith("rlim"))
        certFilter.setRlim(FilterResult.parseFilter(input, 4, false, 0, null, echo, certFilter.getRlim(), null));

    }
    // Action commands
    else { // The commands below will perform an action - we'll prevent that action if v-command has not been run
      if (filterChangedViewNotUpdated)
        System.out.println("Run 'c' command, then try again. Filters may have changed or no components have been listed yet");
      else if (input.equalsIgnoreCase("m"))
        listMessagePaths(state, messagePathFilter, echo);
      else if (input.equalsIgnoreCase("r"))
        listCertificates(state, config, input, echo);
      else {
        System.out.println("Unknown command: " + input);
      }
    }
  }

  private static void script(State state, Config config, List<String> commmands) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    // There is at least ONE command - this has been checked in parseOptions()
    boolean echoOn = !commmands.get(0).startsWith("e") || !commmands.get(0).endsWith("off");
    for (String command : commmands) {
      if (echoOn)
        System.out.println(">" + command);
      executeCommand(state, config, command);
      echoOn = echo; // The executeCommand will update the echo-flag if the command is "e off" or "e on"
      if (command.equalsIgnoreCase("x") || command.equalsIgnoreCase("exit") || command.equalsIgnoreCase("quit"))
        break;
    }
    if (echo)
      System.out.println("Exiting CDShell");
  }

  private static void interactive(State state, Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    System.out.println("CDShell for ECP CD " + VERSION + " - Interactive mode. Enter h for help");
    Scanner scanner = new Scanner(System.in);
    String input;
    do {
      System.out.print(">");
      input = scanner.nextLine();
      if (!input.trim().isEmpty()) {
        executeCommand(state, config, input);
      }
    } while (!input.equalsIgnoreCase("x") && !input.equalsIgnoreCase("exit") && !input.equalsIgnoreCase("quit"));
    scanner.close();
    System.out.println("Exiting CDShell");
  }

  private static void reset(boolean echo, String input) {
    String filter = null;
    if (input.equals("fr")) {
      componentFilter.reset();
      messagePathFilter.reset();
      certFilter.reset();
      filter = "all";
    } else if (input.equals("frc")) {
      componentFilter.reset();
      filter = "component";
    } else if (input.equals("frm")) {
      messagePathFilter.reset();
      filter = "messagepath";
    } else if (input.equals("frr")) {
      certFilter.reset();
      filter = "certificate";
    }

    if (filter != null) {
      filterChangedViewNotUpdated = true;
      System.out.println(echo ? ("Reset " + filter + " filters") : "");
    }

  }

  private static List<Component> refreshComponentList(Config config, State state) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    List<Component> cList = ListComponents.Retrieve.retrieveAndFilter(config, componentFilter, messagePathFilter, state); // updates the cList and prints it
    ListComponents.Print.printToConsole(cList, echo);
    filterChangedViewNotUpdated = false;
    return cList;
  }

  private static void listMessagePaths(State state, MessagePathFilter messagePathFilter, boolean echo) throws NoSuchAlgorithmException, IOException, KeyManagementException {
//    List<Component> cList = ListMessagePaths.Retrieve.retrieveAndFilter(state.getCList(), config, messagePathFilter);
    ListMessagePaths.Print.printMP(state.getCList(), messagePathFilter, echo);
  }

  private static void listCertificates(State state, Config config, String input, boolean echo) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    Integer tmp = SimpleParser.getInt(input, false);
    List<CertInfo> certList = new ArrayList<>();
    boolean certListRefreshed = false;
    if (tmp == null || tmp < 0) {
      certList = ListCertificates.Retrieve.retrieveAndFilter(state.getCList(), config, certFilter);
      certListRefreshed = true;
    } else {
      System.out.println("Invalid component number: " + tmp + ", command ignored");
    }
    if (certListRefreshed) {
      state.setCertList(certList);
      ListCertificates.Print.print(state.getCertList(), echo);
    }
  }

  private static void addCMP() {
    // POST http://localhost:8080/ECP_MODULE/directory/paths
        /*
             {
                 "applyToEndpoints": [
                     {
                         "applied": true,
                         "canBeRevoked": false,
                         "code": "50V-SN-TEST2-AT7",
                         "company": {
                             "contactEmail": "ecp@statnett.no",
                             "contactPerson": "Morten Simonsen",
                             "contactPhone": "004793480511",
                             "organization": "Statnett (TEST2)"
                         },
                         "componentDirectory": "50V000000000111W",
                         "componentStatus": null,
                         "created": "2024-05-21T15:57:22.527",
                         "envName": null,
                         "id": "50V-SN-TEST2-AT7",
                         "implementationVersion": "4.16.0.2186",
                         "isNotPushed": true,
                         "lastModification": "2025-03-20T15:04:46.387",
                         "location": null,
                         "networks": [
                             "DefaultNetwork"
                         ],
                         "projectName": null,
                         "type": "ENDPOINT",
                         "url": [
                         ],
                         "validTo": "2025-12-02T17:18:11"
                     }
                 ],
                 "broker": "46V000000000015S",
                 "messageType": "TEST",
                 "senders": [
                     ""
                 ],
                 "sendersAll": true,
                 "type": "INDIRECT",
                 "validFrom": "2025-06-26T15:00:00"
             }
         */

  }

  private static void filterHistory() {
    for (int i = 0; i < filterHistory.size(); i++) {
      System.out.println((i + 1) + ": " + filterHistory.get(i));
    }
  }

  // Other methods - not directly related to the main method

  private static void showSettings(String input) {
    input = input.trim();
    boolean showC = input.endsWith("c") || input.equals("fl");
    boolean showM = input.endsWith("m") || input.equals("fl");
    boolean showR = input.endsWith("r") || input.equals("fl");
    boolean showAll = input.equals("fl");

    Integer cCol = componentFilter.getCsrt().number();
    String csrtFilter = cCol == 0 ? "Not specified" : (cCol > 0 ? "A" : "De") + "scending by column " + cCol;
    String cschFilter = componentFilter.getCsch().arg() == null ? "<ALL>" : toTextWithNeg(componentFilter.getCsch().arg());
    String cverFilter = componentFilter.getCver().arg() == null ? "<ALL>" : toTextWithOp(componentFilter.getCver(), "");
    String cvalFilter = componentFilter.getCval().arg() == null ? "<ALL>" : toTextWithOp(componentFilter.getCval(), "");
    String cscpFilter = componentFilter.getCscp().arg() == null ? "<ALL>" : componentFilter.getCscp().arg();
    String cmpcFilter = componentFilter.getCmpc().number() < 0 ? "<ALL>" : toTextWithOp(componentFilter.getCmpc(), "MP");
    String climFilter = componentFilter.getClim().number() < 1 ? "<ALL>" : "First " + componentFilter.getClim().number() + " rows";
    String caclFilter = componentFilter.getCacl().arg() == null ? "<ALL>" : componentFilter.getCacl().arg();


    Integer mCol = messagePathFilter.getMsrt().number();
    String msrtFilter = mCol == 0 ? "Not specified" : (mCol > 0 ? "A" : "De") + "scending by column " + mCol;
    String mstaFilter = messagePathFilter.getStatExplained();
    String mcodFilter = messagePathFilter.getMcod().arg() == null ? "<ALL>" : toTextWithNeg(messagePathFilter.getMcod().arg());
    String mbroFilter = messagePathFilter.getMbro().arg() == null ? "<ALL>" : toTextWithNeg(messagePathFilter.getMbro().arg());
    String mtypFilter = messagePathFilter.getMtyp().arg() == null ? "<ALL>" : toTextWithNeg(messagePathFilter.getMtyp().arg());
    String mlimFilter = messagePathFilter.getMlim().number() < 1 ? "<ALL>" : "First " + messagePathFilter.getMlim().number() + " rows";


    Integer rCol = certFilter.getRsrt().number();
    String rsrtFilter = rCol == 0 ? "Not specified" : (rCol > 0 ? "A" : "De") + "scending by column " + rCol;
    String rtypFilter = certFilter.getRtyp().arg() == null ? "<ALL>" : toTextWithNeg(certFilter.getRtyp().arg());
    String rstaFilter = certFilter.getRsta().arg() == null ? "<ALL>" : toTextWithNeg(certFilter.getRsta().arg());
    String rpreFilter = certFilter.getRpre().arg() == null ? "<ALL>" : toTextWithNeg(certFilter.getRpre().arg());
    String ractFilter = certFilter.getRact().arg() == null ? "<ALL>" : toTextWithOp(certFilter.getRact(), "");
    String rvalFilter = certFilter.getRval().arg() == null ? "<ALL>" : toTextWithOp(certFilter.getRval(), "");
    String rlimFilter = certFilter.getRlim().number() < 1 ? "<ALL>" : "First " + certFilter.getRlim().number() + " rows";


    System.out.println("Object    Type   Operation        Cmd       Comp      : Setting");
    System.out.println("--------------------------------------------------------------------");
    if (showC) {
      System.out.println("Comp      Sort   Column-sort      (csrt)    (=)       : " + csrtFilter);
      System.out.println("Comp      Filter Search (regexp)  (csch)    (regex)   : " + cschFilter);
      System.out.println("Comp      Filter Version          (cver)    (>,<,=)   : " + cverFilter);
      System.out.println("Comp      Filter ValidTo          (cval)    (>,<,=)   : " + cvalFilter);
      System.out.println("Comp      Filter Scope            (cscp)    (loc,rem) : " + cscpFilter);
      System.out.println("Comp      Filter MsgPath-Count    (cmpc)    (=)       : " + cmpcFilter);
      System.out.println("Comp      Filter Cert-list codes  (cacl)    (on, off) : " + caclFilter);
      System.out.println("Comp      Filter Limit-N-rows     (clim)    (=)       : " + climFilter);
    }
    if (showAll)
      System.out.println("--------------------------------------------------------------------");
    if (showM) {
      System.out.println("MsgPath   Sort   Column-sort      (msrt)    (=)       : " + msrtFilter);
      System.out.println("MsgPath   Filter Stat             (msta)    (a,y,n,?) : " + mstaFilter);
      System.out.println("MsgPath   Filter Code             (mcod)    (regex)   : " + mcodFilter);
      System.out.println("MsgPath   Filter Broker           (mbro)    (regex)   : " + mbroFilter);
      System.out.println("MsgPath   Filter MessageType      (mtyp)    (regex)   : " + mtypFilter);
      System.out.println("MsgPath   Filter Limit-N-rows     (mlim)    (=)       : " + mlimFilter);
    }
    if (showAll)
      System.out.println("--------------------------------------------------------------------");
    if (showR) {
      System.out.println("Cert      Sort   Column-sort      (rsrt)    (=)       : " + rsrtFilter);
      System.out.println("Cert      Filter Type             (rtyp)    (=)       : " + rtypFilter);
      System.out.println("Cert      Filter Status           (rsta)    (=)       : " + rstaFilter);
      System.out.println("Cert      Filter Preferred        (rpre)    (=)       : " + rpreFilter);
      System.out.println("Cert      Filter ActiveSince      (ract)    (>,<,=)   : " + ractFilter);
      System.out.println("Cert      Filter ValidTo          (rval)    (>,<,=)   : " + rvalFilter);
      System.out.println("Cert      Filter Limit-N-rows     (rlim)    (=)       : " + rlimFilter);
    }
    if (showAll)
      System.out.println("--------------------------------------------------------------------");
  }

  private static String toTextWithNeg(String searchString) {
    return searchString.startsWith("!") ? searchString.substring(1) + " (negated:true)" : searchString + " (negated:false)";
  }

  private static String toTextWithOp(FilterResult filter, String designation) {
    if (filter.number() != null) {
      // Adjusts designation for singular case
      if (designation != null && designation.length() > 1 && designation.endsWith("s") && filter.number() == 1) {
        designation = designation.substring(0, designation.length() - 1);
      }
      return filter.op() + " " + filter.number() + (designation == null ? "" : " " + designation);
    } else {
      return filter.op() + " " + filter.arg() + (designation == null ? "" : " " + designation);
    }
  }

  // Interactive help
  private static void help() {
    System.out.println(" MISC:");
    System.out.println("   h                - help");
    System.out.println("   x                - exit");
    System.out.println("   e<on|off>        - off: will output only the most necessary information, useful for script-mode");
    System.out.println(" LISTINGS:");
    System.out.println("   c                - view components - the #MPs column (6) will depend on the message path filters specified");
    System.out.println("   m                - view messagepaths for all components listed with c-command");
    System.out.println("   r                - list certificates for all components listed with c-command");
    System.out.println("   fh               - show filter (search) history");
    System.out.println("   fl[c|m|r]        - show filter/sort settings");
    System.out.println(" RESET FILTER:");
    System.out.println("   fr[c|m|r]        - reset all filters. Optional: reset only component-filter 'frc', messagepath-filter 'frm' or cert-filter 'frt'");
    System.out.println("   <filter-command> - a filter command (see below) without it's argument will reset the filter to match <ALL>.");
    System.out.println(" COMPONENT FILTERS:");
    System.out.println("   csrt<number>     - sort column ascending using column number 1 to 8, or descending from -1 to -8");
    System.out.println("   csch<string>     - match lines with this regexp/string. Prefixing with ! will negate search.");
    System.out.println("                      example of searches:");
    System.out.println("                        test            - match rows containing 'test'");
    System.out.println("                        !4.12|4.11      - match rows NOT containing '4.12' nor '4.11'");
    System.out.println("                        4.12.*Stat      - match rows containing '4.12<zero-or-more-char>Stat'");
    System.out.println("   cver<op><ver>    - list versions greater/lower/equal to <ver>. <op> can be '>','<' or '='.");
    System.out.println("   cval<op><tms>    - list validTo greater/lower/equal to <tms>. <op> can be '>','<' or '='.");
    System.out.println("                      example of version-filters:");
    System.out.println("                        >2025");
    System.out.println("                        <2025-01-01");
    System.out.println("                        >2025-01-01T18:00");
    System.out.println("                        =null");
    System.out.println("   cscp<type>       - <type> can be 'loc' (for 'local endpoint'), 'rem' (for 'remote endpoint')");
    System.out.println("   cmpc<op><number> - <number> is the exact number of message path count to filter for. <op> must be '=' ");
    System.out.println("   clim<number>     - limit output to this number of lines (after filter/sorting is applied).");
    System.out.println("   cacl<on|off>     - apply latest certificate list ('r' command) as component code filter");
    System.out.println(" MESSAGEPATH FILTERS:");
    System.out.println("   msrt<number>     - sort column ascending using column number 1 to 8, or descending from -1 to -8");
    System.out.println("   msta<filter>     - The 4 columns 'valid', 'connected', 'local' and 'active' is filtered based on the <filter> which is always 4 character long.");
    System.out.println("                      Each character represents one column: 'a' = 'all', 'y' = 'yes/true', 'n' = 'no/false' and '?' = 'unknown/failed'.");
    System.out.println("                      The ? option is used for 2 colums with a distinct 3 state");
    System.out.println("                          connected - connection-status is unknown");
    System.out.println("                          local     - central path has failed to be exported to endpoint");
    System.out.println("                      example of filters: ");
    System.out.println("                          'aaaa' (show all)");
    System.out.println("                          'yayn' (valid paths, connected & disconnected, only local, only inactive)");
    System.out.println("                          'yany' (valid paths, connected & disconnected, only central, only active)");
    System.out.println("                          'a?aa' (show paths with unknown connected status)");
    System.out.println("                          'aa?a' (show central paths not exported correctly to endpoints)");
    System.out.println("   mcod<code>       - match endpoint codes for the messagepath using regex/string. Prefixing with ! will negate search.");
    System.out.println("   mbro<code>       - match broker codes for the messagepath using regex/string. Prefixing with ! will negate search.");
    System.out.println("   mtyp<type>       - match messagepath types for the messagepath using regex/string. Prefixing with ! will negate search.");
    System.out.println("   mlim<number>     - limit output to this number of lines (after filter/sorting is applied).");
    System.out.println(" CERTIFICATE FILTERS:");
    System.out.println("   rsrt<number>     - sort column ascending using column number 1 to 6, or descending from -1 to -6");
    System.out.println("   rtyp<type>       - <type> can be eiter 'auth', 'enc' or 'sign'");
    System.out.println("   rsta<status>     - <status> can be either 'active' or 'expired'");
    System.out.println("   rpre<preferred>  - <preferred> can be either 'true' or 'false'");
    System.out.println("   ract<op><tms>    - list activeSince greater/lower/equal to <tms>. <op> can be '>','<' or '='. Examples: see cval-filter");
    System.out.println("   rval<op><tms>    - list validTo greater/lower/equal to <tms>. <op> can be '>','<' or '='. Examples:see cval-filter");
    System.out.println("   rlim<number>     - limit output to this number of lines (after filter/sorting is applied).");
    System.out.println(" ACTIONS - TAKE CARE!:");
    System.out.println("   a<number>        - accept request with this number. Accept ALL requests in the printed list if number is -1");
  }

  // Command line help
  private static void usage() {
    System.out.println("CDShell (CDS) " + VERSION + " is made to browse and manage ECP components. Some use cases are listed below, ");
    System.out.println("with command sequence to accomplish it");
    System.out.println();
    System.out.println("1) Find contact email addresses for all endpoints with too old ECP-version");
    System.out.println("   >cscp loc");
    System.out.println("   >cver < 4.14");
    System.out.println("   >c\n");

    System.out.println("2) Find endpoints without active messagepaths");
    System.out.println("   >msta aaay");
    System.out.println("   >cmpc = 0");
    System.out.println("   >c\n");

    System.out.println("3) Find local endpoints without message path to specific broker");
    System.out.println("   >cscp loc");
    System.out.println("   >mbro 50V000000000112U");
    System.out.println("   >cmpc = 0");
    System.out.println("   >c\n");

    System.out.println("4) Find endpoints where central message path has not been applied");
    System.out.println("   >msta aa?a");
    System.out.println("   >cmpc > 0");
    System.out.println("   >c\n");

    System.out.println("5) Find endpoints where AUTH certs expire within a specific date");
    System.out.println("   >rtyp AUTH");
    System.out.println("   >rsta ACTIVE");
    System.out.println("   >rval < 2026-01-25");
    System.out.println("   >r");
    System.out.println("   >cacl");
    System.out.println("   >c\n");


    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar CDS [OPTIONS] CD-URL\n");
    System.out.println("\n\nThe options and arguments:");
    System.out.println(" OPTIONS       : ");
    System.out.println("                 -f <commands-file> : each line in the file specifies an interactive command.");
    System.out.println("                 -c <commands>      : a semicolon separated list of interactive commands. Ignored if -f is specified");
    System.out.println(" CD-URL        : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("\nExample 1:  Start CDS in interactive mode");
    System.out.println("\tjava -jar ekit.jar CDS https://admin:password@localhost:8443");
  }
}
