package no.statnett.ecp.cds;

import no.statnett.ecp.cds.actions.ListComponents;
import no.statnett.ecp.cds.actions.ListMessagePaths;
import no.statnett.ecp.cds.state.*;
import no.statnett.ecp.utils.Options;
import no.statnett.ecp.utils.SimpleParser;
import no.statnett.ecp.utils.URLParser;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

public class FAMonitor {

  public static final String VERSION = "v0.9.0";

  // Options from input-args
  private static List<String> commands = null;

  // Useful global variables, set based on commands
  private static final ComponentFilter componentFilter = new ComponentFilter();
  private static final MessagePathFilter messagePathFilter = new MessagePathFilter();


  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    String commandsFromFile = Options.parseString(initArgsList, "-f");
    String commandsFromArg = Options.parseString(initArgsList, "-c");
    if (commandsFromFile != null) {
      File commandFile = Options.checkFile(commandsFromFile, false);
      commands = Files.readAllLines(commandFile.toPath());
    } else if (commandsFromArg != null) {
      commands = Arrays.asList(commandsFromArg.split(";"));
    }
    if (commands != null) { // Some commands have been specified, we'll filter out empty lines and comments
      commands = commands.stream().filter(c -> !c.trim().isEmpty()).filter(c -> !c.trim().startsWith("#")).collect(Collectors.toList());
      if (commands.isEmpty()) {
        System.out.println("Error: No commands found in file (-f) or in argument, exiting");
        System.exit(1);
      }
    }
    return initArgsList;
  }


  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    // READ OPTIONS
    List<String> mandatoryArgs = parseOptions(args);
    if (mandatoryArgs.size() != 1) {
      if (!mandatoryArgs.isEmpty()) {
        System.out.println("\nError: Expected exactly 1 (one) mandatory argument: URL to the CD, but received " + mandatoryArgs.size() + "\n");
      }
      System.exit(1);
    }

    // LOAD CONFIG - PREPARE INITIAL STATE
    Map<String, String> cdConfig = URLParser.parse(mandatoryArgs.get(0), false);
    Config config = new Config(cdConfig);
    State state = new State();

    // Establish which ComponentDirectory this is
    state.setDashboard(retrieveDashboard(config));

    // RUN
    executeMonitor(state, config);
  }

  public static Dashboard retrieveDashboard(Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
    Dashboard dashboard = new Dashboard();

    URL url = new URL(protocol, config.get("host"), Integer.parseInt(config.get("port")), "/ECP_MODULE/dashboard");
    String response = getECPHTTResponse("GET", url, config.get("user"), config.get("password"), null);
    if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
      return null;
    }
    JSONObject jsonObject = new JSONObject(response);
        /*
             {
                "activeComponentsCount": 79,
                "certificateExpiration": "2026-06-13",
                "certificateType": "AUTHENTICATION",
                "dateNow": "2025-06-20T15:37:08.311084367",
                "description": "",
                "directoryRestart": null,
                "id": "50V000000000111W",
                "jobs": {
                    "completedJobsCount": 7,
                    "failedJobsCount": 0,
                    "waitingToRunJobsCount": 1
                },
                "requestCount": 0,
                "synchronization": "OK",
                "version": "4.16.0.2186",
                "versionWarning": null

         */

    dashboard.setCdCode(jsonObject.getString("id"));
    dashboard.setVersion(jsonObject.getString("version"));
    dashboard.setActiveComponents(jsonObject.getInt("activeComponentsCount"));
    return dashboard;
  }

  private static void executeMonitor(State state, Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    // There is at least ONE command - this has been checked in parseOptions()
    long startMs = System.currentTimeMillis();
    componentFilter.setCsch(new FilterResult(null, null, "FA\\)"));
    while (true) {
      // 1.time the componentFilter.getEndpoints() is empty - will search for all endpoints
      state.setCList(refreshComponentList(config, state));
      if (state.getCList().isEmpty()) {
        System.out.println("After " + (System.currentTimeMillis() - startMs) / 1000 + " seconds, all endpoints have successfully synchronized");
        break;
      }
      // Next round - only search for these endpoints!
      componentFilter.setEndpoints(state.getCList().stream().map(c -> c.getCode()).collect(Collectors.toList()));
      System.out.println("We've run for " + (System.currentTimeMillis() - startMs) / 1000 + " seconds and still have " + componentFilter.getEndpoints().size() + " that reports (FA)iled synchronization");
      try {
        Thread.sleep(60000);
      } catch (InterruptedException e) {
        throw new RuntimeException(e);
      }
    }
    System.out.println("Exiting CDShell");
  }

  private static List<Component> refreshComponentList(Config config, State state) throws NoSuchAlgorithmException, IOException, KeyManagementException {
    List<Component> cList = ListComponents.Retrieve.retrieveAndFilter(config, componentFilter, messagePathFilter, state); // updates the cList and prints it
    ListComponents.Print.printToConsole(cList, true);
    return cList;
  }

}
