package no.statnett.ecp.cds.actions;

import no.statnett.ecp.cds.Config;
import no.statnett.ecp.cds.state.*;
import no.statnett.ecp.utils.Const;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;

import static no.statnett.ecp.utils.EcpHTTP.getECPHTTResponse;

public class ListCertificates {

  public static class Print {
    public static void print(List<CertInfo> certList, boolean echo) {
      int i = 1;

      if (echo) {
        String codeHd = "Code (1)";
        String typeHd = "Type (2)";
        String statusHd = "Status (3)";
        String preferredHd = "Preferred (4)";
        String activeHd = "ActiveSince (5)";
        String validHd = "ValidTo (6)";

        int codeWidth = Math.max(certList.stream().mapToInt(m -> m.getCode().length()).max().orElse(0), codeHd.length());
        int typeWidth = 14; // Max is AUTHENTICATION
        int statusWidth = 10; // Active, Expired
        int preferredWidth = 13; // true, false
        int activeWidth = 19; // 2025-11-03T09:50:01
        int validWidth = 19; // 2025-11-03T09:50:01

        System.out.println(String.format("%-4s | %-" + codeWidth + "s | %-" + typeWidth + "s | %-" + statusWidth + "s | %-" + preferredWidth + "s | %-" + activeWidth + "s | %-" + validWidth + "s", "#", codeHd, typeHd, statusHd, preferredHd, activeHd, validHd) + " (" + LocalDateTime.now().format(Const.localTmsSec) + ")");
        int totalWidth = codeWidth + typeWidth + statusWidth + preferredWidth + activeWidth + validWidth + 4 + 6 * 3;
        System.out.println("-".repeat(totalWidth));

        for (CertInfo c : certList) {
          String sb = String.format("%-4d |", i) +
              String.format(" %-" + codeWidth + "s |", c.getCode()) +
              String.format(" %-" + typeWidth + "s |", c.getType()) +
              String.format(" %-" + statusWidth + "s |", c.getStatus()) +
              String.format(" %-" + preferredWidth + "s |", c.isPreferred()) +
              String.format(" %-" + activeWidth + "s |", c.getActiveSince()) +
              String.format(" %-" + validWidth + "s", c.getValidTo());
          System.out.println(sb);
          i++;
        }
      }
    }
  }


  public static class Retrieve {
    public static List<CertInfo> retrieveAndFilter(List<Component> cList, Config config, CertFilter certFilter) throws NoSuchAlgorithmException, IOException, KeyManagementException {
      List<CertInfo> certList = retrieve(cList, config);
      List<CertInfo> filteredByType = filterByType(certList, certFilter.getRtyp().arg());
      List<CertInfo> filteredByStatus = filterByStatus(filteredByType, certFilter.getRsta().arg());
      List<CertInfo> filteredByPreferred = filterByPreferred(filteredByStatus, certFilter.getRpre().arg());
      List<CertInfo> filteredByActiveSince = filterOnTms("activeSince", filteredByPreferred, certFilter.getRact().arg(), certFilter.getRact().op());
      List<CertInfo> filteredByValidTo = filterOnTms("validTo", filteredByActiveSince, certFilter.getRval().arg(), certFilter.getRval().op());
      List<CertInfo> sortedCertList = sortCertificates(filteredByValidTo, certFilter.getRsrt().number());
      List<CertInfo> limitedCList = certFilter.getRlim().number() > 0 ? sortedCertList.subList(0, Math.min(certFilter.getRlim().number(), sortedCertList.size())) : sortedCertList;
      return limitedCList;
    }

    private static List<CertInfo> retrieve(List<Component> cList, Config config) throws NoSuchAlgorithmException, IOException, KeyManagementException {
      String protocol = config.get("suggest") == null ? "https" : config.get("suggest").equals("tls") ? "https" : "http";
      List<CertInfo> certList = new ArrayList<>();
      for (Component c : cList) {
        URL mpURL = new URL(protocol, config.get("host"), Integer.parseInt(config.get("port")), "/ECP_MODULE/components/" + c.getCode() + "/certificates?page=0&count=1000");
        String response = getECPHTTResponse("GET", mpURL, config.get("user"), config.get("password"), null);
        if (!response.trim().startsWith("{") || !response.trim().endsWith("}")) {
          System.out.println("Error occured when trying to retrieve url: " + mpURL);
          System.out.println("Response:\n" + response);
        }

        JSONObject jsonObject = new JSONObject(response);

            /*
               {
                   "data": [
                       {
                           "activeSince": "2024-11-15T14:57:51",
                           "canBeRevoked": false,
                           "certificateAuthority": null,
                           "componentCode": null,
                           "componentDirectory": null,
                           "id": "Q049NDZWMDAwMDAwMDAwMDAxMiwgQ049ZWNwNC10ZXN0LnN2ay5zZSwgTz1TVkssIEM9U0UyMTk3MDY3MTczMTY4MjY3MTc0Mw",
                           "preferred": false,
                           "reason": null,
                           "status": "Expired",
                           "type": "AUTHENTICATION",
                           "validTo": "2025-11-15T15:57:51"
                       },
                  ...
                  ]
               }
               */

        JSONArray dataArray = jsonObject.getJSONArray("data");

        // Iterate over the data array
        for (int i = 0; i < dataArray.length(); i++) {
          CertInfo certInfo = new CertInfo();
          certInfo.setCode(c.getCode());
          JSONObject dataObject = dataArray.getJSONObject(i);
          certInfo.setType(dataObject.getString("type"));
          certInfo.setStatus(dataObject.getString("status"));
          certInfo.setPreferred(dataObject.getBoolean("preferred"));
          certInfo.setActiveSince(dataObject.getString("activeSince"));
          certInfo.setValidTo(dataObject.getString("validTo"));
          certList.add(certInfo);
        }
      }
      return certList;
    }
  }

  private static List<CertInfo> filterByType(List<CertInfo> certList, String type) {
    if (type == null || type.trim().isEmpty())
      return certList;
    List<CertInfo> filteredCertList = new ArrayList<>();
    for (CertInfo c : certList) {
      if (c.getType().toLowerCase().substring(0, 1).equalsIgnoreCase(type.substring(0, 1))) {
        filteredCertList.add(c);
      }
    }
    return filteredCertList;
  }

  private static List<CertInfo> filterByStatus(List<CertInfo> certList, String status) {
    if (status == null || status.trim().isEmpty())
      return certList;
    List<CertInfo> filteredCertList = new ArrayList<>();
    for (CertInfo c : certList) {
      if (c.getStatus().toLowerCase().substring(0, 1).equalsIgnoreCase(status.substring(0, 1))) {
        filteredCertList.add(c);
      }
    }
    return filteredCertList;
  }

  private static List<CertInfo> filterByPreferred(List<CertInfo> certList, String preferred) {
    if (preferred == null || preferred.trim().isEmpty())
      return certList;
    List<CertInfo> filteredCertList = new ArrayList<>();
    for (CertInfo c : certList) {
      if (c.isPreferred() == preferred.equalsIgnoreCase("t")) {
        filteredCertList.add(c);
      }
    }
    return filteredCertList;
  }

  private static List<CertInfo> filterOnTms(String property, List<CertInfo> certList, String filterArg, String filterOp) {
    if (filterArg == null || filterArg.trim().isEmpty())
      return certList;
    List<CertInfo> filteredCList = new ArrayList<>();
    for (CertInfo c : certList) {
      String compareProperty = null;
      if (property.equals("validTo"))
        compareProperty = c.getValidTo();
      else if (property.equals("activeSince"))
        compareProperty = c.getActiveSince();
      int compareResult = compareProperty.compareTo(filterArg);
      if (filterOp.equals("=") && compareResult == 0) {
        filteredCList.add(c);
      } else if (filterOp.equals(">") && compareResult > 0) {
        filteredCList.add(c);
      } else if (filterOp.equals("<") && compareResult < 0) {
        filteredCList.add(c);
      }
    }
    return filteredCList;
  }

  private static List<CertInfo> sortCertificates(List<CertInfo> certList, int sort) {
    if (sort > 0) {
      switch (sort) {
        case 1:
          certList.sort(Comparator.comparing(CertInfo::getCode));
          break;
        case 2:
          certList.sort(Comparator.comparing(CertInfo::getType));
          break;
        case 3:
          certList.sort(Comparator.comparing(CertInfo::getStatus));
          break;
        case 4:
          certList.sort(Comparator.comparing(CertInfo::isPreferred));
          break;
        case 5:
          certList.sort(Comparator.comparing(CertInfo::getActiveSince));
          break;
        case 6:
          certList.sort(Comparator.comparing(CertInfo::getValidTo));
          break;
      }
    } else if (sort < 0) {
      switch (sort) {
        case -1:
          certList.sort(Comparator.comparing(CertInfo::getCode).reversed());
          break;
        case -2:
          certList.sort(Comparator.comparing(CertInfo::getType).reversed());
          break;
        case -3:
          certList.sort(Comparator.comparing(CertInfo::getStatus).reversed());
          break;
        case -4:
          certList.sort(Comparator.comparing(CertInfo::isPreferred).reversed());
          break;
        case -5:
          certList.sort(Comparator.comparing(CertInfo::getActiveSince).reversed());
          break;
        case -6:
          certList.sort(Comparator.comparing(CertInfo::getValidTo).reversed());
          break;
      }
    }
    return certList;
  }
}

