package no.statnett.ecp.dc;

import no.statnett.ecp.utils.Const;
import no.statnett.ecp.utils.EcpHTTP;
import no.statnett.ecp.utils.LogRotator;
import no.statnett.ecp.utils.Options;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

public class DashboardCheck {

  public static final String VERSION = "v1.1.0";

  private static File logFile;
  private static String rotationType = null;
  private static int noOfLogFiles = 0;
  private static int noOfDaysDelayRotation = 0;
  private static boolean compress = false;

  private static Integer timeout;

  private static String user;
  private static String pw;
  private static URL url;

  private static List<String> parseOptions(String[] initialArgs) throws IOException {
    List<String> initArgsList = new ArrayList<>(Arrays.asList(initialArgs));
    timeout= Options.parseInt(initArgsList,"-t",1,10000,0)*1000;
    String logFilename = Options.parseString(initArgsList, "-o");
    if (logFilename != null) {
      logFile = Options.checkFile(logFilename, true);
    }
    rotationType = Options.parseString(initArgsList, "-r");
    if (rotationType != null && logFile == null) {
      System.err.println("Rotate option (-r) requires output file option (-o) - exiting");
      System.exit(1);
    }
    if (rotationType != null && !rotationType.matches("^(day|week|month)$")) {
      System.err.println("Rotate option (-r) must be either 'day', 'week' or 'month' - exiting");
      System.exit(1);
    }
    noOfLogFiles = Options.parseInt(initArgsList, "-m", 1, 1000, 10);
    noOfDaysDelayRotation = Options.parseInt(initArgsList, "-n", 0, 1000, 0);
    compress = Options.parseBoolean(initArgsList, "-z");
    return initArgsList;
  }


  private static void parseMandatoryArg(List<String> mandatoryArgs) throws MalformedURLException {
    if (mandatoryArgs.size() == 1) {
      URL initialURL = new URL(mandatoryArgs.get(0));
      url = new URL(initialURL.getProtocol(), initialURL.getHost(), initialURL.getPort(), "/ECP_MODULE/dashboard");
      if (initialURL.getUserInfo() != null) {
        user = initialURL.getUserInfo().split(":")[0];
        pw = initialURL.getUserInfo().split(":")[1];
      } else {
        System.out.println("The URL must contain user:password");
        System.exit(1);
      }
    } else {
      usage();
      System.exit(1);
    }
  }

  public static void main(String[] args) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    List<String> mandatoryArgs = parseOptions(args);

    parseMandatoryArg(mandatoryArgs);

    LocalDateTime now = LocalDateTime.now();

    String output = EcpHTTP.getECPHTTResponse("GET", url, user, pw, null, timeout);

    if (logFile == null) {
      System.out.println(output);
    } else {
      Files.writeString(logFile.toPath(), output + "\n", StandardOpenOption.APPEND);
      if (rotationType != null && Long.parseLong(now.format(Const.tmsNumberMillisec)) > 0) {
        LogRotator logRotator = new LogRotator(logFile, now.atZone(ZoneId.systemDefault()), noOfDaysDelayRotation);
        logRotator.rotateLogs(rotationType);
        logRotator.deleteExcessLogs(noOfLogFiles);
        if (compress)
          logRotator.compress();
      }
    }
  }


  private static void usage() {
    System.out.println("DashboardCheck (DC) " + VERSION + " can connect to an ECP-endpoint and retrieve information from the dashboard");
    System.out.println();
    System.out.println("Usage  : java -jar ekit.jar DC <ECP-URL>\n");
    System.out.println("\n\nThe arguments:");
    System.out.println(" OPTIONS      ");
    System.out.println("     -t<sec>           : Specify a timeout for the request between 0 and 10000. 0 means no timeout, which is default.");
    System.out.println("     -o <OUTPUT-FILE>  : Write output to file instead of stdout");
    System.out.println("     -r day|week|month : Rotate log-file every day, week or month. Requires -o option");
    System.out.println("     -m<noLogFiles>    : Allow maximum number of log-files. Default is 10. Requires -r option");
    System.out.println("     -n<noDays>        : Do not rotate data younger than noDays. Default is 0. If set higher, the output log will");
    System.out.println("                         always contain the last noDays of data, even if the log-file is rotated. This is useful");
    System.out.println("                         for analysis of the last days of data (GapMonitor). Requires -r option");
    System.out.println("     -z                : Compress rotated logs. Requires -r option");
    System.out.println("MANDATORY ARGUMENTS");
    System.out.println("     <ECP-URL>         : Specify url on this form: http[s]://user:pass@host:port");
    System.out.println();
    System.out.println("\nExample 1:  Get dashboard from localhost-endoint");
    System.out.println("\tjava -jar ekit.jar DC https://admin:password@localhost:8443");
  }

}
