package no.statnett.ecp.edx;

import no.statnett.ecp.utils.EcpEdxLog;

import java.util.*;

public class Measurement {
  private long yyyyMMddHHmm;
  private boolean snd;
  private Set<String> ecpIds = new HashSet<>();
  private Set<String> edxIds = new HashSet<>();
  private List<String> ecpLogLines = new ArrayList<>();
  private List<String> edxLogLines = new ArrayList<>();
  private List<String> ecpExpired = new ArrayList<>(); // List of expired message due to TTL in ECP
  private List<String> negativeACK = new ArrayList<>(); // List of failed messages in EDX
  private Map<String, String> correlationMap = new HashMap<>();


  public Measurement(long yyyyMMddHHmm, boolean snd) {
    this.yyyyMMddHHmm = yyyyMMddHHmm;
    this.snd = snd;
  }

  public long getTms() {
    return yyyyMMddHHmm;
  }

  public void addEcpId(String ecpId, String logLine) {
    ecpIds.add(ecpId);
    ecpLogLines.add(logLine);
  }

  public void addEdxId(String edxId, String logLine) {
    edxIds.add(edxId);
    edxLogLines.add(logLine);
  }

  public void removeEdxId(String edxId) {
    edxIds.remove(edxId);
    List newList = new ArrayList();
    for (String edxLogLine : edxLogLines) {
      if (!edxLogLine.contains(edxId)) {
        newList.add(edxLogLine);
      }
    }
    edxLogLines = newList;
  }

  public void addEcpExpired(String expired) {
    this.ecpExpired.add(expired);
  }

  public void addNegativeACK(String negativeACK) {
    this.negativeACK.add(negativeACK);
  }

  public int diff() {
    // If you send messages, you can allow that ECP size is bigger than EDX size
    // If you receive messages, you can allow that EDX size is bigger than ECP size
    if (snd) {
      return ecpIds.size() - edxIds.size(); // Should never be less than 0
    } else {
      return edxIds.size() - ecpIds.size(); // Should never be less than 0
    }
  }

  public int getEcpTraffic() {
    return ecpIds.size();
  }

  public int getEdxTraffic() {
    return edxIds.size();
  }

  public List<String> getEcpLogLines() {
    return ecpLogLines;
  }

  public List<String> getEdxLogLines() {
    return edxLogLines;
  }

  public List<String> getEcpExpired() {
    return ecpExpired;
  }

  public List<String> getNegativeACK() {
    return negativeACK;
  }

  public void correlate(String edxMessageId, String ecpMessageId) {
    correlationMap.put(edxMessageId, ecpMessageId);
  }

  public int getCorrelationSize() {
    return correlationMap.size();
  }

  public List<String> getSndDiff() {
    // Go through list of ECP sent and match with EDX sent. Remove matches, so that we're left with those EDX sent that have no
    // corresponding ECP sent. Our correlationMap will not always contain a mapping - but if it does, we can reduce the number
    // of diffs that we're going to print. The whole point is to have fewer diff-lines to go through.
    List<String> diffLines = new ArrayList<>();

    for (String edxLogLine : edxLogLines) {
      String edxId = EcpEdxLog.findEdxMessageIDInLogLine(edxLogLine);
      String ecpId = correlationMap.get(edxId);
      if (ecpId == null) {
        diffLines.add(edxLogLine);
      }
    }
    return diffLines;
  }

}
