package no.statnett.ecp.emd;

import java.util.Map;
import java.util.TreeMap;

public class OppositeSummary {

    // "timestamp": "2023-06-23 08:00:00.000Z", "direction": "snd",
    private String time; // yyyy-MM eller yyyy-MM-dd
    private String direction; // snd/rcv

    private int delayLim;
    private int expireLim;


    private Map<String, OppositeEndpointSummary> endpointMap = new TreeMap<>();

    public OppositeSummary(String time, String direction, String endpoint, int delayLim, int expireLim, String line) {
        this.time = time;
        this.direction = direction;
        this.delayLim = delayLim;
        this.expireLim = expireLim;
        add(endpoint, line);
    }

    public String getTime() {
        return time;
    }

    // line: "cnt":    2, "avgMs":    753, "avgKB":     2, "avgKBs":     2, "freq":   2, "cntLim":    0, "avgMsLim":      0, "avgKBLim":     0, "avgKBsLim":     0, "freqLim":   0, "freqLimSelf":   0, "troubleIndex":        0
    // In version 1.10.0 changed to
    // line: "cnt":    2, "avgMs":    753, "avgKB":     2, "freq":   2, "cntLim":    0, "avgMsLim":      0, "avgKBLim":     0, "freqLim":   0, "freqLimSelf":   0, "troubleIndex":        0
    public OppositeSummary add(String endpoint, String line) {
        OppositeEndpointSummary endpointSummary = endpointMap.get(endpoint);
        if (endpointSummary == null) {
            endpointSummary = new OppositeEndpointSummary(endpoint);
            endpointMap.put(endpoint, endpointSummary);
        }
        endpointSummary.add(line);
        return this;
    }

    public String print() {
        StringBuffer sb = new StringBuffer();
        long totalCnt = endpointMap.values().stream().mapToLong(e -> e.cnt).sum();
        long totalLimitCnt = endpointMap.values().stream().mapToLong(e -> e.cntLim).sum();
        for (OppositeEndpointSummary endpointSummary : endpointMap.values()) {
            sb.append(endpointSummary.print(totalCnt, totalLimitCnt));
        }
        return sb.toString();
    }

    public class OppositeEndpointSummary {
        private String endpoint;

        private long cnt; // count
        private long cntLim; // count delayed
        private long totMs; // total duration
        private long totMsLim; // total duration delayed


        public OppositeEndpointSummary(String endpoint) {
            this.endpoint = endpoint;
        }

        public OppositeEndpointSummary add(String line) {
            String[] array = line.split(",");

            long cnt = get(array[0]);
            long cntLim = get(array[4]);
            long avgMs = get(array[1]);
            long avgMsLim = get(array[5]);
            long totMs = cnt * avgMs;
            long totMsLim = cntLim * avgMsLim;

            this.cnt += cnt;
            this.cntLim += cntLim;
            this.totMs += totMs;
            this.totMsLim += totMsLim;

            return this;
        }

        private long get(String cell) {
            return Long.parseLong(cell.split(":")[1].trim());
        }

        public String print(long totalCount, long totalLimitCount) {
            StringBuffer sb = new StringBuffer();
            long cntOK = cnt - cntLim;
            long avgMs = cnt > 0 ? totMs / cnt : 0;
            long avgMsOK = cntOK > 0 ? (totMs - totMsLim) / (cntOK) : 0;
            long avgMsDel = cntLim > 0 ? totMsLim / cntLim : 0;

            if (time.length() == 7) {
                sb.append("{ \"month\": \"").append(time).append("\",    ");
            } else {
                sb.append("{  \"date\": \"").append(time).append("\", ");
            }

            long freqAll = beregnProsent(cnt, totalCount); // totalCount always greater than 0 - range 0-100
            long freqDelayLimitAll = beregnProsent(cntLim, totalLimitCount);
            long freqDelayLimitSelf = (100L * cntLim) / cnt; // how much delayLimits compared to all messages to this endpoint (100% = as many delayLimits as messages)

            long delayLimitOverrepresentation = freqAll == 0 ? 0 : (100L * freqDelayLimitAll) / freqAll; // are there more delayLimits than to be expected based on the overall traffic to this endpoint? (100% = as many delayLimits as expected)
            int delayLimitMs = delayLim * 1000;
            long avgLimLimDiff = avgMsDel - delayLimitMs; // difference between the average of those messages exceeding the delayLimit and the delayLimit itself
            long maxExpire = Math.max(expireLim, delayLim) * 1000L;
            long delayLimitAvgIndex = avgLimLimDiff <= 0 ? 0 : (avgLimLimDiff >= maxExpire ? 100 : 1 + ((99L * avgLimLimDiff) / maxExpire)); // 0 = no delay, 1 = average delay equals delayLimit, 100 = 10 minutes delay or more on average (for those exceeding the delayLimit)
            long troubleIndex = (delayLimitOverrepresentation * freqDelayLimitSelf * cntLim * delayLimitAvgIndex) / 100L;

            sb.append("\"direction\": \"").append(direction).append("\", ").
                    append("\"endpoint\": \"").append(endpoint).append("\", ").
                    append("\"cnt\": ").append(String.format("%7d", cnt)).append(", ").
                    append("\"cntOK\": ").append(String.format("%7d", cntOK)).append(", ").
                    append("\"cntLim\": ").append(String.format("%5d", cntLim)).append(", ").
                    append("\"avgMs\": ").append(String.format("%10d", avgMs)).append(", ").
                    append("\"avgMsOK\": ").append(String.format("%5d", avgMsOK)).append(", ").
                    append("\"avgMsLim\": ").append(String.format("%10d", avgMsDel)).append(", ").
                    append("\"pct\": ").append(String.format("%3d", freqAll)).append(", ").
                    append("\"pctLim\": ").append(String.format("%3d", freqDelayLimitAll)).append(", ").
                    append("\"pctLimSelf\": ").append(String.format("%3d", freqDelayLimitSelf)).append(", ").
                    append("\"troubleIndex\": ").append(String.format("%9d", troubleIndex)).
                    append(" }\n");
            return sb.toString();
        }

        private int beregnProsent(long cnt, long totalCount) {
            if (totalCount == 0) {
                return 0;
            }
            int prosent = Math.round((100f * cnt) / totalCount);
            if (prosent == 0 && cnt > 0) {
                return 1;
            } else {
                return prosent;
            }
        }
    }

}
