package no.statnett.ecp.pf;

import jakarta.jms.*;
import no.statnett.ecp.utils.Broker;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

public class Consumer implements Runnable {

    private Map<String, String> amqpConfig;
    private String queueName;

    private Connection connection;
    private Session session;
    private MessageConsumer consumer;

    private AtomicInteger msgConsumedCounter = new AtomicInteger(0);
    private AtomicInteger msgConsumedCounterTotal = new AtomicInteger(0);

    public Consumer(Map<String, String> amqpConfig, String queueName) throws JMSException {
        this.amqpConfig = amqpConfig;
        this.queueName = queueName;
        init(amqpConfig, queueName);
    }

    public int resetMsgConsumedCounter() {
        return msgConsumedCounter.getAndSet(0);
    }

    public int resetMsgConsumedCounterTotal() {
        return msgConsumedCounterTotal.getAndSet(0);
    }

    public int getMsgConsumedCounterTotal() {
        return msgConsumedCounterTotal.get();
    }

    private void init(Map<String, String> amqpConfig, String queueName) throws JMSException {
        connection = Broker.createAndStartConnection(amqpConfig);
        session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        consumer = session.createConsumer(session.createQueue(queueName), null);
    }

    @Override
    public void run() {
        try {
            while (true) {
                BytesMessage message = (BytesMessage) consumer.receive(10000);
                if (message != null) {
                    msgConsumedCounter.incrementAndGet();
                    msgConsumedCounterTotal.incrementAndGet();
                }
            }
        } catch (Throwable t) {
            System.out.println("Consumer experienced an error " + t + ", but why quit? Let's restart the connection instead and see what happens.");
            try {
                if (this.consumer != null) {
                    consumer.close();
                }
                if (this.session != null) {
                    session.close();
                }
                if (this.connection != null) {
                    connection.close();
                }
                init(amqpConfig, queueName);
            } catch (Exception e) {
                System.out.println("The restart of the connection failed, exiting beacuse of " + e + " (" + e.getMessage() + ")");
                throw new RuntimeException(e);
            }
        }
    }
}
