package no.statnett.ecp.qm;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;

public class QMIHistory {
  public static final int MINUTE_MS = 60 * 1000;// 60 seconds
  public static final long SAFETY_MARGIN_MS = 30 * 1000; // will ensure that a QMI is processed even if it's a little bit off
  private List<QMI> qmiList = new ArrayList<>();
  private boolean ok = true; // OK
  private boolean stop = false;
  private boolean slow = false;
  private boolean delayMonitoring = false;
  private String queueName;

  public QMIHistory(String queueName, boolean delayMonitoring) {
    this.queueName = queueName;
    this.delayMonitoring = delayMonitoring;
  }

  public void addQMI(QMI newQMI) {
    qmiList.add(0, newQMI);
  }

  public void removeOlderThan(LocalDateTime tooOldLDT) {
    qmiList.removeIf(q -> q.getTimestamp() < tooOldLDT.toInstant(ZoneOffset.UTC).toEpochMilli());
  }

  // If the list contains QMI with dequeueCount higher than the one we're adding, remove the element because it must be from before a restart of the broker
  public void removeWithHigherDequeuCount(QMI qmi) {
    qmiList.removeIf(q -> q.getDequeueCount() > qmi.getDequeueCount());
  }

  public boolean hasStopped(long nMinutes) {
    if (qmiList.size() < 2)
      return false; // not possible to compare with only one QMI
    // The first qmi is the newest
    int newestDequeueCount = qmiList.get(0).getDequeueCount();
    long newestTimestamp = qmiList.get(0).getTimestamp();
    for (int i = 1; i < qmiList.size(); i++) {
      // We search for a dequeueCount lower than the newest before we reach the time limit specified by i (minutes)
      QMI qmi = qmiList.get(i);
      if (qmi.getTimestamp() < newestTimestamp - (nMinutes * MINUTE_MS + SAFETY_MARGIN_MS)) {
        return true; // We should have found a QMI with a lower dequeueCount than the newest, but we didn't
      }
      if (qmi.getDequeueCount() < newestDequeueCount) {
        return false; // We found a QMI with a lower dequeueCount than the newest
      }
    }
    return false; // We came here because we didn't reach the time limit - by default we return false (to avoid raise alarm before nMinutes has passed)
  }

  public boolean isSlow(long nMinutes) {
    if (qmiList.size() < 2)
      return false;
    long newestTimestamp = qmiList.get(0).getTimestamp();
    int lowestSize = qmiList.get(0).getSize(); // default to the newest size, unless we find a lower one
    // Two things happen in the loop:
    // 1) We find the QMI with the lowest size
    // 2) Find the QMI that is from nMinutes ago (the oldest QMI we want to compare with)
    QMI qmiNMinutesAgo = null;
    for (QMI qmi : qmiList) {
      long maxTimestamp = newestTimestamp - (nMinutes * MINUTE_MS);
      long minTimestamp = maxTimestamp - SAFETY_MARGIN_MS;
      if (qmi.getTimestamp() <= maxTimestamp) {
        if (qmi.getTimestamp() >= minTimestamp) {
          qmiNMinutesAgo = qmi; // We found the QMI that is nMinutes old, we will ignore older QMIs
        }
        break;
      }
      if (qmi.getSize() < lowestSize) {
        lowestSize = qmi.getSize();
      }
    }
    if (qmiNMinutesAgo == null)
      return false;
    if (qmiNMinutesAgo.getSize() <= lowestSize) {
      return true;
    }
    return false;
  }

  public boolean isOk() {
    return ok;
  }

  public void setOk(boolean ok) {
    this.ok = ok;
  }

  public boolean isStop() {
    return stop;
  }

  public void setStop(boolean stop) {
    this.stop = stop;
  }

  public boolean isSlow() {
    return slow;
  }

  public void setSlow(boolean slow) {
    this.slow = slow;
  }

  public boolean isDelayMonitoring() {
    return delayMonitoring;
  }

  public void setDelayMonitoring(boolean delayMonitoring) {
    this.delayMonitoring = delayMonitoring;
  }

  public String getQueueName() {
    return queueName;
  }

  public void setQueueName(String queueName) {
    this.queueName = queueName;
  }

  public String toString() {
    StringBuilder sb = new StringBuilder();
    String delayFlag = isDelayMonitoring() ? "D" : "-";
    sb.append(String.format("%45s", getQueueName()) + "/" + delayFlag + ":");
    List<String> qmiListStr = new ArrayList<>();
    for (int i = 0; i < qmiList.size(); i++) {
      qmiListStr.add(0, qmiList.get(i).toString(i));
    }
    sb.append(String.join(",", qmiListStr));
    return sb.toString();
  }

  public List<QMI> getQmiList() {
    return qmiList;
  }
}
