package no.statnett.ecp.qp;

import jakarta.jms.*;
import jakarta.jms.Queue;
import no.statnett.ecp.utils.*;
import org.apache.qpid.jms.message.JmsMessage;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;

public class Protect {
    public static int consume(Session session, Queue queue, File directory, Set<String> senders, int maxAgeSeconds) throws JMSException {
        int counter = 0;
        for (String sender : senders) {
            String selector = Broker.buildSelector(null, "" + maxAgeSeconds, sender, null, null);
            MessageConsumer consumer = session.createConsumer(queue, selector);
            System.out.println(LogOut.i() + "Trying to consume messages from queue " + queue.getQueueName() + " with selector: " + selector);
            while (true) {
                Message message = consumer.receive(100);
                if (message == null)
                    break;
                processConsume(message, directory);
                counter++;
            }
        }
        return counter;
    }

    private static void processConsume(Message msg, File directory) throws JMSException {
        String messageType = EcpMsg.findMessageType(msg);
        long ageSeconds = EcpMsg.calculateAgeSeconds(msg);

        // Creates a unique id to be used in the filename - if not found in the message, a random UUID is used
        String messageID = EcpMsg.getMessageID(msg);
        String uniqeId = messageID == null ? UUID.randomUUID().toString() : messageID;

        // Finds a senderCode, from ECP or EDX, if any - this header is used as a part of a natural key for the filename, but not essential
        String sender = EcpMsg.getSender(msg);

        // Finds a receiverCode, from ECP or EDX, if any - this header is used as a part of a natural key for the filename, but not essential
        String receiver = EcpMsg.getReceiver(msg);

        // internalType is only used here to print out in the log
        String internalType = EcpMsg.getInternalType(msg);

        String naturalKeys = sender + "_" + receiver + "_" + messageType;


        String persisted = "NO";
        if (directory != null) {
            if (internalType.equals("STANDARD_MESSAGE") || internalType.equals("PLAIN")) {
                String headers = readHeaders(msg);
                persisted = writeToFile(msg, naturalKeys, uniqeId, headers, directory);
            }
        }

        long hours = ageSeconds / 3600;
        long mins = (ageSeconds % 3600) == 0 ? 0 : (ageSeconds % 3600) / 60;
        String hourStr = hours + "h " + String.format("%02d", mins) + "m";
        System.out.println(LogOut.d() + "      " + "Consume" + " : MsgType=" + messageType + ", Age=" + ageSeconds + " sec (" + hourStr + "), Sender=" + sender + ", InternalType=" + internalType + ", Persisted: " + persisted);
    }


    public static String writeToFile(Message msg, String naturalKeys, String uniqeId, String headers, File directory) {
        try {
            // First write headers from message to file
            Files.writeString(Paths.get(directory.getAbsolutePath() + "/QP-" + naturalKeys + "_" + uniqeId + ".headers.txt"), headers);
            if (msg instanceof TextMessage) {
                TextMessage textMessage = (TextMessage) msg;
                Path pathMetadata = Paths.get(directory.getAbsolutePath() + "/QP-" + naturalKeys + "_" + uniqeId + ".metadata.xml");
                Files.writeString(pathMetadata, textMessage.getText());
                return directory.getAbsolutePath();
            } else if (msg instanceof BytesMessage) {
                BytesMessage bytesMessage = (BytesMessage) msg;
                byte[] bytes = new byte[(int) bytesMessage.getBodyLength()];
                bytesMessage.readBytes(bytes);
                Path pathMetadata = Paths.get(directory.getAbsolutePath() + "/QP-" + naturalKeys + "_" + uniqeId + ".metadata.xml");
                Files.write(pathMetadata, bytes);
                return directory.getAbsolutePath();
            } else if (msg instanceof StreamMessage) {
                StreamMessage streamMessage = (StreamMessage) msg;
                try {
                    while (true) {
                        Object obj = streamMessage.readObject(); // Method may throw MessageEOFException
                        if (obj instanceof String) {
                            Path pathMetadata = Paths.get(directory.getAbsolutePath() + "/QP-" + naturalKeys + "_" + uniqeId + ".metadata.xml");
                            Files.writeString(pathMetadata, obj.toString());
                        } else if (obj instanceof byte[]) {
                            Path pathPayload = Paths.get(directory.getAbsolutePath() + "/QP-" + naturalKeys + "_" + uniqeId + ".payload.bin");
                            Files.write(pathPayload, (byte[]) obj);
                        }
                    }
                } catch (MessageEOFException e) {
                    // This is expected - the only way out of the loop
                }
                return directory.getAbsolutePath();
            } else if (msg instanceof JmsMessage) {
                JmsMessage jmsMessage = (JmsMessage) msg;
                return "FAILED because we don't know how to handle JmsMessage yet, its type is " + jmsMessage.getJMSType();
            } else {
                return "FAILED because message is of type " + msg.getClass().getName() + " which is not supported";
            }
        } catch (Exception e) {
            System.out.println(LogOut.e() + "Cannot write to file, exception: " + e.getMessage() + ", Stacktrace: " + Div.onelinerStacktrace(e.getStackTrace(), "no.statnett.ecp"));
            return "FAILED because of exception " + e.getMessage();
        }
    }

    /**
     * The following headers are set in a message on the ecp.endpoint.download queue, according to Hawtio - but we only read out Boolean, Byte, Int, Long and String properties - the others are JMS-internal.
     * Ohter queues will have different headers, but some will be the same. It does not matter a lot, as long as we put them back to the same queue later using BurstRestore.
     * Header - JMSCorrelationID	6809751e-14c4-41ea-b12f-83fcd57208ae
     * Header - JMSDeliveryMode	PERSISTENT
     * Header - JMSDestination	queue://ecp.endpoint.download
     * Header - JMSExpiration	1708013697313
     * Header - JMSMessageID	ID:0c7ca085-ebf3-4448-ab36-a6b5936e2bca:1:1479:1-1
     * Header - JMSPriority	4
     * Header - JMSRedelivered	false
     * Header - JMSTimestamp	2024-02-01T16:14:57Z
     * Header - JMSXGroupSeq	0
     * Header - BodyLength	5674
     * Header - BrokerPath	null
     * Header - OriginalDestination
     * Boolean Property - JMS_AMQP_HEADER	true
     * Boolean Property - JMS_AMQP_NATIVE	true
     * Boolean Property - JMS_AMQP_PROPERTIES	true
     * Byte Property - JMS_AMQP_MA_x-opt-jms-dest	0
     * Byte Property - JMS_AMQP_MA_x-opt-jms-msg-type	4
     * Int Property - messageMversion	2
     * Long Property - CamelMessageTimestamp	1706804097566
     * Long Property - generated	1706804097311
     * String Property - baMessageID	BATCH-1-FILE-71
     * String Property - ecpMessageDirection	DOWN
     * String Property - internalType	STANDARD_MESSAGE
     * String Property - messageID	c348e82b-16fe-498c-a1d7-a3306ae02257
     * String Property - messageType	BATCH-1-FILE-71-1KB
     * String Property - receivedFromComponent	50V000000000112U
     * String Property - receiverCode	50V-SN-TEST--ATK
     * String Property - senderApplication
     * String Property - senderCode	50V000000000115O
     */
    private static String readHeaders(Message msg) throws JMSException {
        StringBuilder headerSB = new StringBuilder();
        Enumeration<String> propertyNames = msg.getPropertyNames();
        while (propertyNames.hasMoreElements()) {
            String propertyName = propertyNames.nextElement();
            Object objectProperty = msg.getObjectProperty(propertyName);
            if (objectProperty != null)
                headerSB.append(objectProperty.getClass().getName()).append(" ").append(propertyName).append("=").append(objectProperty).append("\n");
        }
        return headerSB.toString();
    }
}
