package no.statnett.ecp.utils;

import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;

public class ArtemisConsoleAPI {

  // Broker-list QueryString /console/jolokia/search/org.apache.activemq.artemis:broker=*"
  public static String retrieveBrokerNameURLEncoded(String protocol, String host, String port, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    URL url = new URL(protocol + "://" + host + ":" + port + "/console/jolokia/search/org.apache.activemq.artemis:broker=*");
    String json = getJSON(url, username, password);
    // Example of json: {"request":{"mbean":"org.apache.activemq.artemis:broker=*","type":"search"},"value":["org.apache.activemq.artemis:broker=\"ECCo SP Artemis\""],"timestamp":1720632774,"status":200}
    // Extract broker-name from json
    int valuesPos = json.indexOf("value\":") + 9;
    int brokerNameStartPos = json.indexOf("broker=", valuesPos) + 7;
    int brokerNameEndPos = json.indexOf("]", brokerNameStartPos);
    String brokerNameWithQuotes = json.substring(brokerNameStartPos, brokerNameEndPos);
    String brokerName = brokerNameWithQuotes.replaceAll("\"", "").replaceAll("\\\\", "");
    return brokerName.replaceAll(" ", "%20");
  }

  // Broker-info QueryString  /console/jolokia/read/org.apache.activemq.artemis:broker=%22ECCo%20SP%20Artemis%22
  public static String retrieveBrokerInfoJSON(String protocol, String host, String port, String brokerNameEncoded, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    URL url = new URL(protocol + "://" + host + ":" + port + "/console/jolokia/read/org.apache.activemq.artemis:broker=%22" + brokerNameEncoded + "%22");
    return getJSON(url, username, password);
  }

  // Queue-info QueryString:  /console/jolokia/exec/org.apache.activemq.artemis:broker=%22ECCo%20SP%20Artemis%22/listQueues/%7B%22field%22:%22name%22%2C%22operation%22:%22CONTAINS%22%2C%22value%22:%22ecp%22%7D/1/1000
  public static String retrieveQueueInfoJSON(String protocol, String host, String port, String brokerNameEncoded, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    URL url = new URL(protocol + "://" + host + ":" + port + "/console/jolokia/exec/org.apache.activemq.artemis:broker=%22" + brokerNameEncoded + "%22/listQueues/%7B%22field%22:%22name%22%2C%22operation%22:%22NOT%5FCONTAINS%22%2C%22value%22:%22abcdefghijkl%22%7D/1/1000");
    return getJSON(url, username, password);
  }

/*
    public static String retreieveQueueDetailJSON(String protocol, String host, String port, String brokerNameEncoded, String username, String password, String queueName) throws NoSuchAlgorithmException, KeyManagementException, IOException {
        URL url = new URL(protocol + "://" + host +":"+ port+ "/console/jolokia/read/org.apache.activemq.artemis:broker==%22\" + brokerNameEncoded + \"%22%2Ccomponent%3Daddresses%2Caddress%3D%22"+queueName+"%22");
        return getJSON(url, username, password);
    }
*/


  // RemoveMessages QueryString: /console/jolokia/exec/org.apache.activemq.artemis:broker=%22ECCo%20SP%20Artemis%22%2Ccomponent=addresses%2Caddress=%22QUEU%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22QUEUE%22/removeAllMessages
  public static ACAPIResult removeAllMessages(String protocol, String host, String port, String brokerNameEncoded, String username, String password, String queueName) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    String rawResponse = getJSON(new URL(protocol + "://" + host + ":" + port + getQueryStringQueue(brokerNameEncoded, queueName) + "/removeAllMessages"), username, password);
    String status = findAttributeValue(rawResponse, "status");
    String value = findAttributeValue(rawResponse, "value");
    return new ACAPIResult(status, value, rawResponse, null);
  }

  public static ACAPIResult removeMessages(String protocol, String host, String port, String brokerNameEncoded, String username, String password, String queueName, String filter) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    String rawResponse = getJSON(new URL(protocol + "://" + host + ":" + port + getQueryStringQueue(brokerNameEncoded, queueName) + "/removeMessages%28java.lang.String%29/" + filter), username, password);
    String status = findAttributeValue(rawResponse, "status");
    String value = findAttributeValue(rawResponse, "value");
    return new ACAPIResult(status, value, rawResponse, null);
  }

  // ListDeliveringMessages:     /console/jolokia/exec/org.apache.activemq.artemis:broker=%22ECCo%20SP%20Artemis%22%2Ccomponent=addresses%2Caddress=%22QUEUE%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22QUEUE%22/listDeliveringMessages
  public static ACAPIResult listDeliveringMessages(String protocol, String host, String port, String brokerNameEncoded, String username, String password, String queueName) throws IOException, NoSuchAlgorithmException, KeyManagementException {
    String rawResponse = getJSON(new URL(protocol + "://" + host + ":" + port + getQueryStringQueue(brokerNameEncoded, queueName) + "/listDeliveringMessages"), username, password);
    JSONObject json = new JSONObject(rawResponse);
    int status = json.getInt("status");
    if (status == 200) {
      String valueArray = findValueArray(rawResponse);
      return new ACAPIResult("" + status, valueArray, rawResponse, json);
    } else {
      return new ACAPIResult(Integer.toString(status), null, rawResponse, null);
    }
  }

  // ListMesageInfo:             /console/jolokia/exec/org.apache.activemq.artemis:broker=%22ECCo%20SP%20Artemis%22%2Ccomponent=addresses%2Caddress=%22QUEUE%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22QUEUE%22/listMessages"
  public static ACAPIResult listMessages(String protocol, String host, String port, String brokerNameEncoded, String username, String password, String queueName, String filter) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    String rawResponse = getJSON(new URL(protocol + "://" + host + ":" + port + getQueryStringQueue(brokerNameEncoded, queueName) + "/listMessages/" + filter), username, password);
    JSONObject json = new JSONObject(rawResponse);
    int status = json.getInt("status");
    if (status == 200) {
      String valueArray = findValueArray(rawResponse);
      return new ACAPIResult("" + status, valueArray, rawResponse, json);
    } else {
      return new ACAPIResult(Integer.toString(status), null, rawResponse, null);
    }
  }

  // ListConsumers: QueueControl exposes listConsumersAsJSON on many Artemis versions. Jolokia returns a quoted JSON string in "value".
  // Example path: /console/jolokia/exec/org.apache.activemq.artemis:broker=%22<broker>%22%2Ccomponent=addresses%2Caddress=%22<QUEUE>%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22<QUEUE>%22/listConsumersAsJSON
  public static ACAPIResult listConsumers(String protocol, String host, String port, String brokerNameEncoded, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    String rawResponse = getJSON(new URL(protocol + "://" + host + ":" + port + getQueryStringQueue(brokerNameEncoded, null) + "/listAllConsumersAsJSON"), username, password);

    JSONObject json = new JSONObject(rawResponse);
    int status = json.getInt("status");
    if (status == 200) {
      // Because this method is "listConsumersAsJSON", the value-property is a quoted JSON string. This is the *only* place (so far) where we have to unquote the value.
      Object valueJson = new JSONTokener((String) json.get("value")).nextValue();
      return new ACAPIResult("" + status, null, rawResponse, valueJson);
    } else {
      return new ACAPIResult(Integer.toString(status), null, rawResponse, null);
    }
  }

  // ListConsumers: QueueControl exposes listConsumersAsJSON on many Artemis versions. Jolokia returns a quoted JSON string in "value".
  // Example path: /console/jolokia/exec/org.apache.activemq.artemis:broker=%22<broker>%22%2Ccomponent=addresses%2Caddress=%22<QUEUE>%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22<QUEUE>%22/listConsumersAsJSON
  public static ACAPIResult listConnections(String protocol, String host, String port, String brokerNameEncoded, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    String rawResponse = getJSON(new URL(protocol + "://" + host + ":" + port + getQueryStringQueue(brokerNameEncoded, null) + "/listConnectionsAsJSON"), username, password);

    JSONObject json = new JSONObject(rawResponse);
    int status = json.getInt("status");
    if (status == 200) {
      // Because this method is "listConsumersAsJSON", the value-property is a quoted JSON string. This is the *only* place (so far) where we have to unquote the value.
      Object valueJson = new JSONTokener((String) json.get("value")).nextValue();
      return new ACAPIResult("" + status, null, rawResponse, valueJson);
    } else {
      return new ACAPIResult(Integer.toString(status), null, rawResponse, null);
    }
  }


  private static String getQueryStringQueue(String brokerNameEncoded, String queueName) {
    String qs = "/console/jolokia/exec/org.apache.activemq.artemis:broker=%22" + brokerNameEncoded + "%22";
    if (queueName != null)
      qs += "%2Ccomponent=addresses%2Caddress=%22" + queueName + "%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22" + queueName + "%22";
    return qs;
  }

  public static boolean destroyQueue(String protocol, String host, String port, String brokerNameEncoded, String username, String password, String queueName) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    // Keep in reserve: %2Ccomponent=addresses%2Caddress=%22" + queueName + "%22%2Csubcomponent=queues%2Crouting-type=%22anycast%22%2Cqueue=%22" + queueName + "%22
    URL url = new URL(protocol + "://" + host + ":" + port + "/console/jolokia/exec/org.apache.activemq.artemis:broker=%22" + brokerNameEncoded + "%22/destroyQueue(java.lang.String)/" + queueName);
    String json = getJSON(url, username, password);
    String status = findAttributeValue(json, "status");
    if (status.equals("200"))
      return true;
    else
      return false;
  }

  private static String getJSON(URL url, String username, String password) throws NoSuchAlgorithmException, KeyManagementException, IOException {
    Map<String, String> requestHeaders = Map.of("Accept", "application/json");
    String str = HTTP.retrieveResponseBody("GET", url, username, password, requestHeaders, null, 0);
    if (str == null) return "";
    if (str.contains("error_type")) {
      // We tried to check the connection.getReponseMessage() != 200, but it did not work as expected, so we resort to checking the output
      String errorMessage = "Error occurred: HTTP-response contains errors: " + str + " for URL: " + url + " using user/pass " + username + "/*****";
      System.out.println("\n" + errorMessage);
      throw new IOException(errorMessage);
    }
    return str;

  }

  private static String findAttributeValue(String json, String name) {
    int statusStartPos = json.indexOf(name + "\":") + name.length() + 2;
    if (statusStartPos <= (-1 + name.length() + 2))
      return null;
    int statusEndPos = json.indexOf(",", statusStartPos) == -1 ? json.indexOf("}", statusStartPos) : json.indexOf(",", statusStartPos);
    return json.substring(statusStartPos, statusEndPos);
  }

  private static String findValueArray(String json) {
    // The array following the "value"-property is not always found with the same offset pos. Besides, "value"-property are so common that we need to make search through the json-string to
    // be sure we have checked the whole json for value-property with an array.
    String valueProp = "\"value\":";
    int searchFromPos = 0;
    while (true) {
      int valueStartPos = json.indexOf(valueProp, searchFromPos);
      if (valueStartPos == -1)
        return null;
      int arrStartPos = json.indexOf("[", valueStartPos);
      if (arrStartPos == -1)
        return null;
      if (arrStartPos - (valueStartPos + valueProp.length()) < 5) {
        // We've found a value-property with an array
        return json.substring(arrStartPos + 1, json.indexOf("]", arrStartPos));
      }
      searchFromPos = valueStartPos + valueProp.length();
    }
  }


  public static class ACAPIResult {
    private String status;
    private String value;
    private String rawResponse;
    private Object jsonObject;

    public ACAPIResult(String status, String value, String rawResponse, Object jsonObject) {
      this.status = status;
      this.value = value;
      this.rawResponse = rawResponse;
      this.jsonObject = jsonObject;
    }

    // This object will either be a JSONObject or a JSONArray
    public Object getJsonObjectUntyped() {
      return jsonObject;
    }

    public JSONArray getJsonArray() {
      return (JSONArray) jsonObject;
    }

    public JSONObject getJsonObject() {
      return (JSONObject) jsonObject;
    }

    public boolean isJsonObjectAnArray() {
      return jsonObject instanceof JSONArray;
    }

    public String getStatus() {
      return status;
    }

    public String getValue() {
      return value;
    }

    public String getRawResponse() {
      return rawResponse;
    }


    public String toString() {
      return "Status: " + status + ", Value: " + value + ", raw-response: " + rawResponse;
    }
  }

}
